#include <signal.h>
#include <pp/base.h>
#include <pp/intl.h>
#include <pp/um.h>
#include <liberic_notify.h>
#include <liberic_misc.h>
#include <liberic_net.h>
#include <liberic_cron.h>
#include "eric_util.h"
#include "reconf.h"

static int reconf_ping_ch(pp_cfg_chg_ctx_t * ctx);
static int reconf_snmpd_ch(pp_cfg_chg_ctx_t * ctx);
#ifdef PP_FEAT_DYNDNS
static void contact_dyndns_server(void);
#endif /* PP_FEAT_DYNDNS */
#ifdef PP_FEAT_AUTH_RADIUS
static int reconf_radius_ch(pp_cfg_chg_ctx_t * ctx);
#endif /* PP_FEAT_AUTH_RADIUS */
static int reconf_mouse_hotkey_ch(pp_cfg_chg_ctx_t * ctx);

/********************* INIT / CLEANUP ***********************/

int
reconf_init(void)
{
    pp_cfg_add_change_listener(reconf_ping_ch, "keep_alive");

    pp_cfg_add_post_tx_change_listener(reconf_snmpd_ch, "log.snmp.enabled");
    pp_cfg_add_post_tx_change_listener(reconf_snmpd_ch, "snmp");

#ifdef PP_FEAT_DYNDNS
    pp_cfg_add_change_listener(reconf_dyndns_ch, "dyndns");
#endif /* PP_FEAT_DYNDNS */
    
#if defined(PP_FEAT_AUTH_RADIUS)
    pp_cfg_add_change_listener(reconf_radius_ch, "auth_mode");
    pp_cfg_add_change_listener(reconf_radius_ch, "radius");
#endif

    pp_cfg_add_change_listener(reconf_mouse_hotkey_ch, "user[?].rc.mousesync.key");

    return PP_SUC;
}

void
reconf_cleanup(void)
{
    pp_cfg_rem_change_listener(reconf_ping_ch, "keep_alive");

    pp_cfg_rem_change_listener(reconf_snmpd_ch, "log.snmp.enabled");
    pp_cfg_rem_change_listener(reconf_snmpd_ch, "snmp");

#ifdef PP_FEAT_DYNDNS
    pp_cfg_rem_change_listener(reconf_dyndns_ch, "dyndns");
#endif /* PP_FEAT_DYNDNS */

#if defined(PP_FEAT_AUTH_RADIUS)
    pp_cfg_rem_change_listener(reconf_radius_ch, "auth_mode");
    pp_cfg_rem_change_listener(reconf_radius_ch, "radius");
#endif

    pp_cfg_rem_change_listener(reconf_mouse_hotkey_ch, "user[?].rc.mousesync.key");
}

/******************* THE CHANGE HANDLERS ***************************/

static int
reconf_ping_ch(pp_cfg_chg_ctx_t * ctx UNUSED)
{
    char PID_file_content[10];
    char cmdline_path[128];
    char cmdline[128];
    FILE *pid_file;
    FILE *cmdline_file;
    int pid = -1;
    int total = 0;
         
    if ((pid_file = fopen("/var/run/keepaliveping.pid", "r")) != NULL) {
    	fread(PID_file_content, 1, sizeof(PID_file_content), pid_file);
	fclose(pid_file);
    	pid = pp_strtol_10(PID_file_content, -1, NULL);
    } else {
	pp_log_err("%s(): Unable to open /var/run/keepaliveping.pid", ___F);
    }
    
    if (pid >= 0) {
	snprintf(cmdline_path, sizeof(cmdline_path), "/proc/%u/cmdline", pid);
	if ((cmdline_file = fopen(cmdline_path, "r")) != NULL) {
	    total = fread(cmdline, 1, sizeof(cmdline), cmdline_file);
	    cmdline[max(total-1, 0)] = '\0';
	    fclose(cmdline_file);
    	
	    /* kill old keepaliveping */
	    if (strstr(cmdline, "keepaliveping.sh") != NULL) {
		kill(pid, SIGTERM);
	    }
	}
    }
    
    return PP_SUC;
}

static int
reconf_snmpd_ch(pp_cfg_chg_ctx_t * ctx UNUSED)
{
    pp_system("/etc/rc.snmpd > /dev/null 2>&1");
    return PP_SUC;
}

#ifdef PP_FEAT_DYNDNS

int
dyndns_save_ip(const char *ip)
{
    if (PP_FAILED(pp_cfg_set(ip, "dyndns.last_ip"))) {
    	pp_cfg_save(DO_FLUSH);
    	return PP_SUC;
    } else {
    	pp_log("%s(): Could not save the IP address\n", ___F);
    	return PP_ERR;
    }
}

static int
get_current_ip(char *ip)
{
    int ret = PP_ERR, len, whole_len = 0;
    const char *check_server = "checkip.dyndns.org";
    const char * check_request_template = 
	"GET /"
	     " HTTP/1.0\r\n"
	     "Host: %s\r\n"
	     "User-Agent: erla ddns/linux/1.0\r\n\r\n";
    char check_request[1024];
    char check_response[1024];
    BIO * checkip_bio;
    char *ip_string;

    memset(check_response, 0, sizeof(check_response));

    /* build the request */
    snprintf(check_request, sizeof(check_request), check_request_template, check_server);
    
    /* open a connection to DynDNS server */
    if ((checkip_bio = eric_net_open_connection(check_server, "80", PP_NET_IO_TIMEOUT)) == NULL) {
    	pp_log("Could not connect to %s\n", check_server);
    	return PP_ERR;
    }
    
    /* send the request & read the reply */
    if (eric_net_write_exact(checkip_bio, check_request, strlen(check_request), 0) != -1) {
	while ((len = eric_net_read(checkip_bio, check_response + whole_len, sizeof(check_response) - whole_len, 0)) > 0) {
	    whole_len += len;
	}
    }

    /* close the connection to DynDNS server */
    eric_net_close_connection(checkip_bio);
    	
    /* parse the response */
#define CURRENT_IP_MAGIC "Current IP Address:"
    if ((ip_string = strstr(check_response, CURRENT_IP_MAGIC)) != NULL) {
    	u_char curr_ip[4];
    	int parsed;

	ip_string += sizeof(CURRENT_IP_MAGIC) - 1;
    	parsed = sscanf(ip_string, "%hhu.%hhu.%hhu.%hhu", &curr_ip[0], &curr_ip[1], &curr_ip[2], &curr_ip[3]);
    	if (parsed == 4) {
    	    snprintf(ip, INET_ADDRSTRLEN, "%hhu.%hhu.%hhu.%hhu", curr_ip[0], curr_ip[1], curr_ip[2], curr_ip[3]);
    	    ret = PP_SUC;
    	} else {
    	    pp_log("Error parsing current IP address '%s'.\n", ip_string);
    	}
    } else {
    	pp_log("Could not find IP address in checkip response.\n");
    }
#undef CURRENT_IP_MAGIC

    return ret;
}

static int
dyndns_update_needed(int retries)
{
    int ret = 0;
    char *last_ip = NULL;
    char current_ip[INET_ADDRSTRLEN];
    int retry;
    
    pp_cfg_get_nodflt(&last_ip, "dyndns.last_ip");
    
    /* query old and current IP */
    for (retry = 0; retry <= retries; ++retry) {
    	if (PP_SUCCED(get_current_ip(current_ip))) {
    	    if (!last_ip || strcmp(last_ip, current_ip)) {
    	    	dyndns_save_ip(current_ip);
    	    	ret = 1;
    	    }
    	    break;
    	} else {
    	    /* in case of an error, always update DynDNS */
    	    pp_log("Could not query external IP, ");
/* FIXME: FIXME - is it always safe to sleep here for such a long time? */
    	    if (retry < retries) sleep(10);
    	}
    }
    
    free(last_ip);
    return ret;
}

static void
contact_dyndns_server_retries(int retries)
{
    int len;
    const char * server = "members.dyndns.org";
    const char * request_template =
	"GET /nic/update?"
	     "system=%s&"
	     "hostname=%s"
	     " HTTP/1.0\r\n"
	     "Host: %s\r\n"
	     "Authorization: Basic %s\r\n"
	     "User-Agent: erla ddns/linux/1.0\r\n\r\n";
    char request[1024];
    char reply[1024];
    char * host = NULL;
    char * username = NULL;
    char * password = NULL;
    char * sys = NULL;
    char uname_pwd[128];
    char uname_pwd_enc[128];
    BIO * dyndns_bio;

    if (!dyndns_update_needed(retries)
	|| PP_FAILED(pp_cfg_get_nodflt(&host, "dyndns.hostname"))
	|| PP_FAILED(pp_cfg_get_nodflt(&username, "dyndns.username"))
	|| PP_FAILED(pp_cfg_get_nodflt(&password, "dyndns.password"))
	) {
        goto bail;
    }

    pp_cfg_get_nodflt(&sys, "dyndns.system");

    pp_log("%s(): Update to %s...\n", ___F, server);

    snprintf(uname_pwd, sizeof(uname_pwd), "%s:%s", username, password);
    
    eric_webs_encode64(uname_pwd_enc, uname_pwd, 128);
    
    /* build the request */
    snprintf(request, sizeof(request), request_template, sys, host, server, uname_pwd_enc);

    /* open a connection to DynDNS server */
    if ((dyndns_bio = eric_net_open_connection(server, "80", PP_NET_IO_TIMEOUT)) == NULL) {
	pp_log("%s(): Could not connect to %s\n", ___F, server);
	/* make sure the next time an update is done */
	dyndns_save_ip("");
	goto bail;
    }

    /* send the request & read the reply */
    if (eric_net_write_exact(dyndns_bio, request, strlen(request), 0) != -1) {
        while ((len = eric_net_read(dyndns_bio, reply, sizeof(reply), 0)) > 0);
    } else {
	pp_log("%s(): Could write the request to %s\n", ___F, server);
	/* make sure the next time an update is done */
	dyndns_save_ip("");
	goto bail;
    }

    /* close the connection to DynDNS server */
    eric_net_close_connection(dyndns_bio);

 bail:
    free(sys);   
    free(host);
    free(username);
    free(password);   
}

static void
contact_dyndns_server(void)
{
    contact_dyndns_server_retries(10);
}

int
reconf_dyndns_ch(pp_cfg_chg_ctx_t * ctx)
{
    int dyndns_enabled_opt = 0;
    char * check_time_opt = NULL;
    u_int check_interval_opt = 0;
    char * utc_offset_opt = NULL;
    struct tm tm;
    time_t check_time, check_interval;
    int i;
    int ret = PP_ERR;
    int offset = 0;
    pp_strstream_t strbuf;

    pp_strstream_init(&strbuf);
    memset(&tm, 0, sizeof(tm));
    eric_cron_delete_action(-1, contact_dyndns_server);

    pp_cfg_is_enabled(&dyndns_enabled_opt, "dyndns.enabled");
    if (dyndns_enabled_opt) {
	if (PP_FAILED(pp_cfg_get_nodflt(&check_time_opt, "dyndns.check_time"))
	    || PP_FAILED(pp_cfg_get_uint_nodflt(&check_interval_opt, "dyndns.check_interval"))) {
	    goto bail;
	}
	
	if (strptime(check_time_opt, "%R", &tm) == NULL) {
	    pp_strappendf(&strbuf, _("Cannot parse check time (%s)\n"), check_time_opt);
	    goto bail;
	}

	pp_cfg_get(&utc_offset_opt, "time.utc_offset");
	if (strchr(utc_offset_opt, '/')) {
	    offset = 0;
	} else if (strchr(utc_offset_opt, '+')) {
	    if (sscanf(utc_offset_opt, "+ %u h", &offset) != 1 || offset > 12) {
		offset = 0;
	    }
	} else if (strchr(utc_offset_opt, '-')) {
	    if (sscanf(utc_offset_opt, "- %u h", &offset) != 1 || offset > 11) {
		offset = 0;
	    } else {
		offset = -offset;
	    }
	}
	
	check_time = (3600 * 24 + 3600 * tm.tm_hour + 60 * tm.tm_min - offset * 3600) % (3600 * 24);
	if (check_interval_opt == 0) {
	    pp_strappendf(&strbuf, _("Check interval must be greater than zero.\n"));
	    goto bail;
	}
	check_interval = check_interval_opt * 60;

	/* update if necessary */
	contact_dyndns_server_retries(0);
      
	for (i = 0; i < 24*3600; i += check_interval) {
	    eric_cron_insert_action(check_time + i, contact_dyndns_server);
	}
    }
    ret = PP_SUC;
 bail:
    if (pp_strstream_pos(&strbuf)) {
        *(ctx->retstr) = pp_strstream_buf(&strbuf);
    } else {
        pp_strstream_free(&strbuf);
    }
    free(check_time_opt);
    free(utc_offset_opt);
    return ret;
}

#endif /* PP_FEAT_DYNDNS */

#ifdef PP_FEAT_AUTH_RADIUS
static int
reconf_radius_ch(pp_cfg_chg_ctx_t * ctx)
{
    char * auth_mode;
    int ret = PP_SUC;
    pp_strstream_t strbuf;
    
    pp_cfg_get(&auth_mode, "auth_mode");
    if (!pp_strcmp_safe(auth_mode, "radius")) {
	int init_state;
	pp_um_radius_close_handles();
	init_state = pp_um_radius_init();
	if (init_state < 0) {
	    pp_strstream_init(&strbuf);
	    pp_strappendf(&strbuf, _("Radius initialization failed."));
	    *(ctx->retstr) = pp_strstream_buf(&strbuf);
	    ret = PP_ERR;
	} else if (init_state > 0) {
	    // init state returns the number of the server that could not be
	    // added by any reason
            pp_strstream_init(&strbuf);
	    pp_strappendf(&strbuf, _("Failed to add RADIUS server (code %d).<br>Please check the server name."), init_state);
            *(ctx->retstr) = pp_strstream_buf(&strbuf);
	    ret = PP_ERR;
	}
    }
    free(auth_mode);
    return ret;
}
#endif /* PP_FEAT_AUTH_RADIUS */

static int
reconf_mouse_hotkey_ch(pp_cfg_chg_ctx_t * ctx)
{
    int ret = PP_SUC;
    vector_t users, *key_comps;
    unsigned int vec_sz, users_sz = 0, u, v;
    char *key, *keycode, *label;
    int uid;
    char *error = NULL;
    pp_strstream_t strbuf;
    
    vec_sz = vector_size(ctx->key_comp_vec);
    pp_strstream_init(&strbuf);
    vector_new(&users, vec_sz, NULL);
    
    /* at first, get user with changed configuration */
    for (u = 0; u < vec_sz; ++u) {
        key_comps = (vector_t *)vector_get(ctx->key_comp_vec, u);
        if (PP_SUCCED(pp_cfg_is_user_key(&uid, (const vector_t *)key_comps))) {
            for (v = 0; uid>=0 && v < users_sz; ++v) {
                if (uid == vector_get_int(&users, v)) break;
	    }
            if (v == users_sz) {
                /* user not in users, add it */
                vector_add(&users, (void*)uid);
                ++users_sz;
            }
        }
    }

    /* apply changes to all users affected */
    for (u = 0; u < users_sz; ++u) {
        /* generate keycode */
        uid = vector_get_int(&users, u);
        
	if (PP_SUCCED(pp_cfg_get_nodflt(&key, "user[%u].rc.mousesync.key", uid))) {
	    keycode = eric_misc_get_key_appletparam(key, &error, KEYPARSER_WITHOUT_DELIS);    
	    if (keycode == NULL) {
		label = pp_cd_lookup_prop(pp_cfg_get_cd(), "user.rc.mousesync.key", "desc");
		pp_strappendf(&strbuf, _("Can't parse %s definition '%s'<br>Unknown key symbol '%s'!\n"),
			      _(label), key, error);    
		ret = PP_ERR;
	    } else {
		pp_cfg_set(keycode, "user[%u].rc.mousesync.code", uid);
	    }
        
	    /* check for single mouse mode */
	    if (*key == '\0' && pp_strstream_pos(&strbuf) == 0) {
		pp_strappendf(&strbuf, _("Operation completed successfully.<br>Single mouse mode not possible, because no Mouse hotkey defined."));
	    }

	    free(key);
	    free(keycode);
	    free(error);
	    error = NULL;
	}
    }
    
    vector_delete(&users);

    if (pp_strstream_pos(&strbuf)) {
        *(ctx->retstr) = pp_strstream_buf(&strbuf);
    } else {
        pp_strstream_free(&strbuf);
    }
    
    return ret;
}

