#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "tmpl_ipmi_common.h"


/* internal prototypes */
static int ipmi_build_filter_form_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);
static void print_filter_form(pp_strstream_t* tab, pp_ipmi_pef_filter_t* filter);
static void display_filter_form(pp_strstream_t* tab, webs_t wp);
static int post_validate_hook(webs_t wp, form_handler_t * fh);


FV_SPEC = {
{
    id:             FV_ID_PEF_FILTER_NUMBER,
    cfgkey:         "ipmi_filter_edit.number",
},
{
    id:             FV_ID_PEF_FILTER_STATUS,
    cfgkey:         "ipmi_filter_edit.status",
},
{
    id:             FV_ID_PEF_FILTER_ALERT,
    cfgkey:         "ipmi_filter_edit.action_alert",
},
{
    id:             FV_ID_PEF_FILTER_RESET,
    cfgkey:         "ipmi_filter_edit.action_reset",
},
{
    id:             FV_ID_PEF_FILTER_POWEROFF,
    cfgkey:         "ipmi_filter_edit.action_poweroff",
},
{
    id:             FV_ID_PEF_FILTER_POWERCYCLE,
    cfgkey:         "ipmi_filter_edit.action_powercycle",
},
{
    id:             FV_ID_PEF_FILTER_ALERTPOLICY,
    cfgkey:         "ipmi_filter_edit.alertpolicy",
},
{
    id:             FV_ID_PEF_FILTER_EVENTSEVERITY,
    cfgkey:         "ipmi_filter_edit.event_severity",
},
{   
    id:             FV_ID_PEF_FILTER_GENERATOR_ID1,
    cfgkey:         "ipmi_filter_edit.filter_generator1",
},
{
    id:             FV_ID_PEF_FILTER_GENERATOR_ID2,
    cfgkey:         "ipmi_filter_edit.filter_generator2",
},
{
    id:             FV_ID_PEF_FILTER_SENSORTYPE,
    cfgkey:         "ipmi_filter_edit.sensor_type",
},
{
    id:             FV_ID_PEF_FILTER_SENSORNUMBER,
    cfgkey:         "ipmi_filter_edit.sensor_number",
},
{
    id:             FV_ID_PEF_FILTER_EVENTTRIGGER,
    cfgkey:         "ipmi_filter_edit.event_trigger",
},
{
    id:             FV_ID_PEF_FILTER_DATA1_OFFSET1,
    cfgkey:         "ipmi_filter_edit.data1_offset1",
},
{
    id:             FV_ID_PEF_FILTER_DATA1_OFFSET2,
    cfgkey:         "ipmi_filter_edit.data1_offset2",
},
{
    id:             FV_ID_PEF_FILTER_DATA1_MASK,
    cfgkey:         "ipmi_filter_edit.data1_mask",
},
{
    id:             FV_ID_PEF_FILTER_DATA1_EXACT,
    cfgkey:         "ipmi_filter_edit.data1_exact",
},
{
    id:             FV_ID_PEF_FILTER_DATA1_DATA,
    cfgkey:         "ipmi_filter_edit.data1_data",
},
{
    id:             FV_ID_PEF_FILTER_DATA2_MASK,
    cfgkey:         "ipmi_filter_edit.data2_mask",
},
{
    id:             FV_ID_PEF_FILTER_DATA2_EXACT,
    cfgkey:         "ipmi_filter_edit.data2_exact",
},
{
    id:             FV_ID_PEF_FILTER_DATA2_DATA,
    cfgkey:         "ipmi_filter_edit.data2_data",
},
{
    id:             FV_ID_PEF_FILTER_DATA3_MASK,
    cfgkey:         "ipmi_filter_edit.data3_mask",
},
{
    id:             FV_ID_PEF_FILTER_DATA3_EXACT,
    cfgkey:         "ipmi_filter_edit.data3_exact",
},
{
    id:             FV_ID_PEF_FILTER_DATA3_DATA,
    cfgkey:         "ipmi_filter_edit.data3_data",
},
};


int ipmi_filter_edit_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("ipmiGetFilterEditForm", ipmi_build_filter_form_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_IPMI_FILTER_EDIT, ACL_OBJ_IPMI_STATUS);
    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int post_validate_hook(webs_t wp, form_handler_t * fh)
{
    pp_ipmi_pef_filter_t* filter;
    pp_ipmi_parameter_t in_parm;
    int save_form;

    filter = &(in_parm.data.pef_filter_set);
    memset(filter, 0, sizeof(pp_ipmi_pef_filter_t));

    save_form = 0;
    if (form_button_clicked(wp, "action_apply")) {
        save_form = 1;
    }

    websSetVar(wp, "filter", fh->fv[FV_ID_PEF_FILTER_NUMBER].val.s);

    if (save_form == 1)
    {
        filter->id = strtol(fh->fv[FV_ID_PEF_FILTER_NUMBER].val.s, NULL, 10);
        filter->record_type = 0x02;
        filter->configuration_type = 0x00;
        
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_STATUS].val.s, "enable") == 0) {
            filter->enable = 1;
        } else {
            filter->enable = 0;
        }
        
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_ALERT].val.s, "alert") == 0) {
            filter->action.alert = 1;
        } else {
            filter->action.alert = 0;
        }
        
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_RESET].val.s, "reset") == 0) {
            filter->action.reset = 1;
        } else {
            filter->action.reset = 0;
        }

        if (strcmp(fh->fv[FV_ID_PEF_FILTER_POWEROFF].val.s, "poweroff") == 0) {
            filter->action.poweroff = 1;
        } else {
            filter->action.poweroff = 0;
        }
        
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_POWERCYCLE].val.s, "powercycle") == 0) {
            filter->action.power_cycle = 1;
        } else {
            filter->action.power_cycle = 0;
        }
        
        filter->alert_policy_number = strtol(fh->fv[FV_ID_PEF_FILTER_ALERTPOLICY].val.s, NULL, 10);
        
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "monitor") == 0) {
            filter->event_severity = 0x01;
        } else
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "information") == 0) {
            filter->event_severity = 0x02;
        } else
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "ok") == 0) {
            filter->event_severity = 0x04;
        } else
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "noncritical") == 0) {
            filter->event_severity = 0x08;
        } else
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "critical") == 0) {
            filter->event_severity = 0x10;
        } else
        if (strcmp(fh->fv[FV_ID_PEF_FILTER_EVENTSEVERITY].val.s, "nonrecoverable") == 0) {
            filter->event_severity = 0x20;
        }
        
        filter->generatorID1 = strtol(fh->fv[FV_ID_PEF_FILTER_GENERATOR_ID1].val.s, NULL, 16);
        filter->generatorID2 = strtol(fh->fv[FV_ID_PEF_FILTER_GENERATOR_ID2].val.s, NULL, 16);
        
        filter->sensor_type = strtol(fh->fv[FV_ID_PEF_FILTER_SENSORTYPE].val.s, NULL, 16);
        filter->sensor_number = strtol(fh->fv[FV_ID_PEF_FILTER_SENSORNUMBER].val.s, NULL, 16);
        filter->event_trigger = strtol(fh->fv[FV_ID_PEF_FILTER_EVENTTRIGGER].val.s, NULL, 16);
        
        filter->offset_mask1 = strtol(fh->fv[FV_ID_PEF_FILTER_DATA1_OFFSET1].val.s, NULL, 16);
        filter->offset_mask2 = strtol(fh->fv[FV_ID_PEF_FILTER_DATA1_OFFSET2].val.s, NULL, 16);
        
        filter->data_filter[0].mask = strtol(fh->fv[FV_ID_PEF_FILTER_DATA1_MASK].val.s, NULL, 16);
        filter->data_filter[0].exact = strtol(fh->fv[FV_ID_PEF_FILTER_DATA1_EXACT].val.s, NULL, 16);
        filter->data_filter[0].data = strtol(fh->fv[FV_ID_PEF_FILTER_DATA1_DATA].val.s, NULL, 16);

        filter->data_filter[1].mask = strtol(fh->fv[FV_ID_PEF_FILTER_DATA2_MASK].val.s, NULL, 16);
        filter->data_filter[1].exact = strtol(fh->fv[FV_ID_PEF_FILTER_DATA2_EXACT].val.s, NULL, 16);
        filter->data_filter[1].data = strtol(fh->fv[FV_ID_PEF_FILTER_DATA2_DATA].val.s, NULL, 16);

        filter->data_filter[2].mask = strtol(fh->fv[FV_ID_PEF_FILTER_DATA3_MASK].val.s, NULL, 16);
        filter->data_filter[2].exact = strtol(fh->fv[FV_ID_PEF_FILTER_DATA3_EXACT].val.s, NULL, 16);
        filter->data_filter[2].data = strtol(fh->fv[FV_ID_PEF_FILTER_DATA3_DATA].val.s, NULL, 16);
        
        if (pp_ipmi_send_command(PP_IPMI_CMD_PEF, PP_IPMI_PEF_FILTER_SET, &in_parm, NULL, NULL, wp->user)) {
            pp_log("liberic_webs.tmpl_ipmi_filter_edit.c - Error sending ipmi set filter command \n");
            set_response(wp, ERIC_RESPONSE_ERROR, _("Error sending the IPMI set filter command"));
            return -1;
        }
    }

    return 0;
}

static void display_filter_form(pp_strstream_t* tab, webs_t wp) {
    pp_ipmi_return_t ipmi_ret;
    const unsigned char* filterNoStr;
    unsigned int filterNo;
    int err;
    vector_t* filters;
    pp_ipmi_pef_filter_t* filter;
    int valid;
    
    valid = 0;
    filter = NULL;
    filters = NULL;
    filterNoStr = websGetVar(wp, "filter", "no_id");
    filterNo = pp_strtoul_10(filterNoStr, 0, &err);
    if (err) {
        filterNo = -1;
        valid = 0;
    }
    else 
    {
        /* read values from ipmi */
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        if (!pp_ipmi_send_command(PP_IPMI_CMD_PEF, PP_IPMI_PEF_FILTER_GET, NULL, &ipmi_ret, NULL, wp->user))
        {
            filters = ipmi_ret.data.pef_filter_list;
            if ((filterNo < 1) | (filterNo > vector_size(filters))) {
                valid = 0;
            } else {
                valid = 1;
                filter = (pp_ipmi_pef_filter_t*)vector_get(filters, filterNo-1);
            }
        }
    }
    
    if (valid) {
        print_filter_form(tab, filter);
    } else {
#if !defined(PP_FEAT_RARITAN_DESIGN)
        pp_strappend(tab, "<table border=\"1\">\n");
#else
        pp_strappend(tab, "<table border=\"0\">\n");
#endif	
        pp_strappend(tab, "<tr><td>&nbsp;<br>");
        pp_strappendf(tab, _("The specified filter %s could not be retrieved"), filterNoStr);
        pp_strappend(tab, "<br>&nbsp;</td></tr>\n</table>\n");
    }

    if (filters) {
        vector_delete(filters);
    }
}

static void print_filter_form(pp_strstream_t* tab, pp_ipmi_pef_filter_t* filter) {
    int ro;
    
    ro = (filter->configuration_type != 0);  // FIXME: use this byte in fields
    
#if !defined(PP_FEAT_RARITAN_DESIGN)
        pp_strappend(tab, "<table border=\"1\">\n");
#else
        pp_strappend(tab, "<table border=\"0\">\n");
#endif

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Filter Number"));
    print_input_cell(tab, FV_ID_PEF_FILTER_NUMBER, "ipmifilteredit", "text", "%d", filter->id, 5, 1);
    pp_strappend(tab, "</tr>\n");
    
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Status"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\"><select ");
#else
    pp_strappend(tab, "<br><select ");
#endif    
    print_fv(tab, FV_ID_PEF_FILTER_STATUS, "ipmifilteredit");
    pp_strappend(tab, ">\n");
    pp_strappend(tab, "<option value=\"enable\"");
    if (filter->enable == 1) pp_strappend(tab, " selected");
    pp_strappend(tab, ">");
    pp_strappend(tab, _("enable"));
    pp_strappend(tab, "\n<option value=\"disable\"");
    if (filter->enable == 0) pp_strappend(tab, " selected");
    pp_strappend(tab, ">");
    pp_strappend(tab, _("disable"));
    pp_strappend(tab, "\n</select>\n");
    pp_strappend(tab, "</td></tr>\n");

    pp_strappend(tab, "<tr>\n");
#if !defined(PP_FEAT_RARITAN_DESIGN)    
    print_head(tab, "");
#endif    
    pp_strappend(tab, "<td class=\"settings_field\">");
    if (filter->configuration_type == 0) {
	pp_strappend(tab, _("user configurable"));
    } else {
	pp_strappend(tab, _("predefined filter"));
    }
    pp_strappend(tab, "</td>\n");	
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Action"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
    pp_strappend(tab, _("Alert"));
#else
    pp_strappend(tab, "<br>\n");
#endif    
    pp_strappend(tab, " <input type=\"checkbox\" ");
    print_fv(tab, FV_ID_PEF_FILTER_ALERT, "ipmifilteredit");
    pp_strappend(tab, " value=\"alert\"");
    if (filter->action.alert == 1) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">\n");
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, _("Alert"));
    pp_strappend(tab, "<br>");
#else    
    pp_strappend(tab, "<br>");
    pp_strappend(tab, _("Reset"));
#endif    
    pp_strappend(tab, " <input type=\"checkbox\" ");
    print_fv(tab, FV_ID_PEF_FILTER_RESET, "ipmifilteredit");
    pp_strappend(tab, " value=\"reset\"");
    if (filter->action.reset == 1) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">\n");
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, _("Reset"));
    pp_strappend(tab, "<br>");
#else
    pp_strappend(tab, "<br>");
    pp_strappend(tab, _("Power off"));
#endif
    pp_strappend(tab, " <input type=\"checkbox\" ");
    print_fv(tab, FV_ID_PEF_FILTER_POWEROFF, "ipmifilteredit");
    pp_strappend(tab, " value=\"poweroff\"");
    if (filter->action.poweroff == 1) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">\n");
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, _("Power off"));
    pp_strappend(tab, "<br>");
#else    
    pp_strappend(tab, "<br>");
    pp_strappend(tab, _("Power cycle"));
#endif
    pp_strappend(tab, " <input type=\"checkbox\" ");
    print_fv(tab, FV_ID_PEF_FILTER_POWERCYCLE, "ipmifilteredit");
    pp_strappend(tab, " value=\"powercycle\"");
    if (filter->action.power_cycle == 1) {
        pp_strappend(tab, " checked");
    }
    pp_strappend(tab, ">\n");
#if defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, _("Power cycle"));
#endif
    pp_strappend(tab, "<br></td>");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Alert Policy"));
    print_input_cell(tab, FV_ID_PEF_FILTER_ALERTPOLICY, "ipmifilteredit", "text", "%d", filter->alert_policy_number, 5, 0);
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Event Severity"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\"><select ");
#else
    pp_strappend(tab, "<br><select ");
#endif    
    print_fv(tab, FV_ID_PEF_FILTER_EVENTSEVERITY, "ipmifilteredit");
    pp_strappend(tab, ">\n");
    pp_strappend(tab, "<option value=\"unspecified\"");
    if (filter->event_severity == 0x00) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("unspecified"));
    pp_strappend(tab, "\n<option value=\"monitor\"");
    if ((filter->event_severity & 0x01) == 0x01) {
        pp_strappend(tab, "selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("monitor"));
    pp_strappend(tab, "\n<option value=\"information\"");
    if ((filter->event_severity & 0x02) == 0x02) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("information"));
    pp_strappend(tab, "\n<option value=\"ok\"");
    if ((filter->event_severity & 0x04) == 0x04) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("ok"));
    pp_strappend(tab, "\n<option value=\"noncritical\"");
    if ((filter->event_severity & 0x08) == 0x08) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("non critical"));
    pp_strappend(tab, "\n<option value=\"critical\"");
    if ((filter->event_severity & 0x10) == 0x10) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("critical"));
    pp_strappend(tab, "\n<option value=\"nonrecoverable\"");
    if ((filter->event_severity & 0x20) == 0x20) {
        pp_strappend(tab, " selected");
    }
    pp_strappend(tab, ">");
    pp_strappend(tab, _("non recoverable"));
    pp_strappend(tab, "\n</select>\n");
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Generator ID"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif    
    print_input(tab, FV_ID_PEF_FILTER_GENERATOR_ID1, "ipmifilteredit", "text", "%.2x", filter->generatorID1, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_GENERATOR_ID2, "ipmifilteredit", "text", "%.2x", filter->generatorID2, 5, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Sensor Type"));
    print_input_cell(tab, FV_ID_PEF_FILTER_SENSORTYPE, "ipmifilteredit", "text", "%.2x", filter->sensor_type, 5, 0);
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Sensor Number"));
    print_input_cell(tab, FV_ID_PEF_FILTER_SENSORNUMBER, "ipmifilteredit", "text", "%.2x", filter->sensor_number, 5, 0);
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Event Trigger"));
    print_input_cell(tab, FV_ID_PEF_FILTER_EVENTTRIGGER, "ipmifilteredit", "text", "%.2x", filter->event_trigger, 5, 0);
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Data 1 Offset mask"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif 
    print_input(tab, FV_ID_PEF_FILTER_DATA1_OFFSET1, "ipmifilteredit", "text", "%.2x", filter->offset_mask1, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA1_OFFSET2, "ipmifilteredit", "text", "%.2x", filter->offset_mask2, 5, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Event Data 1 (AND mask, compare1, compare2)"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif 
    print_input(tab, FV_ID_PEF_FILTER_DATA1_MASK, "ipmifilteredit", "text", "%.2x", filter->data_filter[0].mask, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA1_EXACT, "ipmifilteredit", "text", "%.2x", filter->data_filter[0].exact, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA1_DATA, "ipmifilteredit", "text", "%.2x", filter->data_filter[0].data, 5, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, _("Event Data 2 (AND mask, compare1, compare2)"));
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif 
    print_input(tab, FV_ID_PEF_FILTER_DATA2_MASK, "ipmifilteredit", "text", "%.2x", filter->data_filter[1].mask, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA2_EXACT, "ipmifilteredit", "text", "%.2x", filter->data_filter[1].exact, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA2_DATA, "ipmifilteredit", "text", "%.2x", filter->data_filter[1].data, 5, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    print_head(tab, "Event Data 3 (AND mask, compare1, compare2)");
#if !defined(PP_FEAT_RARITAN_DESIGN)
    pp_strappend(tab, "<td class=\"settings_field\">\n");
#else
    pp_strappend(tab, "<br>\n");
#endif 
    print_input(tab, FV_ID_PEF_FILTER_DATA3_MASK, "ipmifilteredit", "text", "%.2x", filter->data_filter[2].mask, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA3_EXACT, "ipmifilteredit", "text", "%.2x", filter->data_filter[2].exact, 5, 0);
    print_input(tab, FV_ID_PEF_FILTER_DATA3_DATA, "ipmifilteredit", "text", "%.2x", filter->data_filter[2].data, 5, 0);
    pp_strappend(tab, "</td>\n");
    pp_strappend(tab, "</tr>\n");
    
    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "</table>\n");
}

static int ipmi_build_filter_form_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED) {
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;

    pp_strstream_init(&tab);

    display_filter_form(&tab, wp);

    ejSetResult(eid, tab.buf ? tab.buf : "");

    pp_strstream_free(&tab);

    return 0;
}




