#include <liberic_net.h>
#include <pp/intl.h>
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

int
net_common_tmpl_init(void)
{
    return 0;
}

#ifdef PP_FEAT_PORT_CHANGE
static u_short
get_port(webs_t wp, const char * tmpl_name, u_int fv_id, u_short default_port)
{
    char * port_str = get_form_var_value_unsaved(wp, tmpl_name, fv_id);
    u_short port = pp_strtoul_10(port_str, default_port, NULL);
    free(port_str);
    return port;
}
#endif

int
net_common_check_ports(webs_t wp)
{
#ifdef PP_FEAT_PORT_CHANGE
    u_short ports[256];
    u_int port_cnt = 0, i, j;

    ports[port_cnt++] = get_port(wp, TEMPLATE_NET_MISC, FV_ID_NET_HTTPS_PORT, ERIC_NET_DEFAULT_HTTPS_PORT);
    ports[port_cnt++] = get_port(wp, TEMPLATE_NET_MISC, FV_ID_NET_HTTP_PORT, ERIC_NET_DEFAULT_HTTP_PORT);
#ifdef PP_FEAT_TELNET
    ports[port_cnt++] = get_port(wp, TEMPLATE_NET_MISC, FV_ID_NET_TELNET_PORT, ERIC_NET_DEFAULT_TELNET_PORT);
#endif
#ifdef PP_FEAT_SSH
    ports[port_cnt++] = get_port(wp, TEMPLATE_NET_MISC, FV_ID_NET_SSH_PORT, ERIC_NET_DEFAULT_SSH_PORT);
#endif
#ifdef PP_FEAT_TERMINAL_SERVER
    {
	u_short termsrv_baseport = get_port(wp, TEMPLATE_NET_MISC, FV_ID_NET_TERMSRV_BASEPORT, ERIC_NET_DEFAULT_TERMSRV_BASEPORT);
	u_int termsrv_port_cnt = eric_net_get_registered_port_count(PP_NET_PROTOCOL_TYPE_TERMSRV);
	for (i = 0; i < termsrv_port_cnt; ++i) {
	    ports[port_cnt++] = termsrv_baseport + i;
	}
    }
#endif

    /* check port numbers */
    for (i = 0; i < port_cnt; ++i) {
	for (j = 0; j < port_cnt; ++j) {
	    if (i != j && ports[i] == ports[j]) {
		set_response(wp, ERIC_RESPONSE_ERROR, _("The port numbers must be all different."));
		return -1;
	    }
	}
    }
#endif

    return 0;
}

int
net_common_check_ips(webs_t wp, const char * check_ip_str,
		     const char * check_mask_str, int type,
		     const char * msg)
{
    struct in_addr lan_ip, lan_mask, modem_board_ip, modem_client_ip;
    struct in_addr check_ip, check_mask;
    int has_lan_ip = 0, has_modem_board_ip = 0, has_modem_client_ip = 0;
    char * opt = NULL;
    int err;

    if (!check_ip_str || !check_mask_str) { goto internal_error; }

    if (*check_ip_str == '\0' || *check_mask_str == '\0') { return 1; }

    err = inet_pton(AF_INET, check_ip_str, &check_ip);
    if (err <= 0) { goto internal_error; }

    err = inet_pton(AF_INET, check_mask_str, &check_mask);
    if (err <= 0) { goto internal_error; }

    /* get lan ip address */
    if (type & CHECK_AGAINST_LAN) {
	opt = get_form_var_value_unsaved(wp, TEMPLATE_NET_BASIC, FV_ID_NET_IPADDR);
	if (opt) {
	    err = inet_pton(AF_INET, opt, &lan_ip);
	    free(opt);
	    if (err < 0) { goto internal_error; }
	    if (err > 0) { has_lan_ip = 1; }
	}

	/* get lan netmask */
	if (has_lan_ip) {
	    opt = get_form_var_value_unsaved(wp, TEMPLATE_NET_BASIC, FV_ID_NET_NETMASK);
	    if (opt) {
		err = inet_pton(AF_INET, opt, &lan_mask);
		free(opt);
		if (err < 0) { goto internal_error; }
		if (err == 0) { has_lan_ip = 0; }
	    } else {
		has_lan_ip = 0;
	    }
	}
    }

    /* check if modem is enabled */
    if (type & CHECK_AGAINST_MODEM) {
	opt = get_form_var_value_unsaved(wp, TEMPLATE_SERIAL_1, FV_ID_SERIAL_1);
	if (opt) {
	    free(opt);
	    if (!strcmp(opt, "modem")) {
		/* get modem board ip address */
		opt = get_form_var_value_unsaved(wp, TEMPLATE_SERIAL_1, FV_ID_SERIAL_1_MODEM_BOARD_IP);
		if (opt) {
		    err = inet_pton(AF_INET, opt, &modem_board_ip);
		    free(opt);
		    if (err < 0) { goto internal_error; }
		    if (err > 0) { has_modem_board_ip = 1; }
		}
		
		/* get modem client ip address */
		opt = get_form_var_value_unsaved(wp, TEMPLATE_SERIAL_1, FV_ID_SERIAL_1_MODEM_CLIENT_IP);
		if (opt) {
		    err = inet_pton(AF_INET, opt, &modem_client_ip);
		    free(opt);
		    if (err < 0) { goto internal_error; }
		    if (err > 0) { has_modem_client_ip = 1; }
		}
	    }
	}
    }

    /* check against the IP range of the LAN subnet */
    if ((type & CHECK_AGAINST_LAN) && has_lan_ip &&
	(check_ip.s_addr & check_mask.s_addr & lan_mask.s_addr) ==
	(lan_ip.s_addr & check_mask.s_addr & lan_mask.s_addr)) {

	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("The %s must not overlap the IP range of the LAN subnet."),
		     msg);
	return 0;
    }

    /* check against the modem board IP */
    if ((type & CHECK_AGAINST_MODEM) && has_modem_board_ip &&
	(check_ip.s_addr & check_mask.s_addr) ==
	(modem_board_ip.s_addr & check_mask.s_addr)) {

	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("The modem server IP must not lay in the %s."), msg);
	return 0;
    }

    /* check against the modem client IP */
    if ((type & CHECK_AGAINST_MODEM) && has_modem_client_ip &&
	(check_ip.s_addr & check_mask.s_addr) ==
	(modem_client_ip.s_addr & check_mask.s_addr)) {

	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("The modem client IP must not lay in the %s."), msg);
	return 0;
    }

    return 1;

 internal_error:
    set_response(wp, ERIC_RESPONSE_ERROR, pp_intl_translate(internal_error_msg));
    return 0;
}

