#include <liberic_config.h>
#include <pp/intl.h>
#include <pp/ipmi.h>
#include <pp/bmc/tp_ipmi_sens.h>
#include "ej.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "eric_validate.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>

#include "tmpl_ipmi_common.h"

/* internal prototypes */
static void get_all_sensors_list(void);
static void free_all_sensors_list(void);

static void display_outlet_table(pp_strstream_t* tab, webs_t wp UNUSED);
static void add_outlet_table_head(pp_strstream_t *tab);
static void add_outlet_table_entry(pp_strstream_t* tab, int idx);
static int  pdu_build_outlet_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);

static void display_global_table(pp_strstream_t *tab);
static int  pdu_build_global_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);

static void display_break_table(pp_strstream_t* tab, webs_t wp UNUSED);
static void add_break_table_head(pp_strstream_t *tab);
static void add_break_table_entry(pp_strstream_t* tab, int idx);
static int  pdu_build_break_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);

static void display_relay_table(pp_strstream_t* tab, webs_t wp UNUSED);
static void add_relay_table_head(pp_strstream_t *tab);
static void add_relay_table_entry(pp_strstream_t* tab, int idx);
static int  pdu_build_relay_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED);

static int  pdu_set_outlet_on_off(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char **argv);
static void display_detail_table(pp_strstream_t *tab, int idx);
static int  pdu_outlet_detail_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED);
static void display_setup_table(pp_strstream_t *tab, int idx);
static int  pdu_outlet_setup_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED);
static int  pdu_Set_Outlet_Num(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char **argv);

static int pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED);

static int OutletNum=0;

static int nOutletNumber = 0;
static int cOutletNumber = 0;
static int IPMI_STATUS = 0;
static int SDR_STATUS = 0;
static int BREAK_NUMBER = 0;
// Notice : In different models, the outlet number of branches must be the same.
// I have asked the HW engineers, the above assume is ture.
static int BRANCH_NUMBER = 0;
static int OUTLETS_PER_BRANCH_NUMBER = 0;

static int RPC_MODEL_NUMBER = 0xFF;
#define RPC_SENSOR_LIST_SIZE 256
static pp_ipmi_sdr_list_entry_t * rpc_sensor_list[RPC_SENSOR_LIST_SIZE];
#define RPC_MODEL_LIST_SIZE 22
static char rpc_model_list[RPC_MODEL_LIST_SIZE][4][16] = {{"PCR8-15",     "1U",  "8", "15A"}, // 1
							          {"PCR8A-16",    "1U",  "8", "16A"}, // 2
							          {"PCR8A-20L6",  "1U",  "8", "20A"}, // 3
							          {"PCR2X8-15",   "2U", "16", "15A"}, // 4
							          {"PCR2X8-16",   "2U", "16", "16A"}, // 5
							          {"PCRA2X8",     "2U", "16", "20A"}, // 6
							          {"PCS12-20",    "0U", "12", "20A"}, // 7
							          {"PCS12-20L",   "0U", "12", "20A"}, // 8
							          {"PCS12-30L",   "0U", "12", "30A"}, // 9
							          {"PCS12A-16",   "0U", "12", "16A"}, // 10
							          {"PCS20-20",    "0U", "20", "20A"}, // 11
							          {"PCS20-20L",   "0U", "20", "20A"}, // 12
							          {"PCS20-30L",   "0U", "20", "30A"}, // 13
							          {"PCS20A-16",   "0U", "20", "16A"}, // 14
							          {"PCS20A-32",   "0U", "20", "32A"}, // 15
							          {"PCS20A-20L6", "0U", "20", "20A"}, // 16
							          {"PCS20A-30L6", "0U", "20", "30A"}, // 17
							          {"PCR20-20",    "2U", "20", "20A"}, // 18
							          {"PCR20-20L",   "2U", "20", "20A"}, // 19
							          {"PCR20-30L",   "2U", "20", "30A"}, // 20
							          {"PCR20A-16",   "2U", "20", "16A"}, // 21
							          {"PCR20A-32",   "2U", "20", "32A"}, // 22
							         };

FV_SPEC = {
};

int pdu_common_tmpl_init(void)
{
    form_handler_t * fh;
    int i = 0;
    char * sOutletNumber;
    char * sOutletName;
    char * sModelName;

    pp_cfg_get(&sModelName, "device.model");
    pp_log("MODEL - %s\r\n", sModelName);
    for(i = 0; i < RPC_MODEL_LIST_SIZE; i++)
    {
   	if(!strcmp(sModelName, &rpc_model_list[i][0][0]))
   	{
		RPC_MODEL_NUMBER = i;
		pp_log("0\\1\\2U - %s\r\n", &rpc_model_list[i][1][0]);
		pp_log("Outlets - %s\r\n", &rpc_model_list[i][2][0]);
		pp_log("Current - %s\r\n", &rpc_model_list[i][3][0]);
		break;
   	}
    }
    pp_cfg_get(&sOutletNumber, "bmc.rpc.num_outlets");
    cOutletNumber = pp_strtoul_10(sOutletNumber, 0, NULL);
    for(i = 0; i < nOutletNumber; i++)
    {
    	if(PP_FAILED(pp_cfg_get_nodflt(&sOutletName , "bmc.rpc.outlet_name[%u]", i)))
		pp_cfg_set("", "bmc.rpc.outlet_name[%u]", i);
    }
    pp_cfg_save(DO_FLUSH);

    /* register ASPs */
    websAspDefine("pduGetOutletList", pdu_build_outlet_table_asp);
    websAspDefine("pduGetGlobalList", pdu_build_global_table_asp);
    websAspDefine("pduGetBreakList", pdu_build_break_table_asp);
    websAspDefine("pduGetRelayList", pdu_build_relay_table_asp);

    websAspDefine("pduSetOutletOnOff", pdu_set_outlet_on_off); // Warning  - Arg2?
    websAspDefine("pduOutletDetail", pdu_outlet_detail_asp);
    websAspDefine("pduOutletSetup", pdu_outlet_setup_asp);
    websAspDefine("pduSetOutletNum", pdu_Set_Outlet_Num); // Warning  - Arg2?

    fh = CREATE_FH_INSTANCE(TEMPLATE_PDU_OUTLET_LIST, ACL_OBJ_PDU_OUTLET);
    fh->pre_validate_hook = pre_validate_hook;
    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

pp_ipmi_return_t g_ipmi_ret;
pp_strstream_t g_sensor_table = PP_STRSTREAM_INITIALIZER;
static void get_all_sensors_list(void)
{
    u_int sensor_cnt = 0;
    u_int i = 0;

    for (i = 0; i < RPC_SENSOR_LIST_SIZE; i++)
    {
	rpc_sensor_list[i] = NULL; // Reset
    }
    nOutletNumber = 0;
    BRANCH_NUMBER = 0;
    BREAK_NUMBER = 0;

    memset(&g_ipmi_ret, 0, sizeof(g_ipmi_ret));
    pp_strstream_init(&g_sensor_table);
    if (pp_ipmi_send_command(PP_IPMI_CMD_SDR, PP_IPMI_SDR_SUBCMD_LIST,
			     NULL, &g_ipmi_ret, NULL, NULL) != 0)
    {
	IPMI_STATUS = 0;
	goto bail;
    }
    sensor_cnt = vector_size(g_ipmi_ret.data.sdr_list);
    pp_log("found %d SDR entries.\r\n", sensor_cnt);
    if (sensor_cnt == 0)
    {
	SDR_STATUS = 0;
	goto bail;
    }
    for (i = 0; i < sensor_cnt; i++)
    {
	pp_ipmi_sdr_list_entry_t *entry = vector_get(g_ipmi_ret.data.sdr_list, i);
	if (!entry)
	{
	    pp_log("%s(): Error: could not read entry %u\n", ___F, i);
	    continue;
	}
	/* Only ID that between 0x00 to 0xFF have to check*/
	else if((entry) && (entry->sdr_id >= 0) && (entry->sdr_id < RPC_SENSOR_LIST_SIZE))
	{
	    pp_log("TT - %2x - %s\r\n", entry->sdr_id, entry->sdr_name.buf ? : "Unknown");
	    rpc_sensor_list[entry->sdr_id] = entry; // Save to Array
	    if ((((entry->sdr_id & 0xff) >> 3) >= 24) && (((entry->sdr_id & 0xff) >> 3) <= 30) && (((entry->sdr_id & 0xff) & 0x07) == 0))
	    {
	        BRANCH_NUMBER += 1;
	    }
	    else if ((((entry->sdr_id & 0xff) >> 3) >= 0) && (((entry->sdr_id & 0xff) >> 3) <= 23)&& (((entry->sdr_id & 0xff) & 0x07) == 0))
	    {
	        nOutletNumber += 1;
	    }
	    else if ((entry->sdr_id & 0xff) == ((31 << 3) + 0))
	    {
	        BREAK_NUMBER += 1;
	    }
	    else if ((entry->sdr_id & 0xff) == ((31 << 3) + 1))
	    {
	        BREAK_NUMBER += 1;
	    }
	}
    }
    if(BRANCH_NUMBER != 0)
    {
	OUTLETS_PER_BRANCH_NUMBER = nOutletNumber / BRANCH_NUMBER;
    }
    pp_log("TT - OUTLET = %d\r\n", nOutletNumber);
    pp_log("TT - BRANCH = %d\r\n", BRANCH_NUMBER);
    pp_log("TT - BREAK  = %d\r\n", BREAK_NUMBER);
    pp_log("TT - OUTLET/BRANCH = %d\r\n", OUTLETS_PER_BRANCH_NUMBER);
    if((nOutletNumber != 0) || (BRANCH_NUMBER != 0) || (BREAK_NUMBER != 0))
    {
        IPMI_STATUS = 1;
        SDR_STATUS = 1;
    }
    else
    {
        IPMI_STATUS = 0;
        SDR_STATUS = 0;
    }
    return;

 bail:
    pp_log("bail.\r\n");
    return;
}

static void free_all_sensors_list(void) // After you called "get_all_sensors_list()", you have to call "free_all_sensor_list()" at right place.
{
    u_int i = 0;

    for (i = 0; i < RPC_SENSOR_LIST_SIZE; i++)
    {
	rpc_sensor_list[i] = NULL; // Reset
    }
    pp_strstream_free(&g_sensor_table);
    pp_ipmi_cleanup_ret(&g_ipmi_ret);
    return;
}

////////////////////////////////////////////////////////////////
static void display_outlet_table(pp_strstream_t* tab, webs_t wp UNUSED)
{
    int i = 0;

    get_all_sensors_list();
    pp_strappend(tab, "<table border=\"1\">\n");
    add_outlet_table_head(tab);
    for(i = 0; i < cOutletNumber; i++)
    {
    	add_outlet_table_entry(tab, i);
    }
    pp_strappend(tab, "</table>\n");
}

static void add_outlet_table_head(pp_strstream_t *tab)
{
    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Outlet</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Status</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Name</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>RMS Current</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>RMS Voltage</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Real RMS Power</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Average Power</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Actions</td>\n");
    pp_strappend(tab, "</tr>\n");
}

static void add_outlet_table_entry(pp_strstream_t* tab, int idx)
{
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;
    int Status = 0;
    char * sOutletName;
    char tmp[256];
/*
    if(IPMI_STATUS == 0)
    {
	return;
    }
*/
    params.data.oem_pp_rpc_get_recep_state.receptacle = (int)idx;
    if(pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC, 
                            PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE,
                            &params, &ipmi_ret, NULL, NULL))
    {
	Status = 0; // Outlet's Status - Unknow
    }
    else
    {
	Status = 1; // Outlet's Status - Known
    }
    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    sprintf(tmp, "  %u  ", idx+1);
    pp_strappend(tab, tmp);
    pp_strappend(tab, "</tt></td>\n");

    if(Status == 0)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#0000FF\">Unknow</font></tt></td>\n");
    }
    else if((ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.status == 1) && (Status == 1))
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#0000FF\">ON</font></tt></td>\n");
    }
    else if((ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.status != 1) && (Status == 1))
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">OFF</font></tt></td>\n");
    }

    pp_ipmi_sdr_list_entry_t *Current_sen = rpc_sensor_list[(idx << 3) + 0];
    pp_ipmi_sdr_list_entry_t *UnitVoltage_sen = NULL;
    if(OUTLETS_PER_BRANCH_NUMBER != 0)
    {
    	UnitVoltage_sen = rpc_sensor_list[((24 + ((int)(idx / OUTLETS_PER_BRANCH_NUMBER))) << 3) + 0];
    }
    pp_ipmi_sdr_list_entry_t *Power_sen = rpc_sensor_list[(idx << 3) + 3];
    pp_ipmi_sdr_list_entry_t *AveragePower_sen = rpc_sensor_list[(idx << 3) + 6];

    pp_cfg_get(&sOutletName, "bmc.rpc.outlet_name[%u]", idx);
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    if(sOutletName[0] == 0)
    {
    	sprintf(sOutletName, "&nbsp;");
    }
    pp_strappend(tab, sOutletName);
    pp_strappend(tab, "</tt></td>\n");

    if(Current_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", Current_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    if(UnitVoltage_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	 pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	 sprintf(tmp, "%u V", (int) UnitVoltage_sen->data.full.reading.value);
    	 pp_strappend(tab, tmp);
    	 pp_strappend(tab, "</tt></td>\n");
        if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
        else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
        else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
        else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
        else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
        else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    	    pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
        else
    	    pp_strappend(tab, "  <td class = \"normal\" align=right><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    if(Power_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", Power_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    }

    if(AveragePower_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", AveragePower_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    }

    char slink[512] = "";
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    sprintf(slink,"<a href=\"home2.asp?outlet=%d&setup=1\">[On]</a> <a href=\"home2.asp?outlet=%d&setup=0\">[Off]</a> <a href=\"outlet_detail.asp?outlet=%d\">[Details]</a> <a href=\"outlet_setup.asp?outlet=%d\">[Setup]</a>", idx+1, idx+1, idx+1, idx+1);
    pp_strappend(tab, slink);
    pp_strappend(tab, "</tt></td>\n");
}

static int pdu_build_outlet_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    display_outlet_table(&tab, wp);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    return 0;
}

/////////////////////////////////////////////////////////////
static void display_global_table(pp_strstream_t *tab)
{
    char tmp[256];

    pp_strappend(tab, "<table border=\"1\">\n");

    pp_ipmi_sdr_list_entry_t *Current_sen = rpc_sensor_list[(31<<3) + 2];
    pp_ipmi_sdr_list_entry_t *MaxCurrent_sen = rpc_sensor_list[(31<<3) + 3];
    pp_ipmi_sdr_list_entry_t *Temperature_sen = rpc_sensor_list[(31<<3) + 4];
    pp_ipmi_sdr_list_entry_t *Power_sen = rpc_sensor_list[(31<<3) + 5];
    pp_ipmi_sdr_list_entry_t *ApparentPower_sen = rpc_sensor_list[(31<<3) + 6];

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "RMS Current");
    pp_strappend(tab, "</tt></td>\n");
    if(Current_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", Current_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Maximum RMS Current");
    pp_strappend(tab, "</tt></td>\n");
    if(MaxCurrent_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", MaxCurrent_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
	    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }


    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Temperature");
    pp_strappend(tab, "</tt></td>\n");
    if(Temperature_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f C", Temperature_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Temperature_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");

    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Real RMS Power");
    pp_strappend(tab, "</tt></td>\n");
    if(Power_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", Power_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Apparent RMS Power");
    pp_strappend(tab, "</tt></td>\n");
    if(ApparentPower_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", ApparentPower_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "</table>\n");
}

static int pdu_build_global_table_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    display_global_table(&tab);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    return 0;
}

//////////////////////////////////////////////////////////////
static void display_break_table(pp_strstream_t* tab, webs_t wp UNUSED)
{
    int i=0;

    pp_strappend(tab, "<table border=\"1\">\n");
    add_break_table_head(tab);
    for(i=0; i<BREAK_NUMBER; i++)
    {
    	add_break_table_entry(tab, i);
    }
    pp_strappend(tab, "</table>\n");
}

static void add_break_table_head(pp_strstream_t *tab)
{
    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Circuit Breaker</td>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Status</td>\n");
    pp_strappend(tab, "</tr>\n");
}

static void add_break_table_entry(pp_strstream_t* tab, int idx)
{
    char tmp[256];

    pp_ipmi_sdr_list_entry_t *CircuitBreak_sen = rpc_sensor_list[(31<<3) + idx];

    pp_strappend(tab, "<tr>\n");

    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    sprintf(tmp, "%u", idx+1);
    pp_strappend(tab, tmp);
    pp_strappend(tab, "</tt></td>\n");

    if(CircuitBreak_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	/* casting float to int; ok? */
    	if((int)CircuitBreak_sen->data.full.reading.value == 1)
    	{
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt>Open</tt></td>\n");
    	}
    	/* casting float to int; ok? */
    	else if((int)CircuitBreak_sen->data.full.reading.value == 2)
    	{
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt>Close</tt></td>\n");
    	}
    	else
    	{
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt>Unknow</tt></td>\n");
    	}
    }
}

static int pdu_build_break_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    display_break_table(&tab, wp);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    return 0;
}

/////////////////////////////////////////////////////////////
static void display_relay_table(pp_strstream_t* tab, webs_t wp UNUSED)
{
    int i=0;

    pp_strappend(tab, "<table border=\"1\">\n");
    add_relay_table_head(tab);
    for(i=0; i<BRANCH_NUMBER; i++)
    {
    	add_relay_table_entry(tab, i);
    }
    pp_strappend(tab, "</table>\n");
}

static void add_relay_table_head(pp_strstream_t *tab)
{
    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\"           align=center>Relay Board</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>RMS Current</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>RMS Voltage</td>\n");
    pp_strappend(tab, "</tr>\n");
}

static void add_relay_table_entry(pp_strstream_t* tab, int idx)
{
    char tmp[256];

    pp_ipmi_sdr_list_entry_t *Voltage_sen = NULL;
    pp_ipmi_sdr_list_entry_t *Current_sen = NULL;
    if(OUTLETS_PER_BRANCH_NUMBER != 0)
    {
    	Voltage_sen = rpc_sensor_list[((24 + ((int)(idx / OUTLETS_PER_BRANCH_NUMBER))) << 3) + 0];
    	Current_sen = rpc_sensor_list[((24 + ((int)(idx / OUTLETS_PER_BRANCH_NUMBER))) << 3) + 1];
    }
    pp_strappend(tab, "<tr>\n");

    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    sprintf(tmp, "%u", idx+1);
    pp_strappend(tab, tmp);
    pp_strappend(tab, "</tt></td>\n");

    if(Current_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", Current_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    if(Voltage_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%u V", (int) Voltage_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Voltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }
}

static int pdu_build_relay_table_asp(int eid, webs_t wp, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    display_relay_table(&tab, wp);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    free_all_sensors_list();
    return 0;
}

/////////////////////////////////////////////////////////////
static int pdu_set_outlet_on_off(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char **argv)
{
    pp_ipmi_parameter_t params;
    int setup = 2;

    params.data.oem_pp_rpc_set_recep_state.receptacle = pp_strtoul_10(argv[0], 0, NULL) - 1;
    pp_log("OutletNum = %d\r\n", params.data.oem_pp_rpc_set_recep_state.receptacle);
    setup = pp_strtoul_10(argv[1], 0, NULL);
    if(setup == 2)
    {
    	return 0;
    }
    else
    {
    	params.data.oem_pp_rpc_set_recep_state.status = setup;
    	pp_log("On\\Off = %d\r\n", params.data.oem_pp_rpc_set_recep_state.status);
    }
    pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
			PP_IPMI_OEM_PP_RPC_SET_RECEPTACLE_STATE,
			&params, NULL, NULL, NULL);
    return 0;
}

static int pdu_Set_Outlet_Num(int eid UNUSED, webs_t wp UNUSED, int argc UNUSED, char **argv)
{
    OutletNum = pp_strtoul_10(argv[0], 0, NULL) -1;
    pp_log("Set OutletNum = %d\r\n", OutletNum);
    return 0;
}

/////////////////////////////////////////////////////////////
static void display_detail_table(pp_strstream_t *tab, int idx)
{
    char sTitle[128]="";
    char * sOutletName;
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;
    char tmp[128];

    pp_cfg_get(&sOutletName, "bmc.rpc.outlet_name[%u]", idx);

    params.data.oem_pp_rpc_get_recep_state.receptacle = idx;
    pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
			PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE,
			&params, &ipmi_ret, NULL, NULL);

    if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.status == 1)
    	sprintf(sTitle,"Outlet %d,%s, is ON", idx+1, sOutletName);
    else
    	sprintf(sTitle,"Outlet %d,%s, is OFF", idx+1, sOutletName);

    pp_strappend(tab, "<table border=\"1\">\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=3 align=center>");
    pp_strappend(tab, sTitle);
    pp_strappend(tab, "  </td>\n");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>Measurement</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>Value</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>Status</td>\n");

    pp_ipmi_sdr_list_entry_t *Current_sen = rpc_sensor_list[(idx<<3) + 0];
    pp_ipmi_sdr_list_entry_t *MaxCurrent_sen = rpc_sensor_list[(idx<<3) + 2];
    pp_ipmi_sdr_list_entry_t *UnitVoltage_sen = NULL;
    if(OUTLETS_PER_BRANCH_NUMBER != 0)
    {
    	UnitVoltage_sen = rpc_sensor_list[((24 + ((int)(idx / OUTLETS_PER_BRANCH_NUMBER))) << 3) + 0];
    }
    pp_ipmi_sdr_list_entry_t *Power_sen = rpc_sensor_list[(idx<<3) + 3];
    pp_ipmi_sdr_list_entry_t *ApparentPower_sen = rpc_sensor_list[(idx<<3) + 4];
    pp_ipmi_sdr_list_entry_t *AveragePower_sen = rpc_sensor_list[(idx<<3) + 5];
    pp_ipmi_sdr_list_entry_t *MaxPower_sen = rpc_sensor_list[(idx<<3) + 6];

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "RMS Current");
    pp_strappend(tab, "</tt></td>\n");
    if(Current_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", Current_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Current_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Maximum RMS Current");
    pp_strappend(tab, "</tt></td>\n");
    if(MaxCurrent_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f A", MaxCurrent_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(MaxCurrent_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "RMS Voltage");
    pp_strappend(tab, "</tt></td>\n");
    if(UnitVoltage_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%u V", (int) UnitVoltage_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(UnitVoltage_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Real RMS Power");
    pp_strappend(tab, "</tt></td>\n");
    if(Power_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", Power_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(Power_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Apparent RMS Power");
    pp_strappend(tab, "</tt></td>\n");
    if(ApparentPower_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", ApparentPower_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(ApparentPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Average Power");
    pp_strappend(tab, "</tt></td>\n");
    if(AveragePower_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", AveragePower_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(AveragePower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Maximum Real RMS Power");
    pp_strappend(tab, "</tt></td>\n");
    if(MaxPower_sen == NULL)
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	pp_strappend(tab, "???");
    	pp_strappend(tab, "</tt></td>\n");
    }
    else
    {
    	pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    	sprintf(tmp, "%.1f W", MaxPower_sen->data.full.reading.value);
    	pp_strappend(tab, tmp);
    	pp_strappend(tab, "</tt></td>\n");
    	if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_NO_READING)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    	else if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_OK)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#00FF00\">OK</font></tt></td>\n");
    	else if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_NON_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FFFF00\">Non Critical</font></tt></td>\n");
    	else if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_LOWER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else if(MaxPower_sen->data.full.status == PP_IPMI_SENSOR_STATUS_UPPER_CRITICAL)
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF0000\">Critical</font></tt></td>\n");
    	else
    		pp_strappend(tab, "  <td class = \"normal\" align=center><tt><font color=\"#FF00FF\">Unknow</font></tt></td>\n");
    }

    pp_strappend(tab, "<tr>\n");

    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    pp_strappend(tab, "Total Power Consumption");
    pp_strappend(tab, "</tt></td>\n");

    pp_strappend(tab, "  <td class = \"normal\" align=center><tt>");
    if(AveragePower_sen == NULL)
    {
    	sprintf(tmp, "???");
    }
    else
    {
    	sprintf(tmp, "%.1f KWh", AveragePower_sen->data.full.reading.value * 60 * 60 / 1000);
    }
    pp_strappend(tab, tmp);
    pp_strappend(tab, "</tt></td>\n");

    pp_strappend(tab, "  <td class=\"bold\" align=center>&nbsp;</td>\n");

    pp_strappend(tab, "</table>\n");
}

static int pdu_outlet_detail_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    get_all_sensors_list();
    display_detail_table(&tab, OutletNum);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    free_all_sensors_list();
    return 0;
}

/////////////////////////////////////////////////////////////
static void display_setup_table(pp_strstream_t *tab, int idx)
{
    char sTitle[128]="";
    char * sOutletName;
    pp_ipmi_parameter_t params;
    pp_ipmi_return_t ipmi_ret;

    pp_cfg_get(&sOutletName, "bmc.rpc.outlet_name[%u]", idx);

    params.data.oem_pp_rpc_get_recep_state.receptacle = idx;
    pp_ipmi_send_command(PP_IPMI_CMD_OEM_PP_RPC,
			PP_IPMI_OEM_PP_RPC_GET_RECEPTACLE_STATE,
			&params, &ipmi_ret, NULL, NULL);

    if(ipmi_ret.data.oem_pp_rpc_get_recep_state_rs.status == 1)
    	sprintf(sTitle,"Outlet %d,%s, is ON", idx+1, sOutletName);
    else
    	sprintf(sTitle,"Outlet %d,%s, is OFF", idx+1, sOutletName);

    pp_ipmi_sdr_list_entry_t *Current_sen = rpc_sensor_list[(idx<<3) + 0];
    pp_ipmi_sdr_list_entry_t *UnitVoltage_sen = NULL;
    if(OUTLETS_PER_BRANCH_NUMBER != 0)
    {
    	UnitVoltage_sen = rpc_sensor_list[((24 + ((int)(idx / OUTLETS_PER_BRANCH_NUMBER))) << 3) + 0];
    }

    pp_strappend(tab, "<table border=\"1\">\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=9 align=center>");
    pp_strappend(tab, sTitle);
    pp_strappend(tab, "  </td>\n");
    pp_strappend(tab, "</tr>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>Name</td>\n");
    //pp_strappend(tab, "  <td class=\"bold\" colspan=8 align=left>");
    //pp_strappend(tab, sOutletName);
    //pp_strappend(tab, "</td>\n");
    char inputName[256];
    sprintf(inputName, "<td class=\"bold\" colspan=8 align=left> <input type=\"text\" name=\"OutletName\" maxlength=\"63\" size=\"64\" value=\"%s\">\n </td>\n", sOutletName);
    pp_strappend(tab, inputName);

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" rowspan=3 align=center>Measurement</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=8 align=center>Thresholds</td>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=4 align=center>Lower</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=4 align=center>Upper</td>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>Non-Critical</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>Critical</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>Non-Critical</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=2 align=center>Critical</td>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>RMS Current</td>\n");
    pp_strappend(tab, "  <td class=\"bold\" colspan=4 align=center>&nbsp;</td>\n");
    //pp_strappend(tab, "  <td class=\"bold\" align=center>&nbsp;</td>\n");
    //pp_strappend(tab, "  <td class=\"bold\" align=center>&nbsp;</td>\n");
    //pp_strappend(tab, "  <td class=\"bold\" align=center>&nbsp;</td>\n");

    char inputCurrentUpperNonCritical[256];
    if(Current_sen == NULL)
    {
    	sprintf(inputCurrentUpperNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"CurrentUpperNonCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputCurrentUpperNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"CurrentUpperNonCritical\" maxlength=\"4\" size=\"5\" value=\"%.1f\">\n </td>\n", Current_sen->data.full.upper_non_critical.value);
    }
    pp_strappend(tab, inputCurrentUpperNonCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    char inputCurrentUpperCritical[256];
    if(Current_sen == NULL)
    {
    	sprintf(inputCurrentUpperCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"CurrentUpperCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputCurrentUpperCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"CurrentUpperCritical\" maxlength=\"4\" size=\"5\" value=\"%.1f\">\n </td>\n", Current_sen->data.full.upper_critical.value);
    }
    pp_strappend(tab, inputCurrentUpperCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    pp_strappend(tab, "<tr>\n");
    pp_strappend(tab, "  <td class=\"bold\" align=center>RMS Voltage</td>\n");

    char inputVoltageLowerNonCritical[256];
    if(UnitVoltage_sen == NULL)
    {
    	sprintf(inputVoltageLowerNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageLowerNonCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputVoltageLowerNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageLowerNonCritical\" maxlength=\"4\" size=\"5\" value=\"%.0f\">\n </td>\n", UnitVoltage_sen->data.full.lower_non_critical.value);
    }
    pp_strappend(tab, inputVoltageLowerNonCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    char inputVoltageLowerCritical[256];
    if(UnitVoltage_sen == NULL)
    {
    	sprintf(inputVoltageLowerCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageLowerCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputVoltageLowerCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageLowerCritical\" maxlength=\"4\" size=\"5\" value=\"%.0f\">\n </td>\n", UnitVoltage_sen->data.full.lower_critical.value);
    }
    pp_strappend(tab, inputVoltageLowerCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    char inputVoltageupperNonCritical[256];
    if(UnitVoltage_sen == NULL)
    {
    	sprintf(inputVoltageupperNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageUpperNonCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputVoltageupperNonCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageUpperNonCritical\" maxlength=\"4\" size=\"5\" value=\"%.0f\">\n </td>\n", UnitVoltage_sen->data.full.upper_non_critical.value);
    }
    pp_strappend(tab, inputVoltageupperNonCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    char inputVoltageupperCritical[256];
    if(UnitVoltage_sen == NULL)
    {
    	sprintf(inputVoltageupperCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageUpperCritical\" maxlength=\"4\" size=\"5\" value=\"???\">\n </td>\n");
    }
    else
    {
    	sprintf(inputVoltageupperCritical, "<td class=\"bold\" align=center> <input type=\"text\" name=\"VoltageUpperCritical\" maxlength=\"4\" size=\"5\" value=\"%.0f\">\n </td>\n", UnitVoltage_sen->data.full.upper_critical.value);
    }
    pp_strappend(tab, inputVoltageupperCritical);
    pp_strappend(tab, "  <td class=\"bold\" align=center></td>\n");

    pp_strappend(tab, "</table>\n");
}

static int pdu_outlet_setup_asp(int eid, webs_t wp UNUSED, int argc UNUSED, char **argv UNUSED)
{
    pp_strstream_t tab = PP_STRSTREAM_INITIALIZER;
    pp_strstream_init(&tab);
    get_all_sensors_list();
    display_setup_table(&tab, OutletNum);
    ejSetResult(eid, tab.buf ? tab.buf : "");
    pp_strstream_free(&tab);
    free_all_sensors_list();
    return 0;
}

static void str2float(const char * threshold, float * value)
{
    float factor = 1.0;
    int i = 0, j = 0, k = 0, m = 0;

    for(k=0; threshold[k]!=0; k++)
    {
    	if(threshold[k] == '.')
    	{
    		//pp_log("%d\r\n", k);
    		break;
    	}
    }
    for(i=0; i<k; i++)
    {
       for(factor = 1.0, m=i; m<k-1; m++) factor = factor * 10;
    	*value = *value + (threshold[i] - '0') * factor;
    	//pp_log("%d %f %f\r\n", i, *value, factor);
    }
    if(threshold[k] != 0)
    {
    	for(j=k+1; threshold[j]!=0; j++)
    	{
       	factor = factor / 10;
    		*value = *value + (threshold[j] - '0') * factor;
    		//pp_log("%d %f %f\r\n", j, *value, factor);
    	}
    }
}

static int pre_validate_hook(webs_t wp, form_handler_t * fh UNUSED)
{
    const char * fv_outlet_name;
    const char * fv_current_unc;
    const char * fv_current_uc;
    const char * fv_voltage_lnc;
    const char * fv_voltage_lc;
    const char * fv_voltage_unc;
    const char * fv_voltage_uc;

    float current_unc = 0.0;
    float current_uc = 0.0;
    float voltage_lnc = 0.0;
    float voltage_lc = 0.0;
    float voltage_unc = 0.0;
    float voltage_uc = 0.0;

    fv_outlet_name = websGetVar(wp, "OutletName", "");
    pp_cfg_set(fv_outlet_name, "bmc.rpc.outlet_name[%u]", OutletNum);
    pp_cfg_save(DO_FLUSH);

    fv_current_unc = websGetVar(wp, "CurrentUpperNonCritical", ""); // 1
    str2float(fv_current_unc, &current_unc);
    //pp_log("%f\r\n\r\n", current_unc);
    fv_current_uc = websGetVar(wp, "CurrentUpperCritical", ""); // 2
    str2float(fv_current_uc, &current_uc);
    //pp_log("%f\r\n\r\n", current_uc);

    fv_voltage_lnc = websGetVar(wp, "VoltageLowerNonCritical", ""); // 3
    str2float(fv_voltage_lnc, &voltage_lnc);
    //pp_log("%f\r\n\r\n", voltage_lnc);
    fv_voltage_lc = websGetVar(wp, "VoltageLowerCritical", ""); // 4
    str2float(fv_voltage_lc, &voltage_lc);
    //pp_log("%f\r\n\r\n", voltage_lc);
    fv_voltage_unc = websGetVar(wp, "VoltageUpperNonCritical", ""); // 5
    str2float(fv_voltage_unc, &voltage_unc);
    //pp_log("%f\r\n\r\n", voltage_unc);
    fv_voltage_uc = websGetVar(wp, "VoltageUpperCritical", ""); // 6
    str2float(fv_voltage_uc, &voltage_uc);
    //pp_log("%f\r\n\r\n", voltage_uc);
    
    // ToDo : Save these values
    pp_ipmi_parameter_t params;
    pp_ipmi_sensor_set_th_param_t threshold1;
    pp_ipmi_sensor_set_th_param_t threshold2;
    pp_ipmi_sensor_set_th_param_t threshold3;
    pp_ipmi_sensor_set_th_param_t threshold4;
    pp_ipmi_sensor_set_th_param_t threshold5;
    pp_ipmi_sensor_set_th_param_t threshold6;
    char set_sensor_name[32];

    memset(&params, 0, sizeof(params));
    params.data.sensor_threshold_list = vector_new2(NULL, 1, sizeof(pp_ipmi_sensor_set_th_param_t), NULL);

    sprintf(set_sensor_name, "Current R.%d", OutletNum + 1);
    //pp_log("%s\r\n", set_sensor_name);
    threshold1.sensor_name = set_sensor_name;
    threshold1.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT;
    threshold1.value= current_unc;
    threshold1.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold1);

    threshold2.sensor_name = set_sensor_name;
    threshold2.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT;
    threshold2.value= current_uc;
    threshold2.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold2);

    sprintf(set_sensor_name, "RB.1 Voltage");
    //pp_log("%s\r\n", set_sensor_name);
    threshold3.sensor_name = set_sensor_name;
    threshold3.threshold_type = PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT;
    threshold3.value= voltage_lnc;
    threshold3.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold3);

    threshold4.sensor_name = set_sensor_name;
    threshold4.threshold_type = PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT;
    threshold4.value= voltage_lc;
    threshold4.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold4);

    threshold5.sensor_name = set_sensor_name;
    threshold5.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT;
    threshold5.value= voltage_unc;
    threshold5.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold5);

    threshold6.sensor_name = set_sensor_name;
    threshold6.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT;
    threshold6.value= voltage_uc;
    threshold6.use_cim = 0;
    vector_add2(params.data.sensor_threshold_list, &threshold6);

    pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
			PP_IPMI_SENSOR_SUBCMD_THRESHOLD,
			&params, NULL, NULL, NULL);

    vector_delete(params.data.sensor_threshold_list);
    return 0; // Stop & Return
}
