#include <sys/ioctl.h>
#include <pp/kvm.h>
#include <pp/powerswitch.h>
#include <pp/intl.h>
#include <liberic_misc.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

#ifdef PP_FEAT_POWER_CTRL_ENABLE

FV_SPEC = {
    {
	id:		FV_ID_PC_POWER_PORT_0,
	cfgkey:		"pc.port_0",
    },
    {
	id:		FV_ID_PC_POWER_PORT_1,
	cfgkey:		"pc.port_1",
    },
    {
	id:		FV_ID_PC_POWER_DEVICE_0,
	cfgkey:		"pc.device_0",
    },
    {
	id:		FV_ID_PC_POWER_DEVICE_1,
	cfgkey:		"pc.device_1",
    },
    {
	id:		FV_ID_PC_PORT_VALUES,
	cfgkey:		"pc.ps_port_values[%I]",
    },
};

static int post_save_hook(webs_t wp, form_handler_t * fh);
static void display_power_state(webs_t wp, const char* port_str, const char* dev_str, const char* var_name, unsigned int port_id);
static int pc_direct_set_vars_asp(int eid, webs_t wp, int argc, char ** argv);
static int ps_is_supported_asp(int eid, webs_t wp, int argc, char ** argv);
static int ps_has_capability_asp(int eid, webs_t wp, int argc, char ** argv);
static int ps_get_longname_asp(int eid, webs_t wp, int argc, char ** argv);

int
power_control_direct_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("pcDirectSetVars", pc_direct_set_vars_asp);
    websAspDefine("psIsSupported", ps_is_supported_asp);
    websAspDefine("psHasCapability", ps_has_capability_asp);
    websAspDefine("psGetLongName", ps_get_longname_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_POWER_CONTROL_DIRECT, ACL_OBJ_POWER_CONTROL_DIRECT);

    fh->post_save_hook = post_save_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

#define POWER_STATE_SERIAL1	"current_power_state_serial1"
#define POWER_STATE_SERIAL2	"current_power_state_serial2"

static int
post_save_hook(webs_t wp, form_handler_t * fh)
{
    form_var_t *fv_port0  = NULL;
    form_var_t *fv_port1  = NULL;
    form_var_t *fv_dev0   = NULL;
    form_var_t *fv_dev1   = NULL;

    int port = -1;
    int dev = -1;
    int res;
    int seq_switch = 0;
    u_int port_id = 0;
    int switch_on = 1;
    unsigned char use_delays = 0;
    u_short port_values = 0;
    int ps1_logged_in, ps2_logged_in;

    /* open power switch on serial port 1 */
    ps1_logged_in = PP_SUCCED(power_control_common_power_switch_login(wp, PP_POWER_PORT_ID_SERIAL_1));
    
    /* open power switch on serial port 2 */
    ps2_logged_in = PP_SUCCED(power_control_common_power_switch_login(wp, PP_POWER_PORT_ID_SERIAL_2));

    fv_port0 = &fh->fv[FV_ID_PC_POWER_PORT_0];
    fv_port1 = &fh->fv[FV_ID_PC_POWER_PORT_1];
    fv_dev0 = &fh->fv[FV_ID_PC_POWER_DEVICE_0];
    fv_dev1 = &fh->fv[FV_ID_PC_POWER_DEVICE_1];

    if (form_button_clicked(wp, "action_power_on_direct_serial1")
	|| form_button_clicked(wp, "action_power_off_direct_serial1")) {
	if (!ps1_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto showstate;
	}
	switch_on = form_button_clicked(wp, "action_power_on_direct_serial1") ? 1 : 0;
	port = pp_strtol_10(fv_port0->val.s, -1, NULL);
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_1;
    } else if (form_button_clicked(wp, "action_power_on_direct_serial2")
	       || form_button_clicked(wp, "action_power_off_direct_serial2")) {
	if (!ps2_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto showstate;
	}
	switch_on = form_button_clicked(wp, "action_power_on_direct_serial2") ? 1 : 0;
	port = pp_strtol_10(fv_port1->val.s, -1, NULL);
	dev = pp_strtol_10(fv_dev1->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_2;
    } else if (form_button_clicked(wp, "action_reset_direct_serial1")) {
	if (!ps1_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto showstate;
	}
	port = pp_strtol_10(fv_port0->val.s, -1, NULL);
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_1;
	if (PP_SUCCED(pp_power_reset(0, 0, port_id))) {
	    set_response(wp, ERIC_RESPONSE_OK, "Reset performed.");	    
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, "Reset failed.");
	}
	goto showstate;
    } else if (form_button_clicked(wp, "action_reset_direct_serial2")) {
	if (!ps2_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto showstate;
	}
	port = pp_strtol_10(fv_port1->val.s, -1, NULL);
	dev = pp_strtol_10(fv_dev1->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_2;
	if (PP_SUCCED(pp_power_reset(0, 0, port_id))) {
	    set_response(wp, ERIC_RESPONSE_OK, _("Reset performed."));	    
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Reset failed."));
	}
	goto showstate;
    } else if (form_button_clicked(wp, "action_switch_seq_on_serial1")
	       || form_button_clicked(wp, "action_switch_seq_off_serial1")) {
	if (!ps1_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto setvars;
	}
	port = 0;
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_1;
	seq_switch = PWR_SWITCH_SEQ;
	switch_on = form_button_clicked(wp, "action_switch_seq_on_serial1") ? 1 : 0;
	if (PP_SUCCED(pp_power_switch(port, dev,  switch_on, seq_switch, port_id))) {
	    set_response(wp, ERIC_RESPONSE_OK, _("Power switch in progress."));	    
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Power switch failed."));
	}        
	goto setvars;
    } else if (form_button_clicked(wp, "action_switch_seq_on_serial2")
	       || form_button_clicked(wp, "action_switch_seq_off_serial2")) {
	if (!ps2_logged_in) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Connection could not be established!"));
	    goto showstate;
	}
	port = 0;
	dev = pp_strtol_10(fv_dev1->val.s, -1, NULL);
	port_id = PP_POWER_PORT_ID_SERIAL_2;
	seq_switch = PWR_SWITCH_SEQ;
	switch_on = form_button_clicked(wp, "action_switch_seq_on_serial2") ? 1 : 0;
	if (PP_SUCCED(pp_power_switch(port, dev,  switch_on, seq_switch, port_id))) {
	    set_response(wp, ERIC_RESPONSE_OK, _("Power switch in progress."));	    
	} else {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Power switch failed."));
	}        
	goto setvars;
    } else if (form_button_clicked(wp, "action_get_state_serial1")
	       || form_button_clicked(wp, "action_get_state_serial2")) {
	set_response(wp, ERIC_RESPONSE_OK, ""); /* prevent default response */
	goto showstate;
    } else if (form_button_clicked(wp, "action_apply_power_state_serial1")) {
	int i;
	const char *var_buf;
	port_id = PP_POWER_PORT_ID_SERIAL_1;
	port = 0;
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL); /* -1 as error ok? */
	for (i = 0; i < fh->fv[FV_ID_PC_PORT_VALUES].val_cnt; i++) {
	    if (!strcmp(fh->fv[FV_ID_PC_PORT_VALUES].val.m[i], "on")) {
		port_values = port_values | ( 1 << i );
	    }
	}

	var_buf = websGetVar(wp, "_ps_power_use_power_on_delays", NULL);

	if ( var_buf && !strncmp(var_buf, "on", 2) ) {
	    use_delays = 1;
	} else {
	    use_delays = 0;
	}
	
	pp_log("Port Value %d\n", port_values);	
    } else if (form_button_clicked(wp, "action_update_power_device_serial1")) {	
	goto showstate;
    } else if (form_button_clicked(wp, "action_power_on_all_ports_serial1")) {
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	res = pp_power_switch_all(dev, 1, PP_POWER_PORT_ID_SERIAL_1);
	goto showstate;
    } else if (form_button_clicked(wp, "action_power_off_all_ports_serial1")) {
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	res = pp_power_switch_all(dev, 0,PP_POWER_PORT_ID_SERIAL_1);
	goto showstate;
    } else if (form_button_clicked(wp, "action_delay_time_on_serial1") ||
	       form_button_clicked(wp, "action_delay_time_off_serial1")) {
	dev = pp_strtol_10(fv_dev0->val.s, -1, NULL);
	if (form_button_clicked(wp, "action_delay_time_on_serial1")) {
	    res = pp_power_set_delay_enable(dev, 0, PP_POWER_PORT_ID_SERIAL_1);
	} else {
	    res = pp_power_set_delay_enable(dev, 1, PP_POWER_PORT_ID_SERIAL_1);
	}
	goto showstate;
    } else {
#if defined(OEM_TANGTOP)
	int i;
	char btn_name[64];
	for  ( i = 0; i < 16; i++ ) {
	    snprintf(btn_name, sizeof(btn_name), "action_power_on_port_%d_serial1", i);
	    if ( form_button_clicked(wp, btn_name) ) {
		pp_log("Port %d clicked on!\n", i);
		port = i;
		switch_on = 1;
	    }
	}
	for  ( i = 0; i < 16; i++ ) {
	    snprintf(btn_name, sizeof(btn_name), "action_power_off_port_%d_serial1", i);
	    if ( form_button_clicked(wp, btn_name) ) {
		pp_log("Port %d clicked off!\n", i);
		port = i;
		switch_on = 0;
	    }
	}
	dev = strtol(fv_dev0->val.s, NULL, 10);
	port_id = PP_POWER_PORT_ID_SERIAL_1;
#else 
	port = -1;
	dev = -1;
#endif
    }
    
    if (port < 0 || dev < 0) goto finish;

    if (form_button_clicked(wp, "action_apply_power_state_serial1")) {
	res = pp_power_switch_simultan(dev, port_values, port_id);
    } else {
	res = pp_power_switch(port, dev, switch_on, seq_switch, port_id);
    }
    set_response(wp, res ? ERIC_RESPONSE_ERROR : ERIC_RESPONSE_OK,
		 _("Power switch %s."), res ? _("failed") : _("performed"));

 showstate:
    if (ps1_logged_in) {
	display_power_state(wp, fv_port0->val.s, fv_dev0->val.s,
			    POWER_STATE_SERIAL1, PP_POWER_PORT_ID_SERIAL_1);
    }
    if (ps2_logged_in) {
	display_power_state(wp, fv_port1->val.s, fv_dev1->val.s,			
			    POWER_STATE_SERIAL2, PP_POWER_PORT_ID_SERIAL_2);
    }

 setvars:
    websSetVar(wp, "current_power_entry_serial1", fv_port0->val.s);
    websSetVar(wp, "current_power_device_serial1", fv_dev0->val.s);	
    
    websSetVar(wp, "current_power_entry_serial2", fv_port1->val.s);
    websSetVar(wp, "current_power_device_serial2", fv_dev1->val.s);	

 finish:
    /*
     * Close power switches on request cleanup, so other
     * templates and ASPs can reuse the connections
     */
    wp->close_power_switches = 1;
    return 0;
}

/**
 * port must match the following format:
 * "Serialx - power switch name - port_name"
 */ 
static void
display_power_state(webs_t wp, const char* port_str, const char* dev_str, const char* var_id, unsigned int port_id)
{
    int port, dev;
    char var_name[128];
    
    if (!var_name) return;
        
    if (!port_str) {
        websSetVar(wp, var_name, "Unknown");
        return;
    }
    
    port = pp_strtoul_10(port_str, 0, NULL);
    dev = pp_strtoul_10(dev_str, 0, NULL);   

#ifdef OEM_TANGTOP
    int port_states, port_cnt, i, use_delays;
    u_char  *current_tmp;
    u_short *port_tmp;
    u_short *port_delays = malloc(sizeof(u_short) * 16);
    char var_value[128];
  
    (void)var_id; /* avoid unused parameter warning */

    port_states = pp_power_get_state(0, dev, port_id);

    if ( port_states < 0 ) {
	websSetVar(wp, "_ps_err_msg_serial1", _("No Response from Power Switch!"));
	return;
    }

    current_tmp = pp_power_get_current(dev, PP_POWER_PORT_ID_SERIAL_1);
    
    if ( current_tmp ) {
	websSetVar(wp, "_ps_power_current_value", current_tmp);
    } else {
	websSetVar(wp, "_ps_power_current_value", "Unavailable");
    }

    free(current_tmp);

    use_delays = pp_power_get_delay_enable(dev, PP_POWER_PORT_ID_SERIAL_1);

    if ( use_delays ) {
	pp_log("DelayTime on %d\n", use_delays);
	websSetVar(wp, "_ps_power_use_power_on_delays", "on");
    } else {
	pp_log("DelayTime off\n");
	websSetVar(wp, "_ps_power_use_power_on_delays", "off");
    }

    port_cnt = pp_power_get_count(dev, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1);

    port_tmp = pp_power_get_port_delays(0, dev, PP_POWER_PORT_ID_SERIAL_1);
    
    if ( port_tmp ) memcpy(port_delays, port_tmp, 8 * sizeof(u_short));
    
    if ( port_cnt > 8 ) {
	port_tmp = pp_power_get_port_delays(1, dev, PP_POWER_PORT_ID_SERIAL_1);
	if ( port_tmp ) {
	    memcpy(port_delays + 8, port_tmp, 8 * sizeof(u_short));
	} 
    } 
    
    pp_log("Port Status %x\n", port_states);
    for ( i = 0; i < port_cnt; i++) {
	if ( port_states & ( 1 << i ) ) {
	    form_var_vec_name("_ps_ipwr_port_values", i, var_name, sizeof(var_name));		
	    websSetVar(wp, var_name, "on");
	}
	form_var_vec_name("_ps_ipwr_port_delay", i, var_name, sizeof(var_name));		
	snprintf(var_value, sizeof(var_value), "%d", port_delays[i]);
	websSetVar(wp, var_name, var_value);
    }
    
#else /* !OEM_TANGTOP */
    /* get state of selected power port */
    switch (pp_power_get_state(port, dev, port_id)) {	
      case 0:
	  websSetVar(wp, var_id, "Off");
	  break;
      case 1:
	  websSetVar(wp, var_id, "On");
	  break;
      case -1:
      default:
	  websSetVar(wp, var_id, "Unknown");
    }
#endif /* !OEM_TANGTOP */
}

static int
pc_direct_set_vars_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char ** argv UNUSED)
{
    const char * current_power_entry;
    int ps1_logged_in, ps2_logged_in;

    /* open power switch on serial port 1 */
    ps1_logged_in = PP_SUCCED(power_control_common_power_switch_login(wp, PP_POWER_PORT_ID_SERIAL_1));
    
    /* open power switch on serial port 2 */
    ps2_logged_in = PP_SUCCED(power_control_common_power_switch_login(wp, PP_POWER_PORT_ID_SERIAL_2));

    current_power_entry = websGetVar(wp, "current_power_entry_serial1", NULL);
    if (!current_power_entry) {
	websSetVar(wp, "current_power_entry_serial1", "0");
	if (ps1_logged_in) display_power_state(wp, "0", "0", POWER_STATE_SERIAL1, PP_POWER_PORT_ID_SERIAL_1);
    }
    current_power_entry = websGetVar(wp, "current_power_entry_serial2", NULL);
    if (!current_power_entry) {
	websSetVar(wp, "current_power_entry_serial2", "0");
	if (ps2_logged_in) display_power_state(wp, "0", "0", POWER_STATE_SERIAL2, PP_POWER_PORT_ID_SERIAL_2);
    }
    
    /*
     * Close power switches on request cleanup, so other
     * templates and ASPs can reuse the connections
     */
    wp->close_power_switches = 1;
    return 0;
}

static int
ps_is_supported_asp(int eid, webs_t wp UNUSED, int argc, char ** argv)
{
    int supported = 0;

    if (argc == 2) {
	unsigned int port_id;
	if (!strcmp(argv[1], "intern")) {
	    port_id = PP_POWER_PORT_ID_INTERNAL;
	} else if (!strcmp(argv[1], "psu_adapt")) {
	    port_id = PP_POWER_PORT_ID_PSU_ADAPT;
	} else if (!strcmp(argv[1], "serial1")) {
	    port_id = PP_POWER_PORT_ID_SERIAL_1;
	} else if (!strcmp(argv[1], "serial2")) {
	    port_id = PP_POWER_PORT_ID_SERIAL_2;
	} else {
	    goto bail;
	}
	supported = pp_power_switch_is_supported(argv[0], port_id);
    }

 bail:
    ejSetResult(eid, supported ? "1" : "0");
    return 0;
}

static int
ps_has_capability_asp(int eid, webs_t wp UNUSED, int argc, char ** argv)
{
    int has_cap = 0;

    if (argc == 2) {
	int caps;
	if (!strcmp(argv[1], "intern")) {
	    caps = pp_power_get_caps(PP_POWER_PORT_ID_INTERNAL);
	} else if (!strcmp(argv[1], "psu_adapt")) {
	    caps = pp_power_get_caps(PP_POWER_PORT_ID_PSU_ADAPT);
	} else if (!strcmp(argv[1], "serial1")) {
	    caps = pp_power_get_caps(PP_POWER_PORT_ID_SERIAL_1);
	} else if (!strcmp(argv[1], "serial2")) {
	    caps = pp_power_get_caps(PP_POWER_PORT_ID_SERIAL_2);
	} else {
	    goto bail;
	}
	if (caps == -1) goto bail;

	if (!strcmp(argv[0], "switch")) has_cap = caps & POWER_CAP_SWITCH;
	else if (!strcmp(argv[0], "state")) has_cap = caps & POWER_CAP_GET_STATE;
	else if (!strcmp(argv[0], "temp")) has_cap = caps & POWER_CAP_GET_TEMP;
	else if (!strcmp(argv[0], "switch_ext")) has_cap = caps & POWER_CAP_SWITCH_EXT;
	else if (!strcmp(argv[0], "switch_seq")) has_cap = caps & POWER_CAP_SWITCH_SEQ;
	else if (!strcmp(argv[0], "switch_simultan")) has_cap = caps & POWER_CAP_SWITCH_SIMULTAN;
	else if (!strcmp(argv[0], "switch_all")) has_cap = caps & POWER_CAP_SWITCH_ALL;
	else if (!strcmp(argv[0], "reset")) has_cap = caps & POWER_CAP_RESET;
    }

 bail:
    ejSetResult(eid, has_cap ? "1" : "0");
    return 0;
}

static int
ps_get_longname_asp(int eid, webs_t wp UNUSED, int argc, char ** argv)
{
    char *name = NULL;
    
    if (argc != 1 || ((name = pp_power_get_long_name(argv[0])) == NULL)) {
	ejSetResult(eid, _("Unknown PowerSwitch"));
	return 0;
    }
      
    ejSetResult(eid, name);
    free(name);
    
    return 0;    
}

#else /* !PP_FEAT_POWER_CTRL_ENABLE */

int power_control_direct_tmpl_init(void) { return 0; }

#endif /* !PP_FEAT_POWER_CTRL_ENABLE */
