#include <sys/ioctl.h>
#include <pp/powerswitch.h>
#include <pp/intl.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"
#include "wsIntrn.h"

/* Documentation 
   Summary:	Configurates the power switch, in fact, set device and port names and power on delays for each power port
   Action: 	
*/
#ifdef PP_FEAT_POWER_CTRL_ENABLE

FV_SPEC = {
    {
	id:		FV_ID_PS_IPWR_DEV_ADDR,
	cfgkey:		"ps.type.ipwr.dev_addr",
	elemkey:        "dev_addr"
    },
    {
	id:		FV_ID_PS_IPWR_DEV_NAME,
	cfgkey:		"ps.type.ipwr.dev_name",
    },
    {
	id:		FV_ID_PS_IPWR_PORT_NAME,
	cfgkey:		"ps.type.ipwr.port[%I]",
	elemkey:	"name",
    },
    {
	id:		FV_ID_PS_IPWR_PORT_DELAY,
	cfgkey:		"ps.type.ipwr.port[%I]",
	elemkey:	"delay",
    },
    {
        /* this has to be the last entry to store correct number of ports! */
        id:             FV_ID_PS_IPWR_PORT_COUNT,
        cfgkey:         "ps.type.ipwr.port._s_"
    },
};

static int post_validate_hook(webs_t wp, form_handler_t * fh);
static int power_device_cfg_set_vars_asp(int eid, webs_t wp, int argc, char ** argv);

int
power_device_cfg_tmpl_init(void)
{
    form_handler_t * fh;

    /* register ASPs */
    websAspDefine("powerDeviceCfgSetVars", power_device_cfg_set_vars_asp);

    fh = CREATE_FH_INSTANCE(TEMPLATE_POWER_DEVICE_CFG, ACL_OBJ_POWER_SWITCH);

    fh->post_validate_hook = post_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
post_validate_hook(webs_t wp, form_handler_t * fh)
{
    char name_buf_1[64];
    unsigned short delay_buf_1[8];

    char name_buf_2[64];
    unsigned short delay_buf_2[8];
    
    int i, j, serial_1_open = 0, dev_id, port_cnt;
    
    const char *val_const = websGetVar(wp, "_device_id", "0");

    if ( sscanf(val_const, "%d", &dev_id) < 1 ) dev_id = 0;
    websSetVar(wp, "_device_id", val_const ? val_const : "0");
    
    serial_1_open = pp_power_login(PP_POWER_PORT_ID_SERIAL_1);

    if ( serial_1_open != 0 ) {
	pp_log("Opening Power Switch failed");
	return 0;
    }

    port_cnt = fh->fv[FV_ID_PS_IPWR_PORT_NAME].val_cnt;
 
    memset(name_buf_1, 0, sizeof(name_buf_1));

    for (i = 0; i < 8; i++) {
	u_int port_delay = 0;

	if (fh->fv[FV_ID_PS_IPWR_PORT_DELAY].val.m[i]) {
	    port_delay = pp_strtoul_10(fh->fv[FV_ID_PS_IPWR_PORT_DELAY].val.m[i], 1, NULL);
	}
	
	if ( port_delay > 9999 ) port_delay = 9999;
	if ( port_delay < 1 ) port_delay = 1;

	for ( j = 0; j < i; j++ ) {
	    if ( port_delay == delay_buf_1[j] ) {
		set_response(wp, ERIC_RESPONSE_ERROR, _("Delay times must all be different!"));
		goto error;
	    }
	}

	delay_buf_1[i] = (unsigned short) port_delay;
	
	if ( fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i] ) {
	    memcpy(&name_buf_1[i*8], fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i],
		   strlen(fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i]));
	}
    }
    
    if ( port_cnt > 8 ) {
	memset(name_buf_2, 0, sizeof(name_buf_2));
	for (i = 0; i < 8; i++) {
	    u_int port_delay = 0;

	    if (i+8 < port_cnt && fh->fv[FV_ID_PS_IPWR_PORT_DELAY].val.m[i+8]) {
		port_delay = pp_strtoul_10(fh->fv[FV_ID_PS_IPWR_PORT_DELAY].val.m[i+8], 1, NULL);
	    }
	    
	    if ( port_delay > 9999 ) port_delay = 9999;
	    if ( port_delay < 1 ) port_delay = 1;
	  
	    for ( j = 0; j < 8; j++ ) {
		if ( port_delay == delay_buf_1[j] ) {		
		    set_response(wp, ERIC_RESPONSE_ERROR, _("Delay times must all be different!"));
		    goto error;
		}
	    }

	    for ( j = 0; j < i; j++ ) {
		if ( port_delay == delay_buf_2[j] ) {
		    set_response(wp, ERIC_RESPONSE_ERROR, _("Delay times must all be different!"));
		    goto error;
		}
	    }
	  
	    delay_buf_2[i] = (unsigned short) port_delay;

	    if ( i+8 < port_cnt && fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i+8] ) {	    
		memcpy(&name_buf_2[i*8], fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i+8],
		       strlen(fh->fv[FV_ID_PS_IPWR_PORT_NAME].val.m[i+8]));
	    }
	}
    }
    
    if ( pp_power_set_name(0, dev_id, PWR_OBJ_DEVICE, PP_POWER_PORT_ID_SERIAL_1,
			   fh->fv[FV_ID_PS_IPWR_DEV_NAME].val.s) ) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Error while setting IP-Power device name!"));
	goto error;
    }
    if ( pp_power_set_port_delays(0, dev_id, PP_POWER_PORT_ID_SERIAL_1, delay_buf_1) ) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Error while setting IP-Power device name!"));
	goto error;
    }
    if ( pp_power_set_name(0, dev_id, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1, name_buf_1) ) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Error while setting IP-Power device name!"));
	goto error;
    }

    if ( port_cnt > 8 ) {
	if ( pp_power_set_port_delays(1, dev_id, PP_POWER_PORT_ID_SERIAL_1, delay_buf_2) ) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Error while setting IP-Power device name!"));
	    goto error;
	}
        if ( pp_power_set_name(1, dev_id, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1, name_buf_2) ) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Error while setting IP-Power device name!"));
	    goto error;
	}
    }

    pp_power_logout(PP_POWER_PORT_ID_SERIAL_1);
    
    return 0;

 error:
    websSetVar(wp, "_error_flag", "1");  
    return 0;    
}

static int
power_device_cfg_set_vars_asp(int eid UNUSED, webs_t wp, int argc UNUSED, char ** argv UNUSED)
{
    char var_name[128];
    char var_value[128];
    unsigned short *port_tmp;
    unsigned short *port_delays = malloc(sizeof(unsigned short) * 16);
    int i, serial_1_open, dev_id, ver, cnt = 0, port_cnt;
    const char * val_const = websGetVar(wp, "_device_id", "0");
    char * val;

    sscanf(val_const, "%d", &dev_id);
    form_handler_t * fh = lookup_form_handler(TEMPLATE_POWER_DEVICE_CFG);
    
    serial_1_open = pp_power_login(PP_POWER_PORT_ID_SERIAL_1);
    
    memset(port_delays, 0, sizeof(unsigned short) * 16);
    
    if (PP_FAILED(pp_power_login(PP_POWER_PORT_ID_SERIAL_1))) {
	ejSetResult(eid, _("Opening Power Switch failed"));
	pp_log("%s(): Opening Power Switch failed", ___F);
	return 0;
    }
    
    ver = pp_power_get_sw_version(dev_id, PP_POWER_PORT_ID_SERIAL_1);
    
    snprintf(var_value, sizeof(var_value), "V %d.%d", (ver & 0xF0) >> 4, (ver & 0x0F));
    websSetVar(wp, "_ps_ipwr_sw_version", var_value);
    
    cnt = pp_power_get_count(dev_id, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1);
    
    memset(port_delays, 0, sizeof(port_delays));
    
    port_cnt = pp_power_get_count(dev_id, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1);
    
    form_var_sz_name(fh->fv_tmpl[FV_ID_PS_IPWR_PORT_NAME].fvname, 
                     var_name, sizeof(var_name));
    snprintf(var_value, sizeof(var_value), "%u", port_cnt);
    
    websSetVar(wp, var_name, var_value);
    
    form_var_sz_name(fh->fv_tmpl[FV_ID_PS_IPWR_PORT_DELAY].fvname, 
                     var_name, sizeof(var_name));
    snprintf(var_value, sizeof(var_value), "%u", port_cnt);
    
    websSetVar(wp, var_name, var_value);
    
    snprintf(var_name, sizeof(var_name), "%s", fh->fv_tmpl[FV_ID_PS_IPWR_DEV_NAME].fvname);
    val = pp_power_get_name(0, dev_id, PWR_OBJ_DEVICE, PP_POWER_PORT_ID_SERIAL_1);
    websSetVar(wp, var_name, val);
    free(val);
    
    snprintf(var_name, sizeof(var_name), "%s", fh->fv_tmpl[FV_ID_PS_IPWR_DEV_ADDR].fvname);
    val = pp_power_get_name(0, dev_id, PWR_OBJ_DEVICE_ADDR, PP_POWER_PORT_ID_SERIAL_1);
    websSetVar(wp, var_name, val);
    free(val);

    port_tmp = pp_power_get_port_delays(0, dev_id, PP_POWER_PORT_ID_SERIAL_1);

    if ( port_tmp ) memcpy(port_delays, port_tmp, 8 * sizeof(unsigned short));
    
    if (port_cnt > 8) {
	port_tmp = pp_power_get_port_delays(1, dev_id, PP_POWER_PORT_ID_SERIAL_1);
	
	if ( port_tmp ) {
	    memcpy(port_delays + 8, port_tmp, 8 * sizeof(unsigned short));
	}
    }

    for (i = 0; i < port_cnt; i++) {
        form_var_vec_name(fh->fv_tmpl[FV_ID_PS_IPWR_PORT_NAME].fvname, i,
                          var_name, sizeof(var_name));
	val = pp_power_get_name(i, dev_id, PWR_OBJ_PORT, PP_POWER_PORT_ID_SERIAL_1);
	websSetVar(wp, var_name, val);
	free(val);

        form_var_vec_name(fh->fv_tmpl[FV_ID_PS_IPWR_PORT_DELAY].fvname, i,
                          var_name, sizeof(var_name));
	snprintf(var_value, sizeof(var_value), "%d", port_delays[i]);
	websSetVar(wp, var_name, var_value);
    }
    
    pp_power_logout(PP_POWER_PORT_ID_SERIAL_1);
    
    return 0;
}

#else /* !PP_FEAT_POWER_CTRL_ENABLE */

int power_device_cfg_tmpl_init(void) { return 0; }

#endif /* !PP_FEAT_POWER_CTRL_ENABLE */
