/* system includes */
/* firmware includes */
#include <pp/ipmi.h>
#include <pp/hal_rpc.h>

/* local includes */
#include "webs.h"
#include "templates.h"
#include "eric_forms.h"
#include "eric_util.h"
#include "tmpl_rpc_common.h"

FV_SPEC = {
    {
	id:		FV_ID_RPC_SETUP_NAME,
	cfgkey:		"bmc.rpc.outlet[%O].name"
    },
    {
        id:            FV_ID_RPC_SETUP_LNC,
        cfgkey:        "bmc.sensors[%I]",
        elemkey:       "thresh.low_noncrit"
    },
    {
        id:            FV_ID_RPC_SETUP_LC,
        cfgkey:        "bmc.sensors[%I]",
        elemkey:       "thresh.low_crit"
    },
    {
        id:            FV_ID_RPC_SETUP_UNC,
        cfgkey:        "bmc.sensors[%I]",
        elemkey:       "thresh.up_noncrit"
    },
    {
        id:            FV_ID_RPC_SETUP_UC,
        cfgkey:        "bmc.sensors[%I]",
        elemkey:       "thresh.up_crit"
    },
};

/* ASPs */
static int rpc_init_outlet_setup_asp(int eid, webs_t wp, 
                                     int argc, char **argv);

/* hooks */
static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int post_validate_hook(webs_t wp, form_handler_t * fh);

/* FIXME: would someone please formulate some eloquent error messages? */
static const char *rpc_outset_err_str[] = { 
    /* 0*/ N_("The lower non-critical threshold of sensor %s "
              "may not be smaller than the lower critical threshold."),
    /* 1*/ N_("The upper non-critical threshold of sensor %s "
              "may not be smaller than the lower non-critical threshold."),
    /* 2*/ N_("The upper critical threshold of sensor %s "
              "may not be smaller than the upper non-critical threshold."),
    /* 3*/ N_("Thresholds of sensor %s need to be numerical values."),
    /* 4*/ N_("Upper critical threshold of sensor %s is too high and exceedes "
              "maximum total current of %.1f Amps for outlet group."),
    /* 5*/ N_("Upper critical threshold of sensor %s is too high and exceedes "
              "maximum total current of %.1f Amps for unit."),
    };

/* TODO: find a more generic way to specify thresholds! */
static int has_lower_th[] = { 0, 1 };
static int has_upper_th[] = { 1, 1 };

int rpc_outlet_setup_tmpl_init(void)
{
    form_handler_t * fh;

    rpc_common_tmpl_init();
    
    /* register ASPs */
    websAspDefine("rpcInitOutletSetup", rpc_init_outlet_setup_asp);
    
    fh = CREATE_FH_INSTANCE(TEMPLATE_RPC_OUTLET_SETUP, ACL_OBJ_POWER_SWITCH);

    fh->pre_validate_hook = pre_validate_hook;
    fh->post_validate_hook = post_validate_hook;
    
    REGISTER_FH_INSTANCE_AND_RETURN(fh);
    
    return PP_SUC;
}

static int pre_validate_hook(webs_t wp, form_handler_t * fh) {
    if (!form_button_clicked(wp, "action_apply")) {
	/* form not applied -> neighter validate nor save */
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
        
        /* FIXME: why do I have to delete form vars here?
                  without, values are not updated on target outlet switch */
        delete_form_vars(wp);
        
        return PP_SUC;
    }
    
    /* do not validate, even more magic follows! */
    fh->fv[FV_ID_RPC_SETUP_LNC].flags |= 
        FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    fh->fv[FV_ID_RPC_SETUP_LC].flags |= 
        FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    fh->fv[FV_ID_RPC_SETUP_UNC].flags |= 
        FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    fh->fv[FV_ID_RPC_SETUP_UC].flags |= 
        FV_FLAG_SKIP_VALIDATE | FV_FLAG_DONT_SAVE;
    
    return PP_SUC;
}

static int get_real_sensor_thresh(double *real_thresh,
                                  webs_t wp, int sensornum,
                                  const char *sensor_name, int fv_idx) {
    const char *val;
    char var_buf[32];
    int ret = PP_ERR;
    double dval;
    
    snprintf(var_buf, sizeof(var_buf), "real_%d_rpcoutset%c%d",
             fv_idx, fv_vec_idx_deli, sensornum);
    if(NULL != (val = websGetVar(wp, var_buf, NULL))) {
        char *endptr;
        
        dval = strtod(val, &endptr);
        if(endptr == val) {
            set_response(wp, ERIC_RESPONSE_ERROR,
                         _(rpc_outset_err_str[3]), sensor_name);
        } else {
            ret = PP_SUC;
            *real_thresh = dval;
        }
    }
    
    return ret;
}

#define CONVERT_REAL_TO_RAW(__val__) \
    pp_ipmi_sdr_convert_sensor_value_to_raw(entry->data.full.m, \
                                            entry->data.full.b, \
                                            entry->data.full.b_exp, \
                                            entry->data.full.r_exp, \
                                            entry->data.full.analog, \
					    __val__);

static int post_validate_hook(webs_t wp, form_handler_t * fh UNUSED) {
    int outlet = wp->target_outlet - 1;
    int power_branch = rpc_get_power_branch_for_outlet(outlet);
    int sensors[] = OUTLET_SETUP_SENSORS(outlet, power_branch);
    unsigned int sensors_sz = sizeof(sensors) / sizeof(int);
    unsigned int u;
    int ret = PP_SUC;
    double lc, lnc, unc, uc;
    uint8_t lc_raw, lnc_raw, unc_raw, uc_raw;
    pp_ipmi_parameter_t params;
#if !defined(USE_SDR_SENSOR_NAMES)
    const char *sensor_strs[] = OUTLET_SETUP_SENSOR_STRINGS;
#endif /* !USE_SDR_SENSOR_NAMES */
    
    memset(&params, 0, sizeof(params));
    params.data.sensor_raw_threshold_list = 
        vector_new2(NULL, 1, sizeof(pp_ipmi_sensor_set_raw_th_param_t),
                    NULL);
    
    for(u = 0; u < sensors_sz; ++u) {
        unsigned int sensornum = sensors[u];
        pp_ipmi_sdr_list_entry_t *entry = get_sdr_entry_for_sensor(sensornum);
        const char* sensor_name;
        char sensor_tmp[32];
        int has_lower = has_lower_th[u];
        int has_upper = has_upper_th[u];
        pp_ipmi_sensor_set_raw_th_param_t raw_th;
        
        if(!has_lower && !has_upper) {
            /* having no thresholds at all saves a lot of work! */
            continue;
        }
        
        snprintf(sensor_tmp, sizeof(sensor_tmp), 
                 "sensor_%d_enabled", sensornum);
        if(!websGetVar(wp, sensor_tmp, NULL)) {
            /* seems we did not set sensor sensornum */
            continue;
        }
        
        /* if we want to convert something, we need the sdr entry */
        assert(entry);
        
#if defined(USE_SDR_SENSOR_NAMES)
        if(entry && entry->sdr_name.buf && *entry->sdr_name.buf) {
            sensor_name = entry->sdr_name.buf;
        } else {
            snprintf(sensor_tmp, sizeof(sensor_tmp), "%s (%d)",
                     _("Unknown"), sensornum);
            sensor_name = sensor_tmp;
        }
#else /* !USE_SDR_SENSOR_NAMES */
        sensor_name = _(sensor_strs[u]);
#endif /* !USE_SDR_SENSOR_NAMES */
        
        /* convert to floats */
        if((has_lower &&
            get_real_sensor_thresh(&lc, wp, sensornum, sensor_name,
                                   FV_ID_RPC_SETUP_LC) == PP_ERR) ||
           (has_lower &&
            get_real_sensor_thresh(&lnc, wp, sensornum, sensor_name,
                                   FV_ID_RPC_SETUP_LNC) == PP_ERR) ||
           (has_upper &&
            get_real_sensor_thresh(&unc, wp, sensornum, sensor_name,
                                   FV_ID_RPC_SETUP_UNC) == PP_ERR) ||
           (has_upper &&
            get_real_sensor_thresh(&uc, wp, sensornum, sensor_name,
                                   FV_ID_RPC_SETUP_UC) == PP_ERR)) {
            goto error;
        }
        
        /* the easy one, check threshold constraints lc <= lnc < unc <= uc */
        if(has_lower && (lc > lnc)) {
            set_response(wp, ERIC_RESPONSE_ERROR, 
                         _(rpc_outset_err_str[0]), sensor_name);
            ret = PP_ERR;
            goto error;
        }
        if(has_lower && has_upper && (lnc >= unc)) {
            set_response(wp, ERIC_RESPONSE_ERROR, 
                         _(rpc_outset_err_str[1]), sensor_name);
            ret = PP_ERR;
            goto error;
        }
        if(has_upper && (unc > uc)) {
            set_response(wp, ERIC_RESPONSE_ERROR, 
                         _(rpc_outset_err_str[2]), sensor_name);
            ret = PP_ERR;
            goto error;
        }
        
        /* now do the value conversion */
        if(has_lower) {
            lc_raw = CONVERT_REAL_TO_RAW(lc);
            lnc_raw = CONVERT_REAL_TO_RAW(lnc);
        } else {
            lc_raw = lnc_raw = 0;
        }
        if(has_upper) {
            unc_raw = CONVERT_REAL_TO_RAW(unc);
            uc_raw = CONVERT_REAL_TO_RAW(uc);
        } else {
            uc_raw = unc_raw = 0;
        }

        /* check board constraints */
        if(IS_RPC_RECEPTACLE_SENSOR_RMS_CURRENT(sensornum)) {
            /* assert sum of rms current critical thresholds per board does not
               exceed hardware given limit of 16 Amps */
            
            int curr_crit_total, curr_crit_board, i;
            int num_outlets = pp_hal_rpc_get_no_outlets();
            pp_ipmi_return_t ipmi_ret;
            pp_ipmi_parameter_t params2;
            vector_t *get_list;
            int relais_board = rpc_get_relais_board_for_outlet(outlet);
    
            assert(has_upper);
            
            /* init */
            curr_crit_total = curr_crit_board = (int)uc_raw;
            memset(&ipmi_ret, 0, sizeof(ipmi_ret));
            memset(&params2, 0, sizeof(params));
            
            params2.data.sensor_get_by_id_list = 
                vector_new2(NULL, sensors_sz, sizeof(unsigned int), NULL);
            get_list = params2.data.sensor_get_by_id_list;
    
            for(i = 0; i < num_outlets; ++i) {
                int curr_sens = RPC_RECEPTACLE_SENSOR_RMS_CURRENT(i);
                
                vector_add2(get_list, &curr_sens);
            }
            
            if (pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
                                     PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID,
                                     &params2, &ipmi_ret,
                                     NULL, NULL) != 0) {
                RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SENSOR,  PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID failed");
                
                /* TODO: error or not? */
            } else {
                vector_t *raw_thresholds = ipmi_ret.data.sensor_get_thresh_list;
                unsigned int raw_thresholds_sz = vector_size(raw_thresholds);

                if(!raw_thresholds || raw_thresholds_sz == 0) {
                    RPC_DEBUG_IPMI("sensor get raw thresholds returned no reading");
                
                    /* TODO: error or not? */
                } else {
                    /* we got raw thresholds */

                    unsigned int v = 0;
                    
                    for(v = 0; v < raw_thresholds_sz; ++v) {
                        pp_ipmi_raw_thresholds_list_entry_t *thresholds;
                        int thresh_outlet;
                        
                        thresholds = vector_get(raw_thresholds, v);
                            
                        /* paranoia */
                        assert(thresholds);
                        
                        thresh_outlet = RPC_RECEPTACLE(thresholds->id);
                        
                        if(thresh_outlet == outlet) {
                            /* not for current outlet */
                            continue;
                        }
                        
                        if(thresholds->threshold_present.bits.upper_critical) {
                            if(rpc_get_relais_board_for_outlet(thresh_outlet)
                                   == relais_board) {
                                curr_crit_board +=
                                    (int)thresholds->upper_critical;
                            }
                            curr_crit_total += (int)thresholds->upper_critical;
                        }
                        // TODO: else error?
                    }
                }
            }
            
            /* clean up */
            vector_delete(get_list);
            pp_ipmi_cleanup_ret(&ipmi_ret);

            /* check if new threshold sum per unit <= curr_max */
            if(curr_crit_total > pp_hal_rpc_get_curr_max()) {
                set_response(wp, ERIC_RESPONSE_ERROR, 
                             _(rpc_outset_err_str[5]), sensor_name,
                             pp_hal_rpc_get_curr_max() / (double)10.0);
                ret = PP_ERR;
                goto error;
            }

            /* check if new threshold sum per board <= curr_max_board */
            if(curr_crit_board > pp_hal_rpc_get_curr_max_board()) {
                set_response(wp, ERIC_RESPONSE_ERROR, 
                             _(rpc_outset_err_str[4]), sensor_name,
                             pp_hal_rpc_get_curr_max_board() / (double)10.0);
                ret = PP_ERR;
                goto error;
            }
       }
        
        /* store values */
        raw_th.sensor_id = sensornum;
        if(has_lower) {
            raw_th.threshold_type = PP_IPMI_SENSOR_THRESHOLD_LOWER_CRIT;
            raw_th.raw_value = lc_raw;
            vector_add2(params.data.sensor_raw_threshold_list, &raw_th);

            raw_th.threshold_type = PP_IPMI_SENSOR_THRESHOLD_LOWER_NON_CRIT;
            raw_th.raw_value = lnc_raw;
            vector_add2(params.data.sensor_raw_threshold_list, &raw_th);
        }
        
        if(has_upper) {
            raw_th.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_NON_CRIT;
            raw_th.raw_value = unc_raw;
            vector_add2(params.data.sensor_raw_threshold_list, &raw_th);

            raw_th.threshold_type = PP_IPMI_SENSOR_THRESHOLD_UPPER_CRIT;
            raw_th.raw_value = uc_raw;
            vector_add2(params.data.sensor_raw_threshold_list, &raw_th);
        }
    }

    if(vector_size(params.data.sensor_raw_threshold_list)) {
        if(pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
                                PP_IPMI_SENSOR_SUBCMD_SET_RAW_TH,
                                &params, NULL, NULL, NULL) != 0) {
            /* TODO: error checking? */
            RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SENSOR,  PP_IPMI_SENSOR_SUBCMD_SET_RAW_TH failed");
        }
    }
        
 error:
    vector_delete(params.data.sensor_raw_threshold_list);
    return ret;
}

#define INIT_THRESH(__type__, __fvid__) \
    init_real_sensor_thresh(wp, fh, sensornum, entry,  __fvid__, \
                            thresholds->threshold_present.bits.__type__, \
                            thresholds->__type__);

static void init_real_sensor_thresh(webs_t wp, form_handler_t *fh, 
                                int sensornum, pp_ipmi_sdr_list_entry_t *entry,
                                int fv_idx, int present,
                                unsigned char raw_thresh) {
    char cfgkey[64], var_buf[32];
    
    assert(fh);
   
    snprintf(cfgkey, sizeof(cfgkey), "%s.%s",
             fh->fv_tmpl[fv_idx].cfgkey, fh->fv_tmpl[fv_idx].elemkey);
    snprintf(var_buf, sizeof(var_buf), "real_%d_rpcoutset%c%d",
             fv_idx, fv_vec_idx_deli, sensornum);
    if(!entry || !present) {
        /* no entry or requested threshold not set */
        websSetVar(wp, var_buf, "");
    } else {
        double real_thresh = pp_ipmi_sdr_convert_sensor_reading_core(
                                                entry->data.full.m,
                                                entry->data.full.b,
                                                entry->data.full.b_exp,
                                                entry->data.full.r_exp,
                                                entry->data.full.analog,
                                                entry->data.full.linearization,
                                                raw_thresh);
        char val_buf[20];
        int neg_exp = 0;
        
        /* calculate format string
           negative exponents generate decimals */
        if(entry->data.full.b_exp < 0) {
            neg_exp -= entry->data.full.b_exp;
        }
        if(entry->data.full.r_exp < 0) {
            neg_exp -= entry->data.full.r_exp;
        }
        if(neg_exp > 9) {
            neg_exp = 9;
        }
        assert(neg_exp >= 0);
        
        snprintf(val_buf, sizeof(val_buf), "%.*f", neg_exp, real_thresh);
        websSetVar(wp, var_buf, val_buf);
    }
}

static int rpc_init_outlet_setup_asp(int eid UNUSED, webs_t wp, 
                                     int argc, char **argv) {
    int ret = -1;
    int outlet, num_outlets = pp_hal_rpc_get_no_outlets();
    
    /* check if argv is valid outlet number */
    if(argc == 1 &&
       (outlet = pp_strtol_10(argv[0], -1, NULL) - 1) >= 0 &&
       outlet >= 0 && outlet < num_outlets) {
        int power_branch = rpc_get_power_branch_for_outlet(outlet);
        int sensors[] = OUTLET_SETUP_SENSORS(outlet, power_branch);
#if !defined(USE_SDR_SENSOR_NAMES)
        const char *sensor_strs[] = OUTLET_SETUP_SENSOR_STRINGS;
#endif /* !USE_SDR_SENSOR_NAMES */
        unsigned int sensors_sz = sizeof(sensors) / sizeof(int);
        unsigned int u;
        char var_buf[32], val_buf[20];
        form_handler_t *fh = lookup_form_handler(TEMPLATE_RPC_OUTLET_SETUP);
        pp_ipmi_return_t ipmi_ret;
        pp_ipmi_parameter_t params;
        vector_t *sdr_entries = vector_new(NULL, sensors_sz, NULL);
        vector_t *get_list;
        const char *bin_str_map[2] = { "0", "1" };
        
        assert((sizeof(has_lower_th) / sizeof(int)) == sensors_sz);
        assert((sizeof(has_upper_th) / sizeof(int)) == sensors_sz);
        
        /* init */
        memset(&ipmi_ret, 0, sizeof(ipmi_ret));
        memset(&params, 0, sizeof(params));
        
        params.data.sensor_get_by_id_list = 
            vector_new2(NULL, sensors_sz, sizeof(unsigned int), NULL);
        get_list = params.data.sensor_get_by_id_list;
        
        snprintf(val_buf, sizeof(val_buf), "%u", sensors_sz);
        websSetVar(wp, "sensors_sz", val_buf);
        
        /* build vector for sensor reading */
        for(u = 0; u < sensors_sz; ++u) {
            unsigned int sensornum = sensors[u];
            pp_ipmi_sdr_list_entry_t *entry;
            char sensor_name[32], sensor_has_lower[32], sensor_has_upper[32];
            
            form_var_vec_name("sensor_id", u, var_buf, sizeof(var_buf));
            snprintf(val_buf, sizeof(val_buf), "%d", sensornum);
            websSetVar(wp, var_buf, val_buf);
            
            form_var_vec_name("sensor_name", u,
                              sensor_name, sizeof(sensor_name));
            form_var_vec_name("sensor_has_lower", u, 
                              sensor_has_lower, sizeof(sensor_has_lower));
            form_var_vec_name("sensor_has_upper", u,
                              sensor_has_upper, sizeof(sensor_has_upper));
            
            if(NULL != (entry = get_sdr_entry_for_sensor(sensornum))) {
                /* valid sdr entry */

#if defined(USE_SDR_SENSOR_NAMES)
                websSetVar(wp, sensor_name, 
                           entry->sdr_name.buf ? : _("unknown"));
#else /* !USE_SDR_SENSOR_NAMES */
                websSetVar(wp, sensor_name, _(sensor_strs[u]));
#endif /* !USE_SDR_SENSOR_NAMES */
                websSetVar(wp, sensor_has_lower, bin_str_map[has_lower_th[u]]);
                websSetVar(wp, sensor_has_upper, bin_str_map[has_upper_th[u]]);

                vector_add2(get_list, &sensornum);
                vector_add(sdr_entries, entry);
            } else {
                /* invalid sensor, no sdr entry */
                vector_add(sdr_entries, NULL);

                websSetVar(wp, sensor_name, _("not initialized"));
                websSetVar(wp, sensor_has_lower, "0");
                websSetVar(wp, sensor_has_upper, "0");
            }
        }
            
        /* do not disturb bmc if we have nothing to query ;-) */
        if(vector_size(get_list)) {
            /* get raw sensor reading */
            
            if (pp_ipmi_send_command(PP_IPMI_CMD_SENSOR,
                                     PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID,
                                     &params, &ipmi_ret,
                                     NULL, NULL) != 0) {
                RPC_DEBUG_IPMI("pp_ipmi_send_command PP_IPMI_CMD_SENSOR,  PP_IPMI_SENSOR_SUBCMD_GET_THRESH_BY_ID failed");
                
                /* TODO: leave webs vars emtpy? */
            } else {
                vector_t *raw_thresholds = ipmi_ret.data.sensor_get_thresh_list;

                if(!raw_thresholds || !vector_size(raw_thresholds)) {
                    RPC_DEBUG_IPMI("sensor get raw thresholds returned no reading");
                
                    /* TODO: leave webs vars emtpy? */
                } else {
                    /* we got raw thresholds */

                    unsigned int val_count = 0;
                    
                    for(u = 0; u < sensors_sz; ++u) {
                        /* check if we requested thresholds for sensor */
                        unsigned int sensornum = sensors[u];
                        pp_ipmi_sdr_list_entry_t *entry =
                            vector_get(sdr_entries, u);
                        
                        if(entry) {
                            /* now get thresholds from sensor */
                            pp_ipmi_raw_thresholds_list_entry_t *thresholds;
                            
                            assert(val_count < vector_size(raw_thresholds));
                            thresholds = vector_get(raw_thresholds,
                                                    val_count++);
                            
                            /* paranoia */
                            assert(thresholds);
                            assert(thresholds->id == entry->sdr_id);
                            
                            INIT_THRESH(lower_critical, FV_ID_RPC_SETUP_LC);
                            INIT_THRESH(lower_non_critical,
                                        FV_ID_RPC_SETUP_LNC);
                            INIT_THRESH(upper_non_critical,
                                        FV_ID_RPC_SETUP_UNC);
                            INIT_THRESH(upper_critical, FV_ID_RPC_SETUP_UC);
                        }
                    }
                }
            }
        }
        
        ret = 0;

        /* clean up */
        vector_delete(sdr_entries);
        vector_delete(get_list);
        pp_ipmi_cleanup_ret(&ipmi_ret);
    } else {
        /* invalid outlet number */
        if(argc != 1) {
            pp_log("invalid number of arguments!\n");
        } else {
            pp_log("outlet number '%s' invalid!\n", argv[0]);
        }
        abort();
    }
    
    return ret;
}
