#include <liberic_notify.h>
#include <liberic_net.h>
#include <liberic_config.h>
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"
#include "eric_form_vars.h"

FV_SPEC = {
    {
	id:		FV_ID_SEC_GACL_ENABLED,
	cfgkey:		"security.group_acl.enabled"
    },
    {
	id:		FV_ID_SEC_GACL_DEFAULT_POLICY,
	cfgkey:		"security.group_acl.default_policy"
    },
    {
	id:		FV_ID_SEC_GACL_START_IP,
	cfgkey:		"security.group_acl.ruleset[%I]",
        elemkey:        "start_ip"
    },
    {
	id:		FV_ID_SEC_GACL_STOP_IP,
	cfgkey:		"security.group_acl.ruleset[%I]",
        elemkey:        "stop_ip"
    },
    {
	id:		FV_ID_SEC_GACL_GID,
	cfgkey:		"security.group_acl.ruleset[%I]",
        elemkey:        "gid"
    },
    {
	id:		FV_ID_SEC_GACL_POLICY,
	cfgkey:		"security.group_acl.ruleset[%I]",
        elemkey:        "policy"
    },
    {
	id:		FV_ID_SEC_GRULE_NUMBER,
        // this is a dummy cfgkey, get it from ip_fw
	cfgkey:		"security.ip_fw.rulenr"
    },
    {
	id:		FV_ID_SEC_GSTART_IP,
	cfgkey:		"gacl_ruleset.start_ip"
    },
    {
	id:		FV_ID_SEC_GSTOP_IP,
	cfgkey:		"gacl_ruleset.stop_ip"
    },
    {
	id:		FV_ID_SEC_GGID,
	cfgkey:		"gacl_ruleset.gid"
    },
    {
	id:		FV_ID_SEC_GPOLICY,
	cfgkey:		"gacl_ruleset.policy"
    },
};

typedef enum {
    APPEND_RULE,
    INSERT_RULE,
    REPLACE_RULE,
    DELETE_RULE,
    COUNT_ACCEPT_RULES
} rule_action_t;

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int do_rule_action(webs_t wp, rule_action_t rule_action, form_var_t * form_vars);

int
sec_group_acl_tmpl_init(void)
{
    form_handler_t * fh;

#if defined (PRODUCT_KX2)
    fh = CREATE_FH_INSTANCE(TEMPLATE_SEC_GACL, ACL_OBJ_UM);
#else /* !PRODUCT_KX2 */
    fh = CREATE_FH_INSTANCE(TEMPLATE_SEC_GACL, ACL_OBJ_SECURITY);
#endif /* !PRODUCT_KX2 */

    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp,  form_handler_t * fh)
{
    int enabled = !strcmp(fh->fv[FV_ID_SEC_GACL_ENABLED].val.s, "yes");
    int do_append = form_button_clicked(wp, "action_append_gacl");
    int do_insert = form_button_clicked(wp, "action_insert_gacl");
    int do_replace = form_button_clicked(wp, "action_replace_gacl");
    int do_delete = form_button_clicked(wp, "action_delete_gacl");
    int do_apply = form_button_clicked(wp, "action_apply");
    int do_really_apply = form_button_clicked(wp, "action_really_apply");
    int ret = -1;
    size_t i;

    if (!enabled) fh_disable_validate_and_save(fh, FV_ID_SEC_GACL_ENABLED);

    if (do_delete) {
	fh->fv[FV_ID_SEC_GSTART_IP].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GSTOP_IP].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GGID].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GPOLICY].flags |= FV_FLAG_SKIP_VALIDATE;
    } else if (do_apply || do_really_apply) {
        fh->fv[FV_ID_SEC_GRULE_NUMBER].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GSTART_IP].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GSTOP_IP].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GGID].flags |= FV_FLAG_SKIP_VALIDATE;
	fh->fv[FV_ID_SEC_GPOLICY].flags |= FV_FLAG_SKIP_VALIDATE;
    } else if(do_append) {
        /* no rule number for append - simply append ;-) */
        fh->fv[FV_ID_SEC_GRULE_NUMBER].flags |= FV_FLAG_SKIP_VALIDATE;
    }
    
    /* No global validate - only our own template (manually) */
    if (do_append || do_insert || do_replace || do_delete) {
	wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;
	for (i = 0; i < fh->fv_cnt; ++i) {
	    form_var_t * fv = &fh->fv[i];
	    if (fv->flags & FV_FLAG_SKIP_VALIDATE) continue;
	    if (!validate_form_var(wp, fv)) {
		goto bail;
	    }
	}
    }

    if (do_append) {
	if (do_rule_action(wp, APPEND_RULE, fh->fv) == -1) goto bail;
    } else if (do_insert) {
	if (do_rule_action(wp, INSERT_RULE, fh->fv) == -1) goto bail;
    } else if (do_replace) {
	if (do_rule_action(wp, REPLACE_RULE, fh->fv) == -1) goto bail;
    } else if (do_delete) {
	if (do_rule_action(wp, DELETE_RULE, fh->fv) == -1) goto bail;
    } else if (enabled && do_apply
	       && !strcmp(fh->fv[FV_ID_SEC_GACL_DEFAULT_POLICY].val.s, "DROP")
	       && do_rule_action(wp, COUNT_ACCEPT_RULES, fh->fv) < 1) {
	set_response(wp, ERIC_RESPONSE_WARNING,
		     _("With this configuration you will not be able to "
		     "access the card via the LAN interface anymore! "
		       "If this is what you want press \"Really Apply\"."));
	websSetVar(wp, "_show_apply_confirm", "1");
	wp->fh_flags |= FH_FLAG_ABORT_AT_SAVE;
    }

    symDelete(wp->cgiVars, fh->fv[FV_ID_SEC_GRULE_NUMBER].fvname);
    symDelete(wp->cgiVars, fh->fv[FV_ID_SEC_GSTART_IP].fvname);
    symDelete(wp->cgiVars, fh->fv[FV_ID_SEC_GSTOP_IP].fvname);
    symDelete(wp->cgiVars, fh->fv[FV_ID_SEC_GGID].fvname);
    symDelete(wp->cgiVars, fh->fv[FV_ID_SEC_GPOLICY].fvname);
    
    ret = 0;

 bail:
    return ret;
}

static int
do_rule_action(webs_t wp, rule_action_t rule_action, form_var_t * form_vars)
{
    char * endptr;
    char buf[MAX_OPT_KEY_LEN + 1];
    const char * rule_num_str = form_vars[FV_ID_SEC_GRULE_NUMBER].val.s;
    u_int rule_num = strtoul(rule_num_str, &endptr, 10);
    u_int rule_cnt = form_vars[FV_ID_SEC_GACL_START_IP].val_cnt;
    u_int rule_cnt_max = form_vars[FV_ID_SEC_GACL_START_IP].val_cnt_max;
    u_int i, count;

    if (rule_action != APPEND_RULE &&
	rule_action != COUNT_ACCEPT_RULES &&
	(*rule_num_str == '\0' || *endptr != '\0' ||
	 rule_num < 1 || rule_num > rule_cnt)) {

	set_response(wp, ERIC_RESPONSE_ERROR,_("Invalid rule number specified."));
	return -1;
    }

    switch (rule_action) {
      case APPEND_RULE:
	  if (rule_cnt < rule_cnt_max) {
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                            rule_cnt, form_vars[FV_ID_SEC_GSTART_IP].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                            rule_cnt, form_vars[FV_ID_SEC_GSTOP_IP].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                            rule_cnt, form_vars[FV_ID_SEC_GGID].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                            rule_cnt, form_vars[FV_ID_SEC_GPOLICY].val.s);
	      ++rule_cnt;
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                           rule_cnt);
	  } else {
	      set_response(wp, ERIC_RESPONSE_ERROR,
			   _("Maximum number of rules reached."));
	      return -1;
	  }
	  break;
      case INSERT_RULE:
	  if (rule_cnt < rule_cnt_max) {
	      for (i = rule_cnt; i >= rule_num; i--) {
                  form_var_vec_name(form_vars[FV_ID_SEC_GACL_START_IP].fvname, 
                                    i - 1, buf, sizeof(buf));
                  websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                                i, websGetVar(wp, buf, ""));
                  form_var_vec_name(form_vars[FV_ID_SEC_GACL_STOP_IP].fvname, 
                                    i - 1, buf, sizeof(buf));
                  websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                                i, websGetVar(wp, buf, ""));
                  form_var_vec_name(form_vars[FV_ID_SEC_GACL_GID].fvname, 
                                    i - 1, buf, sizeof(buf));
                  websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                                i, websGetVar(wp, buf, ""));
                  form_var_vec_name(form_vars[FV_ID_SEC_GACL_POLICY].fvname, 
                                    i - 1, buf, sizeof(buf));
                  websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                                i, websGetVar(wp, buf, ""));
	      }
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                            rule_num - 1, form_vars[FV_ID_SEC_GSTART_IP].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                            rule_num - 1, form_vars[FV_ID_SEC_GSTOP_IP].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                            rule_num - 1, form_vars[FV_ID_SEC_GGID].val.s);
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                            rule_num - 1, form_vars[FV_ID_SEC_GPOLICY].val.s);
	      ++rule_cnt;
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                           rule_cnt);
              websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                           rule_cnt);
	  } else {
	      set_response(wp, ERIC_RESPONSE_ERROR,
			   _("Maximum number of rules reached."));
	      return -1;
	  }
	  break;
      case REPLACE_RULE:
          websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                        rule_num - 1, form_vars[FV_ID_SEC_GSTART_IP].val.s);
          websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                        rule_num - 1, form_vars[FV_ID_SEC_GSTOP_IP].val.s);
          websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                        rule_num - 1, form_vars[FV_ID_SEC_GGID].val.s);
          websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                        rule_num - 1, form_vars[FV_ID_SEC_GPOLICY].val.s);
	  break;
      case DELETE_RULE:
	  for (i = rule_num; i < rule_cnt; i++) {
              form_var_vec_name(form_vars[FV_ID_SEC_GACL_START_IP].fvname, 
                                i, buf, sizeof(buf));
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname,
                            i - 1, websGetVar(wp, buf, ""));
              form_var_vec_name(form_vars[FV_ID_SEC_GACL_STOP_IP].fvname, 
                                i, buf, sizeof(buf));
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname,
                            i - 1, websGetVar(wp, buf, ""));
              form_var_vec_name(form_vars[FV_ID_SEC_GACL_GID].fvname, 
                                i, buf, sizeof(buf));
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname,
                            i - 1, websGetVar(wp, buf, ""));
              form_var_vec_name(form_vars[FV_ID_SEC_GACL_POLICY].fvname, 
                                i, buf, sizeof(buf));
              websSetVecVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname,
                            i - 1, websGetVar(wp, buf, ""));
	  }
          --rule_cnt;
          websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_START_IP].fvname, rule_cnt);
          websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_STOP_IP].fvname, rule_cnt);
          websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_GID].fvname, rule_cnt);
          websSetSzVar(wp, form_vars[FV_ID_SEC_GACL_POLICY].fvname, rule_cnt);
	  break;
      case COUNT_ACCEPT_RULES:
	  count = 0;
	  for (i = 0; i < rule_cnt; i++) {
              form_var_vec_name(form_vars[FV_ID_SEC_GACL_POLICY].fvname, 
                                i, buf, sizeof(buf));
	      if (!strcmp(websGetVar(wp, buf, ""), "ACCEPT")) {
		  count++;
	      }
	  }
	  return count;
    }

    return 0;
}

