#ifdef PP_FEAT_MASS_STORAGE
#include <stdio.h>
#include <sys/mman.h>
#include <pp/usb.h>
#include <pp/intl.h>
#include <pp/um.h>
#include "tmpl_vfloppy_common.h"
#include "eric_base.h"
#include "eric_util.h"
#include "eric_validate.h"
#include "eric_forms.h"

#define VFLOPPY_HEADS                   2
#define VFLOPPY_SECTORS_PER_TRACK       18
#define VFLOPPY_TRACKS                  80
#define VFLOPPY_SECTOR_SIZE             512
#define VFLOPPY_MAX_SIZE                (VFLOPPY_HEADS * VFLOPPY_SECTORS_PER_TRACK * VFLOPPY_TRACKS * VFLOPPY_SECTOR_SIZE)
#define VFLOPPY_FILE_PATTERN		"/mnt/tmpfs/vfloppy%u.img"

FV_SPEC = {
};

#define VFLOPPY_MS_INDEX "vfloppy_ms_index"

static int pre_validate_hook(webs_t wp, form_handler_t * fh);
static int vfloppy_upload_cb(webs_t wp, char * name, char * filename, void * data,
			     size_t data_offset, size_t data_len, int more_data);

int
vfloppy_intern_tmpl_init(void)
{
    form_handler_t * fh;

    if (vfloppy_common_tmpl_init() != 0) return -1;

    /* register POST callbacks */
    websAddPostDataCallback("vfloppy_file", vfloppy_upload_cb);

    fh = CREATE_FH_INSTANCE(TEMPLATE_VFLOPPY_INTERN, ACL_OBJ_VFLOPPY);

    fh->pre_validate_hook = pre_validate_hook;

    REGISTER_FH_INSTANCE_AND_RETURN(fh);
}

static int
pre_validate_hook(webs_t wp,  form_handler_t * fh UNUSED)
{
    vfloppy_data_t * vfd = NULL;
    u_int ms_index = UINT_MAX;
    int ret = -1;

    if ( wp->invalid_file ) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Invalid File selected."));	
	return -1;
    }
   
    if (!wp->upload_failed) {
	if (form_button_clicked(wp, "action_vfloppy_upload")) {
	    int error;

	    wp->fh_flags |= FH_FLAG_ABORT_AT_VALIDATE;

	    ms_index = pp_strtoul_10(websGetVar(wp, VFLOPPY_MS_INDEX, NULL), UINT_MAX, NULL);
	    if (ms_index < PP_FEAT_USB_MASS_STORAGE_NO) {
		vfd = &vfloppy_data[ms_index];

		MUTEX_LOCK(&vfd->upload_mtx);

		if (!vfd->is_loaded) {
		    set_response(wp, ERIC_RESPONSE_ERROR, _("No floppy image uploaded."));
		    goto bail_unlock;
		}
		if (pp_usb_ms_set_image(ms_index, vfd->fd, vfd->filename, vfd->size, vfd->name, &error)) {
		    set_response(wp, ERIC_RESPONSE_ERROR, pp_error_string(error));
		    goto bail_unlock;
		}

		MUTEX_UNLOCK(&vfd->upload_mtx);
		vfd = NULL; /* prevents second unlock in bail_unlock: */

		set_response(wp, ERIC_RESPONSE_OK, _("Floppy image uploaded successfully."));
		ret = 0;
	    } else {
		set_response(wp, ERIC_RESPONSE_ERROR, internal_error_msg);
	    }
	} else {
	    ret = 0;
	}
    }

 bail_unlock:
    if (vfd) {
	if ( vfloppy_session_check(wp->session, ms_index, 0) ) {
	    vfloppy_discard(ms_index);
	}
	MUTEX_UNLOCK(&vfd->upload_mtx);
    }
    return ret;
}

static int
vfloppy_upload_cb(webs_t wp, char * name UNUSED, char * filename, void * data, size_t data_offset,
		  size_t data_len, int more_data)
{
    vfloppy_data_t * vfd;

    eric_session_touch(wp->session);

    if (data_offset == 0) {
	wp->ms_index = pp_strtoul_10(websGetVar(wp, VFLOPPY_MS_INDEX, NULL), UINT_MAX, NULL);
	if (wp->ms_index >= PP_FEAT_USB_MASS_STORAGE_NO) {
	    set_response(wp, ERIC_RESPONSE_ERROR, internal_error_msg);
	    goto fail_no_unlock;
	}
	vfd = &vfloppy_data[wp->ms_index];
    } else {
	/* we assume wp->ms_index is correct! */
	vfd = &vfloppy_data[wp->ms_index];
    }

    MUTEX_LOCK(&vfd->upload_mtx);

    if (PP_ERR == pp_um_user_has_permission(wp->user, ACL_OBJ_VFLOPPY,
                                            pp_acl_raasip_yes_str)) {
	set_response(wp, ERIC_RESPONSE_ERROR, perm_denied_msg);
	goto fail;
    }

    if ( !more_data && data_len <= 2 ) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("Uploaded file does not exist or is empty."));
	goto  fail;
    }

    if (vfd->is_loaded) {
	set_response(wp, ERIC_RESPONSE_ERROR, _("A floppy image was uploaded already."));
	goto fail;
    }

    if (vfd->owning_session) {
	if (vfd->owning_session != wp->session) {
	    set_response(wp, ERIC_RESPONSE_ERROR,
			 _("A floppy image upload is already running by another user."));
	    goto fail;
	}
    } else {
	vfd->owning_session = wp->session;
	vfd->size = 0;
	if (filename != NULL) {
	    snprintf(vfd->name, sizeof(vfd->name), "%s", filename);
	} else {
	    snprintf(vfd->name, sizeof(vfd->name), "vfloppy%u.img", wp->ms_index);
	}
	snprintf(vfd->filename, sizeof(vfd->filename), VFLOPPY_FILE_PATTERN, wp->ms_index);
	if ((vfd->fd = open(vfd->filename, O_RDWR|O_CREAT|O_TRUNC)) < 0) {
	    pp_log_err("%s: open(%s) failed", ___F, vfd->filename);
	    goto fail_unload;
	}
    }

    if ((vfd->size + data_len) > (VFLOPPY_MAX_SIZE + 2)) {
	set_response(wp, ERIC_RESPONSE_ERROR,
		     _("Floppy image too big (max. %u kB)."),
		     VFLOPPY_MAX_SIZE / 1024);
	goto fail_unload;
    }

    if (data_len > 0) {
	if (write(vfd->fd, data, data_len) != (ssize_t)data_len) {
	    goto fail_unload;
	}
	vfd->size += data_len;
    }

    if (data == NULL) {
	if ( vfloppy_session_check(wp->session, wp->ms_index, 0) ) {
	    vfloppy_discard(wp->ms_index);
	}
    } else if (!more_data && vfd->size >= 2) {
	vfd->size -= 2;

	if (lseek(vfd->fd, vfd->size, SEEK_SET) < 0) {
	    pp_log_err("%s: seek() failed", ___F);
	    goto fail_unload;
	}
	if (ftruncate(vfd->fd, vfd->size)) {
	    pp_log_err("%s: ftruncate() failed", ___F);
	    goto fail_unload;
	}
	if (vfd->size > VFLOPPY_MAX_SIZE
	    || (vfd->size / 512 * 512) != vfd->size) {
	    set_response(wp, ERIC_RESPONSE_ERROR, _("Not a valid Floppy image."));
	    goto fail_unload;
	}
	vfd->is_loaded = 1;
    }

    wp->upload_failed = 0;

    MUTEX_UNLOCK(&vfd->upload_mtx);

    return 0;

 fail_unload:
    if ( vfloppy_session_check(wp->session, wp->ms_index, 0) ) {
	vfloppy_discard(wp->ms_index);
    }

 fail:
    wp->upload_failed = 1;
    MUTEX_UNLOCK(&vfd->upload_mtx);

 fail_no_unlock:
    return -1;
}

void
vfloppy_download_cb(webs_t wp)
{
    int ret = 500;
    
    if (wp->download_fd >= 0) {
	void * image;
	if ((image = mmap(NULL, wp->numbytes, PROT_READ, MAP_SHARED,
			  wp->download_fd, 0)) == MAP_FAILED) {
	    pp_log_err("%s: unable to mmap vfloppy image file\n", ___F);
	    goto bail;
	}
    
	if (websWriteBlock(wp, image, wp->numbytes) > 0) {	
	    wp->written += wp->numbytes;
	}

	ret = (wp->written >= wp->numbytes)  ? 200 : 0;

    bail:
	if (image != MAP_FAILED) munmap(image, wp->numbytes);
	if (wp->download_fd >= 0) wp->download_fd = -1;
    }
    websDone(wp, ret);
}
#endif /* PP_FEAT_MASS_STORAGE */
