/* 
 * webs.h -- GoAhead Web public header
 *
 * Copyright (c) GoAhead Software Inc., 1992-2000. All Rights Reserved.
 *
 * See the file "license.txt" for information on usage and redistribution
 */

#ifndef _WEBS_H
#define _WEBS_H 1

#if ! defined(NDEBUG)
/* tweb webs printf debuging convenience wrapper ;-) */
#define TWD(format, args...) printf("tweb: %s:%d %s(): " format "\n", __FILE__, __LINE__, __FUNCTION__, ##args)
#else
#define TWD(args...)
#endif

/******************************* Description *********************************/

/* 
 * GoAhead Web Server header. This defines the Web public APIs.
 * Include this header for files that contain ASP or Form procedures.
 * Include wsIntrn.h when creating URL handlers.
 */

/******************************** Includes ***********************************/

#include <openssl/bio.h>
#include <pp/vector.h>
#include <liberic_webs.h>
#include <liberic_session.h>
// really needed? #include <lara.h>
#include "uemf.h"

/********************************* Defines ***********************************/

/*
 * By license terms the server software name defined in the following line of
 * code must not be modified.
 */

#define WEBS_NAME		"GoAhead-Webs"
#define WEBS_VERSION		"2.1"

#define WEBS_HEADER_BUFINC	512	/* Header buffer size */
#define WEBS_ASP_BUFINC		512	/* Asp expansion increment */
#define WEBS_MAX_PASS		32	/* Size of password */
#define WEBS_BUFSIZE		1000	/* websWrite max output string */
#define WEBS_MAX_HEADER		(5*1024)/* Sanity check header */
#define WEBS_MAX_URL		4096	/* Maximum URL size for sanity */
#define WEBS_SOCKET_BUFSIZ	32768	/* Bytes read from socket */

#define WEBS_HTTP_PORT		"httpPort"

/* 
 * Request flags. Also returned by websGetRequestFlags().
 */

#define WEBS_LOCAL_PAGE		0x1	/* Request for local webs page */ 
#define WEBS_KEEP_ALIVE		0x2	/* HTTP/1.1 keep alive */
#define WEBS_DONT_USE_CACHE	0x4	/* Not implemented cache support */
#define WEBS_COOKIE		0x8	/* Cookie supplied in request */
#define WEBS_IF_MODIFIED	0x10	/* If-modified-since in request */
#define WEBS_POST_REQUEST	0x20	/* Post request operation */
#define WEBS_LOCAL_REQUEST	0x40	/* Request from this system */
#define WEBS_HOME_PAGE		0x80	/* Request for the home page */ 
#define WEBS_ASP		0x100	/* ASP request */ 
#define WEBS_HEAD_REQUEST	0x200	/* Head request */
#define WEBS_CLEN		0x400	/* Request had a content length */
#define WEBS_FORM		0x800	/* Request is a form */
#define WEBS_REQUEST_DONE	0x1000	/* Request complete */
#define WEBS_POST_DATA		0x2000	/* Already appended post data */
#define WEBS_CGI_REQUEST	0x4000	/* cgi-bin request */
#define WEBS_SECURE		0x8000	/* connection uses SSL */
#define WEBS_AUTH_BASIC		0x10000	/* Basic authentication request */
#define WEBS_AUTH_DIGEST	0x20000	/* Digest authentication request */
#define WEBS_HEADER_DONE	0x40000	/* Already output the HTTP header */
#define WEBS_GET_DATA		0x80000

/*
 * URL handler flags
 */
#define WEBS_HANDLER_FIRST	0x1	/* Process this handler first */
#define WEBS_HANDLER_LAST	0x2	/* Process this handler last */

/*
 * dimension structure used in webs structure
 */
typedef struct {
    int width;
    int height;
} dim_t;

struct websRec;
typedef struct websRec * webs_t;

typedef void (*download_cb_t)(webs_t);
typedef int (*webs_post_data_cb_t)(webs_t wp, char * name, char * filename,
				   void * data, size_t data_offset, size_t data_len,
				   int more_data);
typedef enum {
    CASE_UP,
    CASE_DOWN
} webs_trans_case_type_t;

typedef enum {
    ERIC_RESPONSE_OK,
    ERIC_RESPONSE_WARNING,
    ERIC_RESPONSE_ERROR,
} response_code_t;

/* forward declaration */
typedef struct form_var_s form_var_t;

/* 
 * Per socket connection webs structure
 */
struct websRec {
    eric_session_int_id_t 	session;
    eric_webs_session_data_t *	sd;
    const char *		user;		/* the authenticated user */
    int				uid;		/* uid of authenticated user */
    char *			target_user;	/* the target user */
    int				target_uid;	/* uid of target user */
    char *			target_group;	/* the target group */
    int				target_gid;	/* gid of target group */
    u_int                       target_unit;	/* the target_unit */
    u_int                       target_port;	/* the target port */
#if defined(KIRA_RPC)
    u_int                      target_outlet;	/* the target outlet */
#endif /* KIRA_RPC */
    BIO *			bio;
    ringq_t			header;		/* Header dynamic string */
    char			boundary[256];	/* the multi part boundary */
    size_t			boundary_len;	/* boundary length */
    sym_fd_t			cgiVars;	/* CGI standard variables */
    char			type[64];	/* Mime type */
    char *			dir;		/* Dir containing the page */
    char *			path;		/* Path name without query */
    char *			url;		/* Full request url */
    char *			host;		/* Requested host */
    char *			lpath;		/* Cache local path name */
    char *			query;		/* Request query */
    size_t			query_len;	/* Length of request query */
    size_t			post_start;	/* index where the post data starts */
    char *			decodedQuery;	/* Decoded request query */
    char *			cookie;		/* Cookie string */
    char *			userAgent;	/* User agent (browser) */
    dim_t			uadim;		/* screen size of user agent*/
    char *			protocol;	/* Protocol (normally HTTP) */
    char *			protoVersion;	/* Protocol version */
#if defined(PP_FEAT_WS_MANAGEMENT)
    char *			wsman_auth;	/* header authentication for WS-Management */
    char *			wsman_response;	/* WS-Management response */
    size_t			wsman_response_size;
    int				wsman_code;	/* http response code */
#endif
    int				download_fd;	/* file descriptor for download */
    void *			download_image;	/* Memory image to download */
    const char *		download_name;	/* Name of file to download */
    download_cb_t		download_cb;	/* Download callback */
    const char *		tmp_passwd;	/* ... for passwd validation */
    int				state;		/* Current state */
    int				flags;		/* Current flags */
    int				code;		/* Request result code */
    int				clen;		/* Content length */
    int				docfd;		/* Document file descriptor */
    int				numbytes;	/* Bytes to transfer */
    int				written;	/* Bytes transferred */
    int				auth_ok;	/* 1 if cookie is valid */
    int				skipOutput;	/* 1 for skipping HTML output*/
    char *			support_data;	/* support data */
    unsigned char *		cert_data;	/* cert upload data */
    size_t			cert_data_size; /* size of cert upload data */
    int				upload_failed;	/* !=0 if an upload failed */
    int				invalid_file;   /* file uploaded with null name */
    response_code_t		response_code;	/* response code of forms */
    const char *		response_msg;	/* response msg of forms */ 
    char			msg_buf[512];	/* message buffer */
    form_var_t *		form_vars;	/* form variables */
    vector_t *			fh_vec;		/* vector with form handlers */
    unsigned int		fh_flags;	/* flags for main form handler */
    char *			mp_form_data;	/* multipart form data (tmp) */
    size_t			mp_form_data_len;/* multipart form data len (excluding terminating 0 !) */
    int				close_power_switches; /* !=0: close powersw. */
    u_int			ms_index;	/* mass-storage index during upload */
};

/******************************* Prototypes **********************************/

int	websAddPostDataCallback(const char * name,
					webs_post_data_cb_t cb);
int	websAspDefine(const char *name, int (*fn)(int ejid, webs_t wp,
						  int argc,char **argv));
int	websAspRequest(webs_t wp, char *lpath);
void	websClose(void);
void	websDecodeUrl(char *token, char *decoded, int len);
void	websDone(webs_t wp, int code);
void	websError(webs_t wp, int code, const char *fmt, ...);
char 	*websGetDefaultDir(void);
char 	*websGetDefaultPage(void);
const char *websGetVar(webs_t wp, const char *var, const char *def);
void	websTrimVar(webs_t wp, const char *var);
void	websTransCaseVar(webs_t wp, const char *var, webs_trans_case_type_t type);
char   **websSearchVars(webs_t wp, const char *var);
int	websOpen(void);
void	websRedirect(webs_t wp, const char * url, const char * cookie);
void	websSetDefaultDir(const char *dir);
void	websSetDefaultPage(const char *page);
void	websSetEnv(webs_t wp);
void	websSetRequestLpath(webs_t wp, const char *lpath);
void	websSetRequestPath(webs_t wp, const char *dir, const char *path);
void	websSetVar(webs_t wp, const char *var, const char *value);
int	websUrlHandlerDefine(const char *urlPrefix,const char *webDir,int arg,
			     int (*fn)(webs_t wp, char *urlPrefix, 
				       char *webDir, int arg,
				       char *url, char *path, 
				       char *query), int flags);
int	websUrlHandlerDelete(int (*fn)(webs_t wp, char *urlPrefix,
					char *webDir, int arg,
					char *url, char *path, 
					char *query));
int	websUrlHandlerRequest(webs_t wp);
int	websUrlParse(const char *url, char **buf, const char **host, 
		     const char **path, const char **port, const char **query,
		     const char **proto, const char **tag, char **ext);
char	*websUrlType(const char *webs, char *buf, int charCnt);
int	websWrite(webs_t wp, const char * fmt, ...);
int	websWriteBlock(webs_t wp, const char * buf, int nChars);
int	websValidateUrl(webs_t wp, char *path);

webs_t	websAlloc(void);
void	websFree(webs_t wp);

int	webs_bio_eof(webs_t wp);
int	webs_bio_gets(webs_t wp, char ** buf);
int	webs_bio_read_exact(webs_t wp, void * buf, size_t len);
int	webs_bio_write_exact(webs_t wp, const void * buf, size_t len);
void	webs_bio_close(webs_t wp);

#endif /* _WEBS_H */
