/*--------------------------------------------------------------------------

	CC.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Command Center - IP-Reach Integration Functions

---------------------------------------------------------------------------*/

#include <stdio.h>
#include <pp/syms.h>
#include <pp/OS_Port.h>
#include <pp/SXDB_Parse.h>
#include <pp/SXDB_Parse_Table.h>
#include <pp/SIO.h>
#include <pp/RDM.h>
#include <pp/CC.h>
#include <pp/RDM_DDA_Utility.h>

//----------------------------------------
//				Equates
//----------------------------------------

#define	CC_EVENTS_PER_RETRY			10	// # of events we will get at one time
#define	CC_INCOMING_MSG_MAX			100	// Biggest incoming message we expect

#define	BEGIN_WRITE_ACCESS()	{rdm->db->BeginAccess( 1 );}
#define	BEGIN_READ_ACCESS()	{rdm-?db->BeginAccess( 0 );}
#define	END_ACCESS() {rdm->db->EndAccess();}

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

	int
	CCEventFunc
	(
		void	  *	pUserData,
		RDM_EVENT *	pEvent				// The event
	);

	void
	CCTimeoutProc
	(
		CTimer	*	pThis,
		void	*	pParam1,
		void	*	pParam2
	);

    extern 
    BOOL
    IsDeviceInUse
    (
    );

    void
    CreatePowerRedirectService
    (
    );


//----------------------------------------
//		Restart Function Parse Table
//----------------------------------------

//----------------------------------------
//		CommandCenter Parse Table
//----------------------------------------

#define	PT_STRUCT	CC_CONFIG
PT_BEGIN	( "CommandCenter",	ccConfigTable,	PT_UNKNOWN_OK )
PT_ELEM		( "Enable",			enable,			0,					PT_INT | PT_REQUIRED )
PT_ELEM		( "ClusterID",		clusterID,		RDM_MAX_CLUSTER,	PT_STRING | PT_REQUIRED )
PT_ELEM		( "CCUser",			ccUser,			RDM_MAX_USER_NAME,	PT_STRING | PT_REQUIRED )
PT_ELEM		( "CCPassword",		ccPassword,		RDM_MAX_USER_NAME,	PT_STRING | PT_REQUIRED )
PT_ELEM		( "IPAddress",		ipAddress,		0,					PT_IP | PT_REQUIRED )
PT_ELEM		( "IPPort",			ipPort,			0,					PT_INT )
PT_ELEM		( "HeartbeatTime",	heartbeatTime,	0,					PT_INT | PT_REQUIRED )
PT_ELEM		( "EventKey",		pEventKeyBase64,0,					PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Timeout",		ccTimeout,		0,					PT_INT )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//---------------------------------------------------------------------------
//									CC
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CCommandCenter::CCommandCenter
	(
		CRDM * pRDM
	)
//
//	Initializes the class members
//
//---------------------------------------------------------------------------
{
	msgSent			= 0;
	retryPending	= 0;
	heartBeatGood	= 0;
	hearBeatCount	= 0;
	rdm				= pRDM;

	// Write the data to the database

	memset(&config,0,sizeof(config));
}


//---------------------------------------------------------------------------
//
	void 
    CCommandCenter::InitializeManagement
	(
	)
//
//	Initializes the class members with CommandCenter node values
//
//---------------------------------------------------------------------------
{
	int				result;
	CSXDB_Node	*	pNode;

    // if already managed, stopmanagement

    if (config.enable == 1)
    {
        StopManagement();
        config.enable = 0; 
    }

	BEGIN_WRITE_ACCESS()

	pNode = rdm->db->GetNodeFromXPath("/System/Device/DeviceSettings/Security/CommandCenter",NULL);

	result = SXDB_PT_Get(pNode, ccConfigTable, &config );

	END_ACCESS()

    if (config.enable == 0)
        return;

	if (result < 0)
	{
		config.enable = 0;
		return;
	}

	// We are managed...start up all things CC

	// Setup the UDP socket for event transmission

	SetupSocket( config.ipAddress, config.ipPort );
	
	// Setup the RC4 Key for UDP message encryption

	SetupEventKey( config.pEventKeyBase64 );

	// Start the heartBeat timer

	if (config.heartbeatTime < 5)
		config.heartbeatTime = 5;

	if (config.heartbeatTime > 50000)
		config.heartbeatTime = 50000;

	SetCTimer( config.heartbeatTime * 1000 );
	StartCTimer();

	// Start the Timeout timer

	heartBeatGood = 1;

	if (config.ccTimeout < 1)
		config.ccTimeout = 1200;	// 20 minutes default

	if (config.ccTimeout > 32768*65)
		config.ccTimeout = 32768*65;

	timeoutTimer.SetCTimerProc( CCTimeoutProc, this, 0 );
	timeoutTimer.SetCTimer( config.ccTimeout * 1000 );
	timeoutTimer.StartCTimer();

	// Install Event subscription for the UDP event messages

	SetSubscription( "*", "*" );
	rdm->nm.AddSubscription( this );

// Hack    CreatePowerRedirectService();

	DDA_Notify_CCManagement ( rdm, RDM_EC_System_Start_Management, /*Hack getServerName()*/ "DeviceName", config.ccUser, config.ipAddress );
}

//---------------------------------------------------------------------------
//
	CCommandCenter::~CCommandCenter
	(
	)
//
//	Initializes the object
//
//---------------------------------------------------------------------------
{
	if (IsEnabled())
		StopManagement();
}
 
//---------------------------------------------------------------------------
//
	void
    CCommandCenter::StopManagement
	(
	)
//
//	Initializes the object
//
//---------------------------------------------------------------------------
{
    // Release sessions and reset the path/port status

    config.enable   = 0;
    msgSent         = 0;
    retryPending    = 0;
    heartBeatGood   = 0;
    hearBeatCount   = 0;

	/* hack - Should move to Session Manager --> listed for RDM_EC_System_Stop_Management

    BEGIN_WRITE_ACCESS()
	if ( strcmp(phw->GetPlatformID(),"USTIP")==0 ) //CR 6876 and 6877
		ReleaseResourceAndSession();
	else
	{
		ResetPortStatus();
		ResetPathStatus();
	}
    END_ACCESS()
	*/

    closesocket( udpSocket );

    StopCTimer();

    timeoutTimer.StopCTimer();

    rdm->nm.RemoveSubscription( this );

	DDA_Notify_CCManagement ( rdm, RDM_EC_System_Stop_Management, /* hack getServerName()*/ "DeviceName", config.ccUser, config.ipAddress );
}

/* hack - Should move to Session Manager --> listed for RDM_EC_System_Stop_Management

//---------------------------------------------------------------------------
//
    void
    CCommandCenter::ReleaseResourceAndSession
    (
    )
//
//  Release used resource and session
//
//---------------------------------------------------------------------------
{
    char            str[128];
    CSXPath         p;
    CSXDB_Node      *pNode;
    int             count;
    CSession        *pSession;

    sprintf (str, "/System/Sessions/Session/ResourceUsed[@Pending]/..");

    count = p.Parse(str, rdm->db);

    if (count > 0)
    {
        for (int i = 0; i < count; i++)
        {
            pNode = p.Enum(i);
            pNode = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "id", NULL);
            if (pNode != NULL)
            {
                pSession = gSessionManager.GetSession(pNode->GetData());

                if (pSession != NULL)
                {
                    // Update the resource status before closing the session
                    DDA_ReleaseResources(&rdm, pNode->GetData(), 1);

                    // close and release the session

                    pSession->Close();

                    pSession->Release();
                }
            }
        }
    }
}
//---------------------------------------------------------------------------
//
	void
    CCommandCenter::ReleaseThisResourceAndSession
	(
        char * resourceID
	)
//
//	Initializes the object
//
//---------------------------------------------------------------------------
{
    char            str[128];
	CSXPath			p;
	CSXDB_Node	*pNode;
    int             count;
    CSession	*	pSession;

    // Resource is in use so change "Pending" attribute to false

    sprintf (str, "/System/Sessions/Session[ResourceUsed=%s]", resourceID );

    count = p.Parse(str, rdm->db);

    if (count ==1)
    {
        pNode = p.Enum(0);
        pNode = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "id", NULL);

        pSession = gSessionManager.GetSession(pNode->GetData());

	    if (pSession != NULL)
	    {
            // Update the resource status before closing the session

            DDA_ReleaseResources(&rdm, pNode->GetData(), 1);

            // close and release the session

		    pSession->Close();

            pSession->Release();
    	}
    }
}


//---------------------------------------------------------------------------
//
	void
    CCommandCenter::ResetPathStatus
	(
	)
//
//	Initializes the object
//
//---------------------------------------------------------------------------
{
    int         count;

    CSXPath		p;
	CSXDB_Node	*pNode;
    char        * id;
    char        * status;

    // Retrieve Paths
    
    count = p.Parse("System/Device/Port/Path", rdm->db);
   
    for (int i=0; i<count; i++)
    {
        pNode = p.Enum(i);

        id     = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "id", NULL)->GetData();
        status = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "Status", NULL)->GetData();

        if (status[0] == '0' || status[0] == '2')  // 0-reserved, 2-occupied
            ReleaseThisResourceAndSession( id );
    }
}


//---------------------------------------------------------------------------
//
	void
    CCommandCenter::ResetPortStatus
	(
	)
//
//	Initializes the object
//
//---------------------------------------------------------------------------
{
    int         count;

    CSXPath		p;
	CSXDB_Node	*pNode;
    char        * id;
    char        * status;

    // Retrieve Ports

    count = p.Parse("System/Device/Port", rdm->db);
   
    for (int i=0; i<count; i++)
    {
        pNode = p.Enum(i);

        id     = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "id", NULL)->GetData();
        status = pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "Status", NULL)->GetData();

        if (status[0] == '0' || status[0] == '2')  // 0-reserved, 2-occupied
            ReleaseThisResourceAndSession( id );
    }

}
*/

//---------------------------------------------------------------------------
//
	CUserObject *						// Returns a user object ptr if authenticated
	CCommandCenter::Authenticate
	(
		const char	*pUserName,			// The user name
		const char	*pPassword			// The password
	)
//
//	Authenticates the user name and password for CC Mode.
//
//---------------------------------------------------------------------------
{
	if (stricmp(pUserName,config.ccUser) != 0)
		return NULL;

	if (strcmp(pPassword,config.ccPassword) != 0)
		return NULL;

	hearBeatCount++;
	heartBeatGood = 1;

	return new CCUserObject;
}

//---------------------------------------------------------------------------
//
	void
	CCommandCenter::SetupEventKey
	(
		const char	*pKey
	)
//
//	Initializes the RC4 key for CC UDP message encryption/decryption
//
//---------------------------------------------------------------------------
{
	EVP_ENCODE_CTX	ctx;
	int				keyLength;
	unsigned char	keyData[RDM_MAX_KEY_BASE64*4];
	int				x;

	if (config.pEventKeyBase64 == NULL)
	{
		isEncrypted = 0;
		return;
	}

	isEncrypted = 1;

	EVP_DecodeInit(&ctx);
	EVP_DecodeUpdate(&ctx, keyData, &x, (const unsigned char *) pKey, strlen(pKey));
	keyLength = x;
	EVP_DecodeFinal(&ctx, &keyData[x], &x);
	keyLength += x; 

	RC4_set_key( &eventKey, keyLength, keyData);
}

//--------------------------------------------------------------------------------
//
	void
	CCommandCenter::EncryptMessage
	(
		char	*pData,				// Data to Encrypt/Decrypt
		int		length					// # of bytes
	)
//
//	Encrypts or decrypts data for CC UDP messages
//
//--------------------------------------------------------------------------------
{
	RC4_KEY		key;

	if (isEncrypted == 0)
		return;

	memcpy(&key, &eventKey, sizeof(key));

	RC4( &key, length, (unsigned char *) pData, (unsigned char *) pData );
}

//---------------------------------------------------------------------------
//
	int
	CCommandCenter::SetupSocket
	(
		int	ipAddress,
		int	ipPort
	)
//
//	Creates the UDP socket for CC Event protocol
//
//---------------------------------------------------------------------------
{
	// Create a new socket

	udpSocket = socket( AF_INET, SOCK_DGRAM, 0 );

	if (udpSocket == INVALID_SOCKET)
		return INVALID_SOCKET;

	// Get the port

	if (ipPort == 0)
		ipPort = 5001;

	// Set the address info

	socketAddress.sin_family = AF_INET;
#ifdef WIN32
	socketAddress.sin_addr.S_un.S_addr = htonl(ipAddress);
#else
	socketAddress.sin_addr.s_addr = htonl(ipAddress);
#endif
	socketAddress.sin_port = htons(ipPort);

	return udpSocket;
}

//--------------------------------------------------------------------------------
//
	void
	CCommandCenter::Notify
	(
		RDM_EVENT	*pEvent				// The event
	)
//
//	Sends a notification to this subscription
//
//------------------------------------------------------------------------------//
{
	char	eventText[RDM_MAX_NOTIFICATION];

	RDM_ConvertEventToXML( pEvent, eventText );
	SendCCUDPMessage( eventText );
}

//--------------------------------------------------------------------------------
//
	int
	CCommandCenter::SendCCUDPMessage
	(
		const char	*pXMLText			// The XML text of the message
	)
//
//	Formats, encrypts, and sends a message to CC
//
//--------------------------------------------------------------------------------
{
	char	data[RDM_MAX_NOTIFICATION+20];
	int		length = strlen(pXMLText)+5;

	* ((long *) data) = htonl( length );
	strcpy(&data[4],pXMLText);
	EncryptMessage( data, length );

	msgSent++;

	return sendto( udpSocket, data, length, 0, (sockaddr *) &socketAddress, sizeof(socketAddress) );
}

//--------------------------------------------------------------------------------
//
	void 
	CCommandCenter::TimerProc
	(
	)
//
//	Virtual function called when timer is triggered
//
//--------------------------------------------------------------------------------
{
	char	msg[100];

	// See if we need to send a heart beat

	if (msgSent == 0)
	{
		sprintf(msg,"<CSC_Heartbeat id=\"%s\"/>", RDM_GetDeviceID());
		SendCCUDPMessage(msg);
	}

	msgSent = 0;
}

//--------------------------------------------------------------------------------
//
	int
	CCommandCenter::Retry
	(
		int		start,					// Starting serialNo
		int		count					// # of messages
	)
//
//	Gets the requested events and transmits them to CC
//
//--------------------------------------------------------------------------------
{
	RDM_EVENT	events[CC_EVENTS_PER_RETRY];
	int			want;
	int			have;
	int			unavailCount;
	int			x;
	char		msg[RDM_MAX_NOTIFICATION];

	while (count > 0)
	{
		if (count > CC_EVENTS_PER_RETRY)
			want = CC_EVENTS_PER_RETRY;
		else
			want = count;

		have = rdm->nm.GetEvents( &events[0], start, want );

		if (have < 1)
		{
			// No events were available in the range we asked for

			sprintf(msg,"<CSC_EventUnavailable Start=\"%d\" Count=\"%d\" Next=\"%d\"/>",
							start,count,0);
			SendCCUDPMessage(msg);
			break;
		}

		for (x=0; count>0 && x<have; )
		{
			if (events[x].serialNo == start)
			{
				// Expected event, send it

				RDM_ConvertEventToXML( &events[x], msg );
				SendCCUDPMessage( msg );
				start++;
				count--;
				x++;
			}
			else if (events[x].serialNo < start)
			{
				// problem...bail out... should never get here
				return -1;
			}
			else
			{
				// We missed some...send an unavailable message

				unavailCount = events[x].serialNo - start;
				if (unavailCount > count)
					unavailCount = count;

				sprintf(	msg,
							"<CSC_EventUnavailable Start=\"%d\" Count=\"%d\" Next=\"%d\"/>",
							start,
							unavailCount,
							events[x].serialNo
					   );

				SendCCUDPMessage(msg);

				start += unavailCount;
				count -= unavailCount;
			}
		}
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// !0 message was from CC
	CCommandCenter::ProcessCCMessage
	(
		const char	*pData,				// The message data
		int		length
	)
//
//	Checks the message data (from UDP) to see if it is ment for CC
//
//--------------------------------------------------------------------------------
{
	char	msg[CC_INCOMING_MSG_MAX];
	int		msgLength;
	int		result;

	if (!IsEnabled())
		return 0;

	if (length > CC_INCOMING_MSG_MAX)
		return 0;

	// Copy the message so we can decrypt it

	memcpy(msg,pData,length);

	EncryptMessage( msg, length );

	// See if it looks like a message

	msgLength = ntohl(* ((unsigned long *) msg));

	if (msgLength > length || msgLength < 5)
		return 0;

	// Parse the XML

	CSXDB_Parse		db;
	CSXDB_Node	*	pNode;
	const char		*	data = NULL;;
	int				start,count;

	result = db.Parse( &msg[4], msgLength-4 );

	if (result < 0)
		return 0;

	// Decifer the command

	pNode = db.GetDataBase()->Root();
	if (pNode != NULL)
		pNode = pNode->Child();

	if (pNode != NULL)
		data = pNode->GetName();

	if (data == NULL)
		return 0;

	if (strcmp( data, "CSC_Heartbeat" ) == 0)
	{
		// Heart beat message

		hearBeatCount++;
		heartBeatGood = 1;
	}
	else if (strcmp( data, "CSC_GetEvents" ) == 0)
	{
		// Retry

		hearBeatCount++;
		heartBeatGood = 1;

		data = db.GetDataBase()->GetData( "/CSC_GetEvents/@Start" );

		if (data != NULL)
		{
			start = atoi(data);

			data = db.GetDataBase()->GetData( "/CSC_GetEvents/@Count" );

			if (data != NULL)
			{
				count = atoi(data);

				if (start >= 0 && count >= 1)
					Retry( start, count );
			}
		}
	}
	else
		return 0;

	return 1;
}


bool CCommandCenter::ReceiveUDPMessage( CUDPListener * NOTUSED(pUDP), int NOTUSED(ipAddress), char * pData, int count )
{
	return (bool) ProcessCCMessage(pData,count);
}


//--------------------------------------------------------------------------------
//
	void
	CCommandCenter::Timeout
	(
	)
//
//	Called by the timer proc when the CCTimeout period is over
//
//--------------------------------------------------------------------------------
{
	if (hearBeatCount == 0)
		heartBeatGood = 0;

	hearBeatCount = 0;
}

//--------------------------------------------------------------------------------
//
	void
	CCTimeoutProc
	(
		CTimer	*	NOTUSED(pThis),
		void	*	pParam1,
		void	*	NOTUSED(pParam2)
	)
//
//	Checks to see if we have recieved a heartbeat during the last timeout period
//
//--------------------------------------------------------------------------------
{
	CCommandCenter *p = (CCommandCenter *) pParam1;
	p->Timeout();
}
