/**	
 *	@file	NetConn_Socket.cpp
 *	@brief	CNetConn derived class for unencrypted socket communications
 * 
 * 	Implementation of the CNetConn_Socket class.
 *	Communications channel through a clear text socket
 *
 */

#include	"pp/NetConn_Socket.h"
#include	"assert.h"
#ifdef OS_POSIX
#include	<fcntl.h>
//#include	<signal.h>
#endif

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

	// Timeout values (in milliseconds)

//#define NOTIMEOUT	// for debugging only

#define	MINUTES					60000
#define	SELECT_TIMEOUT			15000		// Must be evenly divisable by a thousand
#define	IO_TIMEOUT				4*MINUTES

	// misc.

#ifdef OS_POSIX
#define	S_ADDR		s_addr
#define	FD_SET_T	fd_set
#define	SOCKADDR	struct sockaddr
#else
#define	S_ADDR		S_un.S_addr
#define	FD_SET_T	struct fd_set
#define	socklen_t	int
#define	SHUT_RDWR	2
#endif

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	static data
 *--------------------------------------*/

DWORD			dataIn = 0;				// # of bytes received
DWORD			dataOut = 0;			// # of bytes sent

/*----------------------------------------
 *	CNetConn_Socket Class
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
/** Constructors **/

CNetConn_Socket::CNetConn_Socket()
{
	ioSocket = INVALID_SOCKET;
	running = 1;
}

CNetConn_Socket::CNetConn_Socket(SOCKET s)
{
	ioSocket = s;
	running = 1;
}

CNetConn_Socket::~CNetConn_Socket()
{
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

int CNetConn_Socket::OpenSocket( int ipAddress, int port )
{
	SOCKET					s;
	struct	sockaddr_in		sa;
	int						result;
	int						on  = 1;

	s = socket( AF_INET, SOCK_STREAM, 0 );
	assert(s != INVALID_SOCKET);
	if (s == INVALID_SOCKET)
		return -1;
	
	sa.sin_port = htons((short) port);
	sa.sin_family = AF_INET;
	sa.sin_addr.S_ADDR = htonl(ipAddress);

	result = connect( s, (SOCKADDR *) &sa, sizeof(sa) );

	setsockopt( s, IPPROTO_TCP, TCP_NODELAY,  (char *) &on, sizeof( on ) );

	#ifdef OS_POSIX_BLOCK
	{
		int	flags;

		flags = fcntl( s, F_GETFL, 0 );
		assert( flags >= 0);
		flags = fcntl( s, F_SETFL, flags | O_NONBLOCK );
		assert( flags >= 0);
	}
	#endif


	ioSocket = s;

	return result;
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

void CNetConn_Socket::CloseSocket( )
{
	closesocket(ioSocket);
}


/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

void CNetConn_Socket::SetSocket( SOCKET s )
{
	ioSocket = s;
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

SOCKET CNetConn_Socket::GetSocket( )
{
	return ioSocket;
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

void CNetConn_Socket::Shutdown( )
{
	running = 0;
	CloseSocket();
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

BOOL CNetConn_Socket::Write(const void * _pData, int count)
{
	const BYTE		*	pData = (const BYTE *) _pData;
	BOOL			block = FALSE;
	int				result,selectResult;
	DWORD			error;
	FD_SET_T		f_write;
	struct	timeval	selectTimeOut;
	DWORD			timeout;

	// Write until all data has been written or there is an error

	while (count && running)
	{
		// write some data

		result = send( this->ioSocket, (const char *) pData, count, 0 );

//		DBLog3(("%d send() result %d\n",OS_GetTickCount(),result));

		// See if there was an error during the write

		if (result < 0)
		{
			error = OS_GetLastSocketError();

			if (error == OS_EWOULDBLOCK)
			{
				// Error just indicates there was nothing to read, we will block until we can write some more

				block = TRUE;
			}
			else
			{
				// Error...fail the call

//				DBLog1(("Write error %d\n",error));
				break;
			}

			result = 0;
		}
		else if (result < count)
		{
			// We wrote some, but not enough, block until we can write some more

			block = TRUE;
		}

		// If there is no more write buffer space for this socket, then we should block

		dataOut += result;

		if (block)
		{
			// Call select to wait, but check to see if we are closing the connections every SELECT_TIMEOUT periods

			timeout = IO_TIMEOUT / SELECT_TIMEOUT;

			do
			{

				// Wait for buffer space to free

				FD_ZERO(&f_write);
				FD_SET( this->ioSocket, &f_write);

				// Compute the timeout values

				selectTimeOut.tv_usec = 0;
				selectTimeOut.tv_sec  = SELECT_TIMEOUT / 1000;

				#ifdef NOTIMEOUT
					selectResult = select( this->ioSocket+1, NULL, &f_write, NULL, NULL );
				#else
					selectResult = select( this->ioSocket+1, NULL, &f_write, NULL, &selectTimeOut );

					timeout--;
				#endif

			} while (selectResult == 0 && timeout && running);

			block = FALSE;

			// Check for timeout

			if (selectResult == 0 && timeout == 0)
				break;
		}

		// Account for the data we have read so far

		count -= result;
		pData += result;

	}

	return count == 0 ;}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

BOOL CNetConn_Socket::Read(void * _pData, int count)
{
	BYTE		*	pData = (BYTE *) _pData;
	BOOL			block = FALSE;
	int				result,selectResult;
	DWORD			error;
	FD_SET_T		f_read;
	struct	timeval	selectTimeOut;
	DWORD			timeout;

	// Read until all data has been read or there is an error
	while (count && running)
	{
		// Read some data

		result = recv( this->ioSocket, (char *) pData, count, 0 );

//		DBLog3(("%d recv() result %d\n",OS_GetTickCount(),result));

		// See if there was an error during the read

		if (result <= 0)
		{
			error = OS_GetLastSocketError();

			if (result == 0)
			{
//				DBLog2(("Read connection closed\n"));
				break;
			}

			if (error == OS_EWOULDBLOCK)
			{
				// Error just indicates there was nothing to read, we will block until there is

				block = TRUE;
			}
			else
			{
				// Error...fail the call

//				DBLog1(("Read error %d\n",error));
				break;
			}

			result = 0;
		}
		else if (result < (int) count)
		{
			// We read some, but not enough, block until there is more data

			block = TRUE;
		}

		// Wait for more data if we need to

		dataIn += result;

		if (block)
		{
			// Call select to wait, but check to see if we are closingthe connections every SELECT_TIMEOUT periods

			timeout = IO_TIMEOUT / SELECT_TIMEOUT;

			do
			{
				// We are going to wait for more data

				FD_ZERO(&f_read);
				FD_SET( this->ioSocket, &f_read);

				// Compute the timeout values

				selectTimeOut.tv_usec = 0;
				selectTimeOut.tv_sec  = SELECT_TIMEOUT / 1000;

				#ifdef NOTIMEOUT
					selectResult = select( this->ioSocket+1, &f_read, NULL, NULL, NULL );
				#else
					selectResult = select( this->ioSocket+1, &f_read, NULL, NULL, &selectTimeOut );

					timeout--;
				#endif

			} while (selectResult == 0 && timeout && running);

			block = FALSE;

			// Check for timeout

			if (selectResult == 0 && timeout == 0)
				break;
		}

		// Account for the data we have read so far

		count -= result;
		pData += result;
	}

	return count == 0 ;
}

/*  --------------------------------------------------------------------*/
/** See NetConn_Socket.h **/

BOOL CNetConn_Socket::Select ( int timeout )
{
	FD_SET_T		f_read;
	int				selectResult;
	struct	timeval	selectTimeOut;

	// We are going to wait for more data

	FD_ZERO(&f_read);
	FD_SET( this->ioSocket, &f_read);

	// Compute the timeout values

	selectTimeOut.tv_usec = 0;
	selectTimeOut.tv_sec  = timeout;

	selectResult = select( this->ioSocket+1, &f_read, NULL, NULL, &selectTimeOut );

	return selectResult > 0;
}


