/**	
 *	@file	NetConn_SSL.h
 *	@brief	Implementation of the CTCPListener class.
 *  		Listens for client sockets on a specified port
 */

#ifdef OS_POSIX
# include	<fcntl.h>
#endif
#include	<assert.h>
#include	<pp/syms.h>
#include	<pp/TCPListener.h>

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

	// Timeout values (in milliseconds)

#define	EXIT_TIMEOUT			5000		// Must be evenly divisable by a thousand

	// misc.

#ifdef OS_POSIX
#define	S_ADDR		s_addr
#define	FD_SET_T	fd_set
#define	SOCKADDR	struct sockaddr
#else
#define	S_ADDR		S_un.S_addr
#define	FD_SET_T	struct fd_set
#define	socklen_t	int
#define	SHUT_RDWR	2
#endif

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

DWORD CTCPListenerGlue(LPVOID param);

/*----------------------------------------
 *	Static Data
 *--------------------------------------*/

/*----------------------------------------
 *	Code
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

CTCPListener::CTCPListener( int _port )
{
	// ----------------------------------------
	// Init our member variables

	listenSocket	= INVALID_SOCKET;
	initialized	= 0;
	port		= _port;
	thread		= NULL;
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

CTCPListener::~CTCPListener()
{
	Shutdown();

	if (thread != NULL)
	{
		OS_WaitForThread( thread );
		OS_DeleteThread( thread );
		thread = NULL;
	}
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

int CTCPListener::Initialize()
{
	struct	sockaddr_in	socketAddress;	// Socket Address structure for the TCP socket
	int		result;
	int		on = 1;
	int retry;
	
	// Open the TCP Listen socket

	this->listenSocket = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );

	if ( !OS_IsSocketValid( this->listenSocket ) )
	{
		return -1;
	}

	// Bind the socket to an address

	socketAddress.sin_family = AF_INET;
	socketAddress.sin_port = htons( (short) ( this->port ) );
	socketAddress.sin_addr.s_addr = INADDR_ANY;

	result = setsockopt( this->listenSocket, SOL_SOCKET,  SO_REUSEADDR, (char *) &on, sizeof( on ) );
	
	retry = 0;
rebind:
	result = bind( this->listenSocket, (SOCKADDR *) &socketAddress, sizeof(socketAddress) );

	if (result != 0)
	{
		if (retry < 15) {
		    usleep(1000000);
		    retry++;
		    goto rebind;
		}
		return -2;
	}

	// Set Socket Options

//	lingerData.l_onoff = 1;
//	lingerData.l_linger = LINGERTIME;
//	result = setsockopt( this->listenSocket, SOL_SOCKET,  SO_LINGER, (char *) &lingerData, sizeof( lingerData ) );

	#ifdef _RTOS
	{
		#error _RTOS defined
		unsigned long			zero = 0;
		unsigned long			one  = 1;

		result = setsockopt( this->listenSocket, SOL_SOCKET,  SO_REUSESOCK, (char *) &on, sizeof( on ) );
		result = setsockopt( this->listenSocket, SOL_SOCKET, SO_NAGLE,  (char *) &zero, sizeof( zero ) );
		result = ioctlsocket(this->listenSocket, FIONBIO, &one );
	}
	#endif

	#ifdef OS_WIN32
		result = setsockopt( this->listenSocket, IPPROTO_TCP, TCP_NODELAY,  (char *) &on, sizeof( on ) );
	#endif

	#ifdef OS_POSIX
	{
		int	flags;

		result = setsockopt( this->listenSocket, IPPROTO_TCP, TCP_NODELAY,  (char *) &on, sizeof( on ) );

		#if 1
			flags = fcntl( this->listenSocket, F_GETFL, 0 );
			assert( flags >= 0);
			flags = fcntl( this->listenSocket, F_SETFL, flags | O_NONBLOCK );
			assert( flags >= 0);
		#endif
	}
	#endif

	// Listen to the write socket

	result = listen( listenSocket, 10 );		// Allocate a valid back log

	if (result != 0)
	{
		// Problem with the socket, time to bail out

		return -3;
	}

	// Everything is ok!

	initialized = 1;

	// Create our TCP listen thread

	this->thread = OS_CreateThread( (void *) CTCPListenerGlue, 0, this );

	if (this->thread == NULL)
	{
		Shutdown();
		return -4;
	}

	return 0;
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

void CTCPListener::Shutdown()
{
	initialized = 0;
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

void CTCPListener::ListenThread()
{
	int					result;
	int					length;
	FD_SET_T			f_read;
	int					error;
	struct	timeval		selectTimeOut;
	SOCKET				clientSocket;
	struct	sockaddr_in	clientAddress;

	while (this->initialized)
	{
		// Wait for something to accept

		FD_ZERO(&f_read);
		FD_SET( (SOCKET) listenSocket, &f_read);

		selectTimeOut.tv_usec = 0;
		selectTimeOut.tv_sec  = EXIT_TIMEOUT / 1000;

		result = select( listenSocket+1, &f_read, NULL, NULL, &selectTimeOut );

		if (result == 0)
		{
			continue;
		}
		else if (result < 0)
		{
			error = OS_GetLastSocketError();
			break;
		}

		// Try to accept the connection

		if ( FD_ISSET( listenSocket, &f_read ) )
		{
			length = sizeof( clientAddress );

			clientSocket = accept( listenSocket, (struct sockaddr *) &clientAddress, (socklen_t *) &length );

			#ifdef OS_POSIX
			{
				int on = 1;
		
				result = setsockopt( clientSocket, IPPROTO_TCP, TCP_NODELAY,  (char *) &on, sizeof( on ) );

				#if 0
					int flags = fcntl( clientSocket, F_GETFL, 0 );
					assert( flags >= 0);
					flags = fcntl( clientSocket, F_SETFL, flags | O_NONBLOCK );
					assert( flags >= 0);
				#endif
			}
			#endif

			if ( OS_IsSocketValid( clientSocket ) )
			{
				AcceptClientSocket( clientSocket, ntohl(clientAddress.sin_addr.S_ADDR), ntohs(clientAddress.sin_port) );
			}
		}
	}
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

void CTCPListener::AcceptClientSocket( SOCKET s, int NOTUSED(ipAddress), int NOTUSED(tcpPort))
{
	assert(0); // Should over ride this fuction !!!
	CloseSocket( s );
}

/*  --------------------------------------------------------------------*/
/** See TCPListener.h **/

void CTCPListener::CloseSocket(SOCKET s)
{
	closesocket( s );
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Glue code to get into the listen thread method.
 * 
 */

DWORD CTCPListenerGlue(LPVOID param)
{
	CTCPListener	*	pTCPListener = (CTCPListener *) param;

	pTCPListener->ListenThread();

	return 0;
}

