// NetConn library test

#include "pp/NetConn.h"
#include "stdio.h"
#include "assert.h"
#include <openssl/rc4.h>

int verbose = 0;
int tcpConnType;
int	udpGood = 0;
CNetConn_SSL_CTX *serverCtx;
char testString[] = "This is the test message";
#define LOCAL_HOST 0x7F000001

// Sub class listener classes

class TestTCPListener : public CTCPListener
{
public:
	TestTCPListener( int port ) : CTCPListener(port) {}
	~TestTCPListener() {};

	/**
	 *	@brief	Call back function. Called when a new tcp connection has
	 *  been accepted
	 *
	 *  This function must be over ridden to accept TCP connections.
	 *
	 * 	@param	s				The socket
	 *  @param	ipAddress		The client IP address
	 *  @param	tcpPort			The client TCP Port
	 */

	virtual void AcceptClientSocket(SOCKET s, int ipAddress, int tcpPort)
	{
		// Echo data
		CNetConn	*pNetConn;
		int 		result;		
		RC4_KEY		rc4Key;

		if (verbose)
			printf("AcceptClientSocket %08X %d\n",ipAddress, tcpPort );

		switch (tcpConnType)
		{
		case 0: // Socket
			{
				CNetConn_Socket *pSocket = new CNetConn_Socket(s);
				pNetConn = (CNetConn *) pSocket;
			}
			break;
		case 1: // RC4
			{
				CNetConn_RC4 *pRC4 = new CNetConn_RC4(s);
				RC4_set_key( &rc4Key, strlen(testString), (unsigned char *) testString );
				pRC4->SetRC4WriteKey( &rc4Key );
				pRC4->SetRC4ReadKey( &rc4Key );
				pNetConn = (CNetConn *) pRC4;
			}
			break;
		case 2: // SSL
			{
				CNetConn_SSL *pSSL = new CNetConn_SSL(s);
				result = pSSL->SSL_Accept(serverCtx);
				assert( result == 1);
				pNetConn = (CNetConn *) pSSL;
			}
			break;
		}
		char buffer[256];
		result = pNetConn->Read(buffer,strlen(testString)+1);
		assert( result != 0);
		if (verbose)
			printf("Server Reading : %s\n",buffer );
		result = pNetConn->Write(buffer,strlen(testString)+1);
		assert( result != 0);
		if (verbose)
			printf("Server writing\n" );
		if (verbose)
			printf("Server closing\n" );
		CloseSocket(s);
		delete pNetConn;
	}
};

class TestUDPListener : public CUDPListener
{
public:
	TestUDPListener( int port ) : CUDPListener(port) {}
	~TestUDPListener() {};

	/**
	 *	@brief	Call back function. Called when a new tcp connection has
	 *  been accepted
	 *
	 *  This function must be over ridden to accept TCP connections.
	 *
	 *  @param	ipAddress		The client IP address
	 *  @param	pData			The data received
	* 	@param	count			# of bytes
	 */

	virtual void ReceiveMessage(int ipAddress, char *pData, int count)
	{
		if (verbose)
			printf("UDP : %08X %d %s\n",ipAddress,count,pData );
		// Echo data
		int result = strcmp(pData,testString);
		assert(result == 0);
		ReturnMessage(pData,count);
		udpGood = 1;
	}
};


int main(int argc, char *argv[])
{
	CNetConn_Socket		socket;
	CNetConn_RC4		rc4;
	CNetConn_SSL		ssl;
	CNetConn_SSL_CTX	*clientCtx;
	CNetConn_UDP		udp(LOCAL_HOST,5000);
	RC4_KEY				rc4Key;
	int					result;
	char				buffer[256];

	// Create our servers

	TestTCPListener *tcpServer = new TestTCPListener(5000);
	TestUDPListener *udpServer = new TestUDPListener(5000);
	result = tcpServer->Initialize();
	assert( result == 0 );
	result = udpServer->Initialize();
	assert( result == 0 );

	// Test CNetConn_Socket

	if (verbose)
		printf("CNetConn_Socket Test\n");

	tcpConnType = 0;
//	memset(buffer,0,sizeof(buffer));
	result = socket.OpenSocket( LOCAL_HOST, 5000 );
	assert( result == 0 );
	result = socket.Write(testString,strlen(testString)+1);
	assert( result != 0 );
	result = socket.Read(buffer,strlen(testString)+1);
	assert( result != 0 );
	buffer[255] = 0;
	result = strcmp(buffer,testString);
	assert( result == 0 );
	socket.Shutdown();

	// Test CNetConn_RC4

	if (verbose)
		printf("CNetConn_RC4 Test\n");

	tcpConnType = 1;
//	memset(buffer,0,sizeof(buffer));
	result = rc4.OpenSocket( LOCAL_HOST, 5000 );
	assert( result == 0 );
	RC4_set_key( &rc4Key, strlen(testString), (unsigned char *) testString );
	rc4.SetRC4WriteKey(&rc4Key);
	rc4.SetRC4ReadKey(&rc4Key);
	result = rc4.Write(testString,strlen(testString)+1);
	assert( result != 0 );
	result = rc4.Read(buffer,strlen(testString)+1);
	assert( result != 0 );
	buffer[255] = 0;
	result = strcmp(buffer,testString);
	assert( result == 0 );
	rc4.Shutdown();

	// Test CNetConn_SSL

	// Initialize the SSL contexts

	result = CNetConn_SSL::SSL_Library_Init(SSL2_TXT_RC4_128_WITH_MD5);
	assert( result == 0 );
	result = CreatePrivateKeyFile("PrivateKey.pem");
	assert( result != 0 );
	result = CreateCertConfigFile("CertConfig.pem");
	assert( result != 0 );
	result = CreateCertificateFile("PrivateKey.pem","CertConfig.pem","Cert.pem");
	assert( result != 0 );

	serverCtx = new CNetConn_SSL_CTX("PrivateKey.pem","Cert.pem");
	clientCtx = new CNetConn_SSL_CTX();

	DestroyFile("PrivateKey.pem");
	DestroyFile("CertConfig.pem");
	DestroyFile("Cert.pem");

	tcpConnType = 2;
//	memset(buffer,0,sizeof(buffer));
	result = ssl.OpenSocket( LOCAL_HOST, 5000 );
	assert( result == 0 );
	result = ssl.SSL_Connect(clientCtx);
	assert( result == 1 );
	result = ssl.Write(testString,strlen(testString)+1);
	assert( result != 0 );
	result = ssl.Read(buffer,strlen(testString)+1);
	assert( result != 0 );
	buffer[255] = 0;
	result = strcmp(buffer,testString);
	assert( result == 0 );
	ssl.Shutdown();

	// Test UDP

	result = udp.Write(testString,strlen(testString)+1);
	assert( result > 0 );
	OS_Sleep(1000);
	assert(udpGood == 1);

	printf("PASS\n");

	return 0;
}
