/**	
 *	@file	UDPListener.h
 *	@brief	Implementation of the CUDPListener class.
 */

#include "pp/UDPListener.h"
#include "assert.h"
#ifdef OS_POSIX
#include <fcntl.h>
#endif

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

	// Timeout values (in milliseconds)

#define	EXIT_TIMEOUT			2000		// Must be evenly divisable by a thousand

	// misc.

#ifdef OS_POSIX
#define	S_ADDR		s_addr
#define	FD_SET_T	fd_set
#define	SOCKADDR	struct sockaddr
#else
#define	S_ADDR		S_un.S_addr
#define	FD_SET_T	struct fd_set
#define	socklen_t	int
#define	SHUT_RDWR	2
#endif

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

DWORD CUDPListenerGlue(LPVOID param);

/*----------------------------------------
 *	Static Data
 *--------------------------------------*/

/*----------------------------------------
 *	Code
 *--------------------------------------*/

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

CUDPListener::CUDPListener( int _port, int _maxUDPMsgSize )
{
	// ----------------------------------------
	// Init our member variables

	listenSocket	= INVALID_SOCKET;
	initialized	= 0;
	port		= _port;
	maxUDPMsgSize	= _maxUDPMsgSize;
	thread		= NULL;
	pFirst		= NULL;
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

CUDPListener::~CUDPListener()
{
	Shutdown();

	if (thread != NULL)
	{
		OS_WaitForThread( thread );
		OS_DeleteThread( thread );
		thread = NULL;
	}
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

int CUDPListener::Initialize()
{
	struct	sockaddr_in	socketAddress;	// Socket Address structure for the udp socket
	int		result;
	int		on = 1;
	
	listenSocket = socket( AF_INET, SOCK_DGRAM, 0 );

	if ( OS_IsSocketValid( listenSocket ) )
	{
		// Bind the socket to an address

		socketAddress.sin_family = AF_INET;
		socketAddress.sin_port = htons(port);
		socketAddress.sin_addr.S_ADDR = htonl(INADDR_ANY);

		result = setsockopt( listenSocket, SOL_SOCKET,  SO_REUSEADDR, (char *) &on, sizeof( on ) );

		result = bind( listenSocket, (SOCKADDR *) &socketAddress, sizeof(socketAddress) );

		if (result != 0)
			return -1;;

		// Make it non blocking

		#ifdef OS_POSIX
		{
			int	flags;

			flags = fcntl( listenSocket, F_GETFL, 0 );
			assert( flags >= 0);
			flags = fcntl( listenSocket, F_SETFL, flags | O_NONBLOCK );
			assert( flags >= 0);
		}
		#else
		{
			int				result;
			unsigned long	zero = 0;

			result = ioctlsocket(listenSocket, FIONBIO, &zero );
		}
		#endif
	}

	initialized = 1;

	// Create our TCP listen thread

	this->thread = OS_CreateThread( (void *) CUDPListenerGlue, 0, this );

	if (this->thread == NULL)
	{
		Shutdown();
		return -4;
	}

	return 0;
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

void CUDPListener::AddUDPHandler( CUDPHandler * pUDPHandler )
{
	if (pFirst == NULL)
	{
		pFirst = pUDPHandler;
		return;
	}

	CUDPHandler * p = pFirst;

	while (p->pNext != NULL)
		p = p->pNext;
	p->pNext = pUDPHandler;
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

void CUDPListener::Shutdown()
{
	initialized = 0;
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

void CUDPListener::ListenThread()
{
	int					result;
	int					length;
	struct	sockaddr_in	clientAddress;
	int					error;
	FD_SET_T			f_read;
	struct	timeval		selectTimeOut;
	char				*buffer = new char[maxUDPMsgSize];

	while (initialized)
	{
		// Wait for something to read

		FD_ZERO(&f_read);
		FD_SET( (SOCKET) listenSocket, &f_read);

		selectTimeOut.tv_usec = 0;
		selectTimeOut.tv_sec  = EXIT_TIMEOUT / 1000;

#ifdef OS_POSIX
sig_restart:
#endif

		result = select( listenSocket+1, &f_read, NULL, NULL, &selectTimeOut );

		if (result == 0)
		{
			continue;
		}
		else if (result < 0)
		{
			error = OS_GetLastSocketError();

		#ifdef OS_POSIX
		// select() sets E_INTR when interrupted by a signal, even when
		// SA_RESTART is used with sigaction(), so we better be safe
			if (error == EINTR)
				goto sig_restart;
		#endif

			//DBLog1(("Select err in UDP Listen %d",error));
			break;
		}

		// Read the data

		length = sizeof( clientAddress );

		result = recvfrom( listenSocket, buffer, maxUDPMsgSize, 0, (struct sockaddr *) &clientAddress, (socklen_t *) &length );

		if (result < 1)
		{
			error = OS_GetLastSocketError();
			//DBLog1(("%d UDP Receive error %d from %08X\n",DBTickCount(),error,ntohl(clientAddress.sin_addr.S_ADDR)));
			continue;
		}

		//DBLog3(("%d UDP Request from %08X\n",DBTickCount(),ntohl(clientAddress.sin_addr.S_ADDR)));

		// Send the UDP Message to the receiver

		pLastClientAddress = (void *) &clientAddress;

		ReceiveMessage( ntohl(clientAddress.sin_addr.S_ADDR), buffer, result );
	}

	if (initialized)
	{
		// We got out of the loop because of an error...

		error = OS_GetLastSocketError();
		assert(0);
	}

	if (buffer != NULL)
		delete[] buffer;
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

void CUDPListener::ReceiveMessage( int ipAddress, char * pData, int count )
{
	bool result = false;
	CUDPHandler * pHandler = pFirst;

	while (pHandler && !result)
	{
		result = pHandler->ReceiveUDPMessage(this,ipAddress,pData,count);
		pHandler = pHandler->pNext;
	}
}

/*  --------------------------------------------------------------------*/
/** See UDPListener.h **/

int CUDPListener::ReturnMessage( char * pData, int count )
{
	return sendto( listenSocket, (char *) pData, count, 0, (struct sockaddr *) pLastClientAddress, sizeof( struct	sockaddr_in ) );
}

/*  --------------------------------------------------------------------*/
/** 
 *	@brief	Glue code to get into the listen thread method.
 * 
 */

DWORD CUDPListenerGlue(LPVOID param)
{
	CUDPListener	*	pUDPListener = (CUDPListener *) param;

	pUDPListener->ListenThread();

	return 0;
}


