/**	
 *	@file	NetConn_SSL.h
 *	@brief	CNetConn derived class for unencrypted socket communications
 *
 */

#ifndef _NetConn_SSL_h_
#define _NetConn_SSL_h_

#include	"pp/NetConn_Socket.h"

#include <openssl/rsa.h>       /* SSLeay stuff */
#include <openssl/crypto.h>
#include <openssl/x509.h>
#include <openssl/pem.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/rc4.h>
#include <openssl/evp.h>

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

class CNetConn_SSL_CTX;

/*----------------------------------------
 *	CNetConn_Socket Class
 *--------------------------------------*/

class CNetConn_SSL : public CNetConn_Socket
{
public:

	CNetConn_SSL();
	CNetConn_SSL( SOCKET s );
	CNetConn_SSL( CNetConn_Socket * s );
	virtual ~CNetConn_SSL();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	OpenSSL Library Initialization
	 * 
	 *  Must be called only once before any CNetConn_SSL objects are created.
	 *  @see Certificate.cpp for function that create key and cert files.
	 *
	 *	@param	pPrivateKeyFile	Name of the Private key file (PEM format)
	 *	@param	pCertFile		Name of the certificate file (PEM format)
	 *	@param	pCipherList		The cipher list defined by OpenSSL (e.g. SSL2_TXT_RC4_128_WITH_MD5 ...)
	 *	@return					0 = no error
	 */

	static int SSL_Library_Init( const char * _pDefaultCipherList );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	OpenSSL Library Shutdown
	 * 
	 *  Must be called only once after all SSL connections are done and program
	 *  is exiting
	 *
	 */

	static void CNetConn_SSL::SSL_Shutdown();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	SSL_Connect does the client side SSL negociations
	 * 
	 *  @see CNetConn_SSL_CTX
	 *
	 *	@param	ctx				The SSL_CTX container
	 *	@return					1 = accepted, 0 = not accepted, -1 = error
	 */

	int SSL_Connect(CNetConn_SSL_CTX * ctx);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	SSL_Accept does the server side SSL negociations
	 *
	 *  @see CNetConn_SSL_CTX
	 *
	 *	@param	ctx				The SSL_CTX container
	 *	@return					1 = accepted, 0 = not accepted, -1 = error
	 */

	int SSL_Accept(CNetConn_SSL_CTX * ctx);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Functions to Get the SSL "object"
	 * 
	 */

	SSL * GetSSL();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Write data to the network connection
	 *
	 *	@param	pData			Ptr to the data to write
	 *	@param	count			# of bytes to write
	 *	@return					true = no errors, false = write failed
	 */

	virtual	BOOL Write( const void * pData, int count );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Read data from the network connection
	 *
	 *	@param	pData		  	Ptr to the buffer
	 *	@param	count			# of bytes to read
	 *	@return					true = no errors, false = read failed
	 */

	virtual	BOOL Read( void * pData, int count );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Stops the connection
	 * 
	 * 	Any Reads or Writes will be terminated and the connection closed
	 *
	 */

	virtual	void Shutdown();

protected:

	int SSL_Wants_Something( int error, int timeout );
	int Do_SSL_shutdown();

	SSL	 		*	pSSL;		  	///< The SSL object
};


/*----------------------------------------
 *	CNetConn_SSL_CTX Class
 *--------------------------------------*/

class CNetConn_SSL_CTX
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Construct a server context
	 * 
	 *	@param	pPrivateKeyFile	Name of the Private key file (PEM format)
	 *	@param	pCertFile		Name of the certificate file (PEM format)
	 *	@param	pCipherList		The cipher list defined by OpenSSL (e.g. SSL2_TXT_RC4_128_WITH_MD5 ...)
	 * 							If pCipherList == NULL, then the default CipherList will be used
	 * 							@see CNetConn::SSL_Library_Init
	 */

	CNetConn_SSL_CTX( const char * pPrivateKeyFile, const char * pCertFile, const char * pCipherList = NULL );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Construct a client context
	 * 
	 *	@param	pPrivateKeyFile	Name of the Private key file (PEM format)
	 * 							If pCipherList == NULL, then the default CipherList will be used
	 * 							@see CNetConn::SSL_Library_Init
	 */

	CNetConn_SSL_CTX( const char * pCipherList = NULL );


	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Get the SSL_CTX
	 * 
	 * @return					The SSL_CTX or NULL if there was an error
	 * 
	 */

	SSL_CTX * GetCTX();

	SSL_METHOD	*	sslMethod;		///< The SSL method(s) we will use
	SSL_CTX		*	sslCtx;			///< The SSL context
};

#endif // _NetConn_SSL_h_
