/*-------------------------------------------------------------------------------

	RDM_EventCode.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager CRDM_EventCode and CRDM_EventCodeSet class

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<string.h>
#include	"pp/CmdLineParse.h"
#include	"pp/RDM_EventCode.h"

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

	// List of event names and there numeric IDs
	// NULL terminated list

static RDM_EVENT_DEFINITION	rdmEventList[] = 
{
	{	"Invalid",						RDM_EC_Invalid				},

	// Interface Events

	{	"Modem_Connect",				RDM_EC_Modem_Connect	},
	{	"Modem_Disconnect",				RDM_EC_Modem_Disconnect	},

	// Unlogged System Events

	{	"Heart_Beat",					RDM_EC_Heart_Beat	},
	{	"Progress_Update",				RDM_EC_Progress_Update },

	// System Events

	{	"System_Startup",				RDM_EC_System_Startup	},
	{	"System_Shutdown",				RDM_EC_System_Shutdown	},
	{	"System_Init_Error",			RDM_EC_System_Init_Error	},
	{	"System_Fatal_Error",			RDM_EC_System_Fatal_Error	},
	{	"System_Run_Error",				RDM_EC_System_Run_Error	},
	{	"System_General",				RDM_EC_System_General	},
	{	"System_Reset",				    RDM_EC_System_Reset	},
	{	"System_Factory_Reset",			RDM_EC_System_Factory_Reset	},
    {	"System_Start_Management",		RDM_EC_System_Start_Management },
    {	"System_Stop_Management",		RDM_EC_System_Stop_Management },
	{   "System_Progress_Begin",		RDM_EC_System_Progress_Begin },
	{   "System_Progress_End",			RDM_EC_System_Progress_End },

	// Device Events

	{	"Device_Added",					RDM_EC_Device_Added	},
	{	"Device_Changed",				RDM_EC_Device_Changed	},
	{	"Device_Deleted",				RDM_EC_Device_Deleted	},
	{	"Device_Change_Request",		RDM_EC_Device_Change_Request	},
	{	"Device_Setting_Change",		RDM_EC_Device_Setting_Change	},
	{	"Device_Setting_Change_Request",RDM_EC_Device_Setting_Change_Request	},
	{	"Device_Status_Changed",		RDM_EC_Device_Status_Changed	},
	{	"Device_Key_Event",             RDM_EC_Device_Key_Event           },
	{	"Device_Hardware_Event",        RDM_EC_Device_Hardware_Event           },
	{	"Device_Update_Started",		RDM_EC_Device_Update_Started },
	{	"Device_Update_Completed",		RDM_EC_Device_Update_Completed },
	{	"Device_Config_Backup",			RDM_EC_Device_Config_Backup },
	{	"Device_Config_Restore",		RDM_EC_Device_Config_Restore },
	{   "PowerSupply_Status_Changed",	RDM_EC_Device_PowerSupply_Status_Changed },
	{	"Device_Reset",					RDM_EC_Device_Reset },

	// Port Events
	
	{	"Port_Added",					RDM_EC_Port_Added	},
	{	"Port_Changed",					RDM_EC_Port_Changed	},
	{	"Port_Deleted",					RDM_EC_Port_Deleted	},
	{	"Port_Change_Request",			RDM_EC_Port_Change_Request	},
	{	"Port_Status_Changed",			RDM_EC_Port_Status_Changed	},

	// User Events
	
	{	"User_Added",					RDM_EC_User_Added	},
	{	"User_Changed",					RDM_EC_User_Changed	},
	{	"User_Deleted",					RDM_EC_User_Deleted	},
	{	"User_Change_Request",			RDM_EC_User_Change_Request	},
	{	"Usre_Password_Changed",		RDM_EC_User_Password_Changed },

	// Group Events
	
	{	"Group_Added",					RDM_EC_Group_Added	},
	{	"Group_Changed",				RDM_EC_Group_Changed	},
	{	"Group_Deleted",				RDM_EC_Group_Deleted	},
	{	"Group_Change_Request",			RDM_EC_Group_Change_Request	},

	// Access Events
	
	{	"Access_Login",					RDM_EC_Access_Login	},
	{	"Access_Logout",				RDM_EC_Access_Logout	},
	{	"Access_Login_Failed",			RDM_EC_Access_Login_Failed	},
	{	"Access_Connection_Lost",		RDM_EC_Access_Connection_Lost	},
	{	"Access_Connection_Timeout",	RDM_EC_Access_Connection_Timeout	},
	{	"Access_Connection_Denied",		RDM_EC_Access_Connection_Denied	},
	{	"Access_Console_Login",			RDM_EC_Access_Console_Login	},
	{	"Access_Console_Logout",		RDM_EC_Access_Console_Logout	},
	{	"Access_Wrong_IP",				RDM_EC_Access_Wrong_IP	},
	{	"Access_Port_Connect",			RDM_EC_Access_Port_Connect	},
	{	"Access_Port_Disconnect",		RDM_EC_Access_Port_Disconnect	},

    {   "Path_Added",                   RDM_EC_Path_Added },
    {   "Path_Changed",                 RDM_EC_Path_Changed },
    {   "Path_Deleted",                 RDM_EC_Path_Deleted },
    {   "Path_Change_Request",          RDM_EC_Path_Change_Request },
    {   "Path_Status_Changed",          RDM_EC_Path_Status_Changed },

	{	NULL,						0	} // End of list
};

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CRDM_EventCode
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRDM_EventCode::CRDM_EventCode
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	Set( RDM_EC_Invalid );
}

//--------------------------------------------------------------------------------
//
	CRDM_EventCode::CRDM_EventCode
	(
		int		eventID					// The numeric event ID
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	Set( eventID );
}

//--------------------------------------------------------------------------------
//
	CRDM_EventCode::~CRDM_EventCode
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
}

//--------------------------------------------------------------------------------
//
	int									// 0 for failure, 1 for success
	CRDM_EventCode::Set
	(
		int		eventID					// The numeric event ID
	)
//
//	Sets the event to the event represented by the number <eventID>
//
//------------------------------------------------------------------------------//
{
	// Check for out of range

	if (eventID > RDM_EC_MAX_CODE)
	{
		eventID = RDM_EC_Invalid;
	}

	// Set the ID

	id = eventID;
	index = id/32;
	mask = 1 << (id % 32);

	return id == RDM_EC_Invalid ? 0 : 1;
}

//--------------------------------------------------------------------------------
//
	int									// 0 for failure, 1 for success
	CRDM_EventCode::Set
	(
		const char *pEventName				// Ptr to the event name
	)
//
//	Sets the event to the event represented by the name <pEventName>
//
//------------------------------------------------------------------------------//
{
	int	eventID;

	// Find the name in the list

	eventID = GetIDByName( pEventName );

	// Set the ID

	id = eventID;
	index = id/32;
	mask = 1 << (id % 32);

	return id == RDM_EC_Invalid ? 0 : 1;
}

//--------------------------------------------------------------------------------
//
	int									// event ID
	CRDM_EventCode::GetID
	(
	)
//
//	Returns the numeric ID of this event
//
//------------------------------------------------------------------------------//
{
	return id;
}

//--------------------------------------------------------------------------------
//
	int									// event ID
	CRDM_EventCode::GetIDByName
	(
		const char	*pEventName				// Ptr to the event name
	)
//
//	Returns the numeric ID of this event - No wild cards are accepted
//
//------------------------------------------------------------------------------//
{
	int	x;

	for (x=0;rdmEventList[x].pName != NULL;x++)
	{
		if (strcmp(pEventName,rdmEventList[x].pName) == 0)
			return rdmEventList[x].id;
	}

	return RDM_EC_Invalid;
}

//--------------------------------------------------------------------------------
//
	int									// event ID
	CRDM_EventCode::GetIDByWildCardName
	(
		const char	*pEventName,			// Ptr to the event name
		int		lastID					// id from last GetID or 0 for first search
	)
//
//	Repeated calls to this function returns all of the events that match the 
//	<pEventName> with the * and ? wild card characters
//	On the first call, lastID should be 0. on subsequent calls, lastId should be
//	the value last returned by GetIDByWildCardName()
//
//------------------------------------------------------------------------------//
{
	int	x,y;

	// Go to the element of the last serach

	for (x=0;rdmEventList[x].pName != NULL && rdmEventList[x].id != lastID;x++)
		;

	x++; // skip it

	// Now find the next one

	for ( ;rdmEventList[x].pName != NULL;x++)
	{
		for (y=0; pEventName[y] && rdmEventList[x].pName[y]; y++)
		{
			if (pEventName[y] == '?')
				continue;

			if (pEventName[y] == '*')
				return rdmEventList[x].id;
			
			if (pEventName[y] != rdmEventList[x].pName[y])
				break;
		}

		if (pEventName[y] == 0 && rdmEventList[x].pName[y] ==0)
			return rdmEventList[x].id;
	}

	return RDM_EC_Invalid;
}

//--------------------------------------------------------------------------------
//
	const char *						// Ptr to the event name or NULL if
	CRDM_EventCode::GetName
	(
	)
//
//	Returns the name of the this event
//
//------------------------------------------------------------------------------//
{
	return GetName( id );
}

//--------------------------------------------------------------------------------
//
	const char *						// Ptr to the event name or NULL if
	CRDM_EventCode::GetName
	(
		int		eventID					// Numeric ID of event
	)
//
//	Returns the name of the specified event
//
//------------------------------------------------------------------------------//
{
	int	x;

	for (x=0;rdmEventList[x].pName != NULL;x++)
	{
		if (rdmEventList[x].id == eventID)
			return rdmEventList[x].pName;
	}

	return NULL;
}

//--------------------------------------------------------------------------------
//									CRDM_EventCodeSet
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CRDM_EventCodeSet::CRDM_EventCodeSet
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	ClearEvents();
}

//--------------------------------------------------------------------------------
//
	CRDM_EventCodeSet::~CRDM_EventCodeSet
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
}

//--------------------------------------------------------------------------------
//
	int									// # of events added
	CRDM_EventCodeSet::AddEvents
	(
		const char	* pEventNames			// Space or Comma separated event name list
										// each name can contain the * and ? wildcards
	)
//
//	Adds events to the event set.
//
//------------------------------------------------------------------------------//
{
	char			token[RDM_EC_MAX];
	const char			*p;
	int				count=0;
	int				id = 0;

	p = pEventNames;

	do
	{
		p = GetNextToken( p, token, RDM_EC_MAX, 0, " ," );

		if (token[0] == 0)
			break;

		do
		{
			id = CRDM_EventCode::GetIDByWildCardName(token, id);

			if (id != RDM_EC_Invalid)
				count += this->AddEvent( id );

		} while (id != RDM_EC_Invalid);

	} while (1);

	return count;
}

//--------------------------------------------------------------------------------
//
	int									// # of events removed
	CRDM_EventCodeSet::RemoveEvents
	(
		const char	* pEventNames			// Space or Comma separated event name list
										// each name can contain the * and ? wildcards
	)
//
//	Removes events from the event set.
//
//------------------------------------------------------------------------------//
{
	char			token[RDM_EC_MAX];
	const char			*p;
	int				count = 0;
	int				id = 0;

	p = pEventNames;

	do
	{
		p = GetNextToken( p, token, RDM_EC_MAX, 0, " ," );

		if (token[0] == 0)
			break;

		do
		{
			id = CRDM_EventCode::GetIDByWildCardName(token, id);

			if (id != RDM_EC_Invalid)
				count += this->RemoveEvent( id );

		} while (id != RDM_EC_Invalid);

	} while (1);

	return count;
}

//--------------------------------------------------------------------------------
//
	int									// # of events added
	CRDM_EventCodeSet::AddEvent
	(
		CRDM_EventCode	*ec				// Event code to add to the set
	)
//
//	Adds an event to the event set.
//
//------------------------------------------------------------------------------//
{
	set[ec->index] |= ec->mask;
	return 1;
}

//--------------------------------------------------------------------------------
//
	int									// # of events added
	CRDM_EventCodeSet::RemoveEvent
	(
		CRDM_EventCode	*ec				// Event code to remove from the set
	)
//
//	Removes an event to the event set.
//
//------------------------------------------------------------------------------//
{
	set[ec->index] &= ~ec->mask;
	return 1;
}

//--------------------------------------------------------------------------------
//
	int									// # of events added
	CRDM_EventCodeSet::AddEvent
	(
		int		id						// Event id to add to the set
	)
//
//	Adds an event to the event set.
//
//------------------------------------------------------------------------------//
{
	CRDM_EventCode	ec;

	ec.Set( id );
	return this->AddEvent( &ec );
}

//--------------------------------------------------------------------------------
//
	int									// # of events added
	CRDM_EventCodeSet::RemoveEvent
	(
		int		id						// Event id to remove from the set
	)
//
//	Removes an event to the event set.
//
//------------------------------------------------------------------------------//
{
	CRDM_EventCode	ec;

	ec.Set( id );
	return this->RemoveEvent( &ec );
}

//--------------------------------------------------------------------------------
//
	void
	CRDM_EventCodeSet::ClearEvents
	(
	)
//
//	Removes all events from the event set.
//
//------------------------------------------------------------------------------//
{
	int	x;

	for (x=0;x<RDM_EC_MAX_ARRAY;x++)
		set[x] = 0;
}

//--------------------------------------------------------------------------------
//
	int									// !0 if event is in the set
	CRDM_EventCodeSet::IsEvent
	(
		CRDM_EventCode	*ec				// Ptr to the event code to test
	)
//
//	Tests to see if the specific event is in the event set.
//
//------------------------------------------------------------------------------//
{
	return set[ec->index] & ec->mask;
}

//--------------------------------------------------------------------------------
//
	int									// !0 if event is in the set
	CRDM_EventCodeSet::IsEvent
	(
		int		eventID					// Numeric ID of the event
	)
//
//	Tests to see if the specific event is in the event set.
//
//------------------------------------------------------------------------------//
{
	CRDM_EventCode ec;

	ec.Set( eventID );
	return this->IsEvent( &ec );
}

//--------------------------------------------------------------------------------
//
	int									// !0 if event is in the set
	CRDM_EventCodeSet::IsEvent
	(
		const char *	pEventName				// Ptr to the event name
	)
//
//	Tests to see if the specific event is in the event set.
//
//------------------------------------------------------------------------------//
{
	CRDM_EventCode ec;

	ec.Set( pEventName );
	return this->IsEvent( &ec );
}

