/*--------------------------------------------------------------------------

	RDM_Session.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager Protocol Session service handler class (CRDM_Session)

	NOTE: This is not the session manager, but just the RDMP command interpretter
	for the Session service.

---------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<pp/syms.h>
#include	<pp/Hash32.h>
#include	<pp/RDM_Session.h>
#include	<pp/SXDB_Parse_Table.h>
#include	<pp/CmdLineParse.h>
#include	<pp/SessionReferralObject.h>

//----------------------------------------
//				Equates
//----------------------------------------

	// Function IDs

enum
{
	CMD_Version,
	CMD_GetSessionID,
	CMD_QueryPermission,
	CMD_Open,
	CMD_Close
};

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------
extern int									// <0 = error (port was not found)
	DDA_UpdateResourceStatus
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// ID of the Path
		int		status					// Path status (0,1,2)
	);

extern int
    DDA_ReleaseResources
	(
		CRDM	*pRDM,					// Ptr to RDM
		const char	*pID,					// ID of the path
		int		status					// Port status (0,1,2)
	);


//----------------------------------------
//				Static Data
//----------------------------------------

	// Function List

static TOKEN	functions[] = 
{
	{ "Version",	CMD_Version },
	{ "GetSessionID",CMD_GetSessionID },
	{ "Permission", CMD_QueryPermission },
	{ "Open",		CMD_Open },
	{ "Close",		CMD_Close },
	{ NULL,			0 } // End of list
};

//----------------------------------------
//		GetSessionID Function Parse Table
//----------------------------------------

#define	PT_STRUCT	int
PT_BEGIN	( "GetSessionID",	getIDTable,		PT_UNKNOWN_OK )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Open Function Parse Table
//----------------------------------------

typedef struct
{
	const char	*pResourceUsed;
} OPEN_DATA;

#define	PT_STRUCT	OPEN_DATA
PT_BEGIN	( "Open",		openTable,		PT_UNKNOWN_OK )
PT_ELEM		( "ResourceUsed",	pResourceUsed,		0,	PT_STRING_PTR | PT_OPTIONAL )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Close Function Parse Table
//----------------------------------------

typedef struct
{
	const char	*pSessionID;
} CLOSE_DATA;

#define	PT_STRUCT	CLOSE_DATA
PT_BEGIN	( "Close",		closeTable,		PT_UNKNOWN_OK )
PT_ELEM		( "SessionID",	pSessionID,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		Close Function Parse Table
//----------------------------------------

typedef struct
{
	const char	*pAccessNode;
	const char	*pPermID;
} PERM_DATA;

#define	PT_STRUCT	PERM_DATA
PT_BEGIN	( "Permission",		permQueryTable,		PT_UNKNOWN_OK )
PT_ATT		( "AccessNode",		pAccessNode,	0,	PT_STRING_PTR )
PT_ROOT_DATA(					pPermID,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Code
//----------------------------------------

//---------------------------------------------------------------------------
//									CRDM_Session
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CRDM_Session::CRDM_Session
	(
		class CRDM	*	pNewRDM		// Ptr to the RDM object
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	// Initialize everthing

	pName		= "Session";
	hashCode	= Hash32( 0, pName );
	pRDM		= pNewRDM;

	// Install our service

	pRDM->rdmp.AddService( this );
}

//---------------------------------------------------------------------------
//
	CRDM_Session::~CRDM_Session
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
}

//---------------------------------------------------------------------------
//
	int									// 0 or error code if request failed
	CRDM_Session::Command
	(
		CSession	*	pUserSession,	// User session
		CSXDB_Node	*	pRootNode,		// Ptr to root node of our functions
		CSIO		*	pResponse		// The response -
										// Contains the return value on exit
	)
//
//	Processes an RDMP request for the database service & returns the response
//
//---------------------------------------------------------------------------
{
	const char		*pFunctionName;
	int			cmdID;
	int			result;
	CSXDB_Node	*pNode;

	// Get ptr to first function node

	pNode = this->GetFirstFunction( pRootNode );

	while (pNode != NULL)
	{
		// Function Name

		pFunctionName = pNode->GetName();

		// Find function

		cmdID = GetTokenValue( pFunctionName, functions );

		// Exexute the function

		if (cmdID != CMD_QueryPermission)
			pResponse->Printf("<%s>",pFunctionName);

		switch (cmdID)
		{
			case CMD_Version:
				pResponse->Puts("1");
				break;

			case CMD_GetSessionID:
				this->GetSessionID( pUserSession, pNode, pResponse );
				break;

			case CMD_Open:
				this->Open( pUserSession, pNode, pResponse );
				break;

			case CMD_Close:
				this->Close( pUserSession, pNode, pResponse );
				break;

			case CMD_QueryPermission:
				result = this->QueryPermission( pUserSession, pNode, pResponse );
				break;

			default:
				this->PrintError( pResponse, RDM_ERROR_FUNCTION_NOT_FOUND );
				break;

		}

		if (cmdID != CMD_QueryPermission)
			result = pResponse->Printf("</%s>",pFunctionName);

		pNode = this->GetNextFunction(pNode);
	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Session::GetSessionID
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the GetSessionID function
//
//--------------------------------------------------------------------------
{
	int				result;

	// Parse the function

	result = SXDB_PT_Get(pNode, getIDTable, NULL, 0);

	if (result != 0 || pUserSession == NULL)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	result = pResponse->Printf("<SessionID>%s</SessionID> <SessionKey>%s</SessionKey>",
				pUserSession->GetSessionID(),pUserSession->GetSessionKey());

	
	return result;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Session::Open
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Open function
//
//--------------------------------------------------------------------------
{
	int				result;
	CSession		*pSession;
    int     condition = 0xFFFFFFFF;
	OPEN_DATA		openData;

	// Check permissions

	if (pUserSession->GetNodePermission("DeviceManagement") != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	// Parse the function

	memset( &openData, 0, sizeof(openData) );

	result = SXDB_PT_Get(pNode, openTable, &openData, condition);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

    // Check if the resource is already used.

    BOOL resUsed = FALSE;

    if (openData.pResourceUsed != NULL )
        resUsed = DEP_GetSessionManager()->IsResourceUsed( openData.pResourceUsed );

    if (resUsed == TRUE)
        return PrintError( pResponse, RDM_ERROR_RESOURCE_IN_USE );

	// Do it

	pSession = DEP_GetSessionManager()->NewSession();

	if (pSession == NULL)
		return PrintError( pResponse, RDM_ERROR_MEMORY_ERROR );
	else
		DEP_SetReferralUserObject ( pSession );

	// Attach a Referral session object to this session with a timeout

	AttatchSessionReferralObject( pSession, RDM_REFERRAL_SESSION_TIMEOUT );

    // Reserve the resource requested by the client

    if (openData.pResourceUsed != NULL )
    {
        pSession->AddUsedResource (openData.pResourceUsed);

        // Update the path status you are reserving.
        DDA_UpdateResourceStatus (this->pRDM, openData.pResourceUsed, 0);
    }

	result = pResponse->Printf("<SessionID>%s</SessionID> <SessionKey>%s</SessionKey>",
				pSession->GetSessionID(),pSession->GetSessionKey());

    pSession->Release();
	
	return result;
}



//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Session::Close
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Close function
//
//--------------------------------------------------------------------------
{
	int				result;
	CLOSE_DATA		closeData;
	CSession	*	pSession;

	// Check permissions

	if (pUserSession->GetNodePermission("UserManagement") != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	// Parse the function

	result = SXDB_PT_Get(pNode, closeTable, &closeData, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	pSession = DEP_GetSessionManager()->GetSession(closeData.pSessionID);

	if (pSession != NULL)
	{
        // Update the resource status before closing the session

        DDA_ReleaseResources(this->pRDM, closeData.pSessionID, 1);

        // close and release the session

		pSession->Close();
		result = pResponse->Printf("<Close>1</Close>");

        pSession->Release();
	}
	else
		result = pResponse->Printf("<Close>0</Close>");

	return result;
}

//---------------------------------------------------------------------------
//
	int								// I/O error on output
	CRDM_Session::QueryPermission
	(
		CSession	*pUserSession,	// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the GetSessionID function
//
//--------------------------------------------------------------------------
{
	int				result;
	PERM_DATA		perm;
	DWORD			type;
	const char		*	pTypeName;

	memset(&perm,0,sizeof(perm));

	// Parse the function

	result = SXDB_PT_Get(pNode, permQueryTable, &perm, 0);

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	if (perm.pAccessNode != NULL)
		type = pUserSession->GetPermission( perm.pPermID, perm.pAccessNode );
	else
		type = pUserSession->GetNodePermission( perm.pPermID );

	switch (type)
	{
		default:
		case TR_PERM_NONE: pTypeName = "none"; break;
		case TR_PERM_READONLY: pTypeName = "readonly"; break;
		case TR_PERM_READWRITE: pTypeName = "readwrite"; break;
	}

	if (perm.pAccessNode != NULL)
	{
		result = pResponse->Printf("<Permission AccessNode=\"%s\" AccessType=\"%s\">%s</Permission>",
									perm.pAccessNode,
									pTypeName,
									perm.pPermID
								  );
	}
	else
	{
		result = pResponse->Printf("<Permission AccessType=\"%s\">%s</Permission>",
									pTypeName,
									perm.pPermID
								  );
	}
	
	return result;
}

//---------------------------------------------------------------------------
//				Methods that must be overriden by the project
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CSessionManager *					// Ptr to the CSessionManager object
	CRDM_Session::DEP_GetSessionManager
	(
	)
//
//	Returns a ptr to the CSessionManager object
//	This default implementation returns a ptr to the global gSessionManager
//
//---------------------------------------------------------------------------
{
	extern	CSessionManager *gSessionManager;

	return gSessionManager;
}

//---------------------------------------------------------------------------
//
    void
    CRDM_Session::DEP_SetReferralUserObject
    (
		CSession * NOTUSED(pSession)
    )
//
//  Derived class implements this function
//
//---------------------------------------------------------------------------
{
    return;
}

