/*--------------------------------------------------------------------------

	RDM_System.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Raritan Device Manager Protocol System service handler class (CRDM_System)

---------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<pp/syms.h>
#include	<pp/Hash32.h>
#include	<pp/RDM.h>
#include	<pp/RDM_System.h>
#include	<pp/SXDB_Parse_Table.h>
#include	<pp/CmdLineParse.h>

//----------------------------------------
//				Equates
//----------------------------------------

	// Function IDs

enum
{
	CMD_Version,
	CMD_Restart,
	CMD_Shutdown,
	CMD_StartManagement,
	CMD_StopManagement,
	CMD_GetTime,
	CMD_SetTime,
	CMD_IPR_StopAllVideo
};


#define RDM_SYSTEM_VERSION		1

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

	// Function List

static TOKEN	functions[] = 
{
	{ "Version",	CMD_Version },
	{ "Restart",	CMD_Restart },
	{ "Shutdown",	CMD_Shutdown },
	{ "StartManagement",CMD_StartManagement },
	{ "StopManagement",CMD_StopManagement },
	{ "GetTime",CMD_GetTime },
	{ "SetTime",CMD_SetTime },
	{ "IPR_StopAllVideo",CMD_IPR_StopAllVideo },
	{ NULL,			0 } // End of list
};

//----------------------------------------
//		StartManagement Function Parse Table
//----------------------------------------

typedef	struct
{
	int		enable;
	const char	*pClusterID;
	const char	*pCCUser;
	const char	*pCCPassword;
	int		ipAddress;
	int		ipPort;
	int		heartbeatTime;
	const char	*pEventKey;
	int		ccTimeout;
} START_DATA;

#define	PT_STRUCT	START_DATA
PT_BEGIN	( "StartManagement",startTable,		PT_NO_UNKNOWN )
PT_ELEM		( "ClusterID",		pClusterID,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "CCUser",			pCCUser,		0,	PT_STRING_PTR | PT_REQUIRED  )
PT_ELEM		( "CCPassword",		pCCPassword,	0,	PT_STRING_PTR | PT_REQUIRED  )
PT_ELEM		( "IPAddress",		ipAddress,		0,	PT_IP | PT_REQUIRED )
PT_ELEM		( "IPPort",			ipPort,			0,	PT_INT )
PT_ELEM		( "HeartbeatTime",	heartbeatTime,	0,	PT_INT | PT_REQUIRED )
PT_ELEM		( "EventKey",		pEventKey,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Timeout",		ccTimeout,		0,	PT_INT )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//		CommandCenter Parse Table
//----------------------------------------

#define	PT_STRUCT	START_DATA
PT_BEGIN	( "CommandCenter",	ccTable,		PT_NO_UNKNOWN )
PT_ELEM		( "Enable",			enable,			0,	PT_INT | PT_REQUIRED )
PT_ELEM		( "ClusterID",		pClusterID,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "CCUser",			pCCUser,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "CCPassword",		pCCPassword,	0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "IPAddress",		ipAddress,		0,	PT_IP | PT_REQUIRED )
PT_ELEM		( "IPPort",			ipPort,			0,	PT_INT )
PT_ELEM		( "HeartbeatTime",	heartbeatTime,	0,	PT_INT | PT_REQUIRED )
PT_ELEM		( "EventKey",		pEventKey,		0,	PT_STRING_PTR | PT_REQUIRED )
PT_ELEM		( "Timeout",		ccTimeout,		0,	PT_INT )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Code
//----------------------------------------

//---------------------------------------------------------------------------
//								CRDM_System
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CRDM_System::CRDM_System
	(
		class CRDM	*	pNewRDM			// Ptr to the RDM object
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	// Initialize everthing

	pName		= "System";
	hashCode	= Hash32( 0, pName );
	pRDM		= pNewRDM;

	version = RDM_SYSTEM_VERSION;

	// Install our service

	pRDM->rdmp.AddService( this );
}

//---------------------------------------------------------------------------
//
	CRDM_System::~CRDM_System
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
}

//---------------------------------------------------------------------------
//
	int									// 0 or error code if request failed
	CRDM_System::Command
	(
		CSession	*	pUserSession,	// User session
		CSXDB_Node	*	pRootNode,		// Ptr to root node of our functions
		CSIO		*	pResponse		// The response -
										// Contains the return value on exit
	)
//
//	Processes an RDMP request for the database service & returns the response
//
//---------------------------------------------------------------------------
{
	const char		*pFunctionName;
	int			cmdID;
	int			result;
	CSXDB_Node	*pNode;

	// Get ptr to first function node

	pNode = this->GetFirstFunction( pRootNode );

	while (pNode != NULL)
	{
		// Function Name

		pFunctionName = pNode->GetName();

		// Find function

		cmdID = GetTokenValue( pFunctionName, functions );

		// Exexute the function

		pResponse->Printf("<%s>",pFunctionName);

		switch (cmdID)
		{
			case CMD_Version:
				char buf[10];
				sprintf( buf,"%d", GetVersion() );
				pResponse->Puts(buf);
				break;

			case CMD_Restart:
				this->Restart( pUserSession, pNode, pResponse );
				break;

			case CMD_Shutdown:
				this->Shutdown( pUserSession, pNode, pResponse );
				break;

			case CMD_StartManagement:
				this->StartManagement( pUserSession, pNode, pResponse );
				break;

			case CMD_StopManagement:
				this->StopManagement( pUserSession, pNode, pResponse );
				break;

			case CMD_GetTime:
				this->GetTime(pUserSession, pNode, pResponse );
				break;
				
			case CMD_SetTime:
				this->SetTime(pUserSession, pNode, pResponse );
				break;

			case CMD_IPR_StopAllVideo:
				this->IPR_StopAllVideo(pUserSession, pNode, pResponse );
				break;
				
			default:
				this->PrintError( pResponse, RDM_ERROR_FUNCTION_NOT_FOUND );
				break;

		}

		result = pResponse->Printf("</%s>",pFunctionName);

		pNode = pNode->Next();
	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int								// return the version number
	CRDM_System::GetVersion
	(
	)
//
//	The function returns the version number
//
//---------------------------------------------------------------------------
{
	return this->version;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::Restart
	(
		CSession	* pUserSession,		// User session
		CSXDB_Node	* NOTUSED(pNode),	// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Restart function
//
//---------------------------------------------------------------------------
{
	int	result;

	if (pUserSession->GetNodePermission( "Restart" ) != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	result = DEP_Restart( );

	if (result != 0)
		return PrintError( pResponse, result );

	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::Shutdown
	(
		CSession	* pUserSession,		// User session
		CSXDB_Node	* NOTUSED(pNode*),	// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Shutdown function
//
//---------------------------------------------------------------------------
{
	int	result;

	if (pUserSession->GetNodePermission( "Restart" ) != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	result = DEP_ShutDown( );

	if (result != 0)
		return PrintError( pResponse, result );

	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::StartManagement
	(
		CSession	*pUserSession,		// User session
		CSXDB_Node	*pNode,			// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Get function
//
//---------------------------------------------------------------------------
{
	START_DATA		start;
	int				result;
	CSXDB_Node	*	pSecurityNode;

	// Check permissions

	if (pUserSession->GetNodePermission("DeviceManagement") != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	// Default values

	memset(&start,0,sizeof(start));

	// Parse the function

	result = SXDB_PT_Get(pNode, startTable, &start, 0);

	if (result == 0)
	{
		if (strlen(start.pEventKey) >= RDM_MAX_KEY_BASE64-1)
			result = -1;
		if (strlen(start.pCCUser) >= RDM_MAX_USER_NAME-1)
			result = -1;
		if (strlen(start.pCCPassword) >= RDM_MAX_PASSWORD-1)
			result = -1;
	}

	if (result != 0)
		return PrintError( pResponse, RDM_ERROR_BAD_PARAMETER );

	// Write the data to the database

	start.enable = 1;

	pRDM->db->BeginAccess(1);

	pSecurityNode = DEP_GetCCDeviceSettings();

	if (pSecurityNode == NULL)
		return PrintError( pResponse, RDM_ERROR_NOT_SUPPORTED );

	pSecurityNode = pSecurityNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, "Security" );

	result = SXDB_PT_Put(pSecurityNode, PT_CHILD_UPDATE, ccTable, &start );

	pRDM->db->EndAccess();

	DEP_CommitCCDeviceSettings();

	// Show results

	if (result < 0)
		return PrintError( pResponse, result );

    if (DEP_InitializeManagement() == FALSE)
	{
		// Stop the management because init failed

		memset(&start,0,sizeof(start));
		start.enable = 0;
		start.pCCPassword = start.pCCUser = start.pClusterID = start.pEventKey = "";
		result = SXDB_PT_Put(pSecurityNode, PT_CHILD_UPDATE, ccTable, &start );

		// Return error

        return PrintError( pResponse, RDM_ERROR_DEVICE_IS_IN_USE );
	}
    		
	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::StopManagement
	(
		CSession	* pUserSession,		// User session
		CSXDB_Node	* NOTUSED(pNode),	// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Shutdown function
//
//---------------------------------------------------------------------------
{
	START_DATA		start;
	int				result;
	CSXDB_Node	*	pSecurityNode;

	// Check permissions

	if (pUserSession->GetNodePermission("DeviceManagement") != TR_PERM_READWRITE)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	// Default values

	memset(&start,0,sizeof(start));

	// Write the data to the database

	start.enable = 0;
	start.pCCPassword = start.pCCUser = start.pClusterID = start.pEventKey = "";

	pSecurityNode = DEP_GetCCDeviceSettings();

	if (pSecurityNode == NULL)
		return PrintError( pResponse, RDM_ERROR_NOT_SUPPORTED );

	pSecurityNode = pSecurityNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, "Security" );

	result = SXDB_PT_Put(pSecurityNode, PT_CHILD_UPDATE, ccTable, &start );

	DEP_CommitCCDeviceSettings();

	// Show results

	if (result < 0)
		return PrintError( pResponse, result );
    
    if (DEP_AreWeManaged())
        DEP_StopManagement();

	return 0;
}



//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::GetTime
	(
		CSession	* NOTUSED(pUserSession),// User session
		CSXDB_Node	* NOTUSED(pNode),	// The node of the function
		CSIO		*pResponse		// SIO Response
	)
//
//	Processes the Restart function
//
//---------------------------------------------------------------------------
{
	char cTime[MAX_RDM_TIME_STRING];
	RDM_TIME	time;

	memset(cTime, 0, MAX_RDM_TIME_STRING);

	RDM_GetSystemTime(&time);
	RDM_RDMTimeToString(&time,cTime);

	pResponse->Puts(cTime);

	return 0;
}


//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::SetTime
	(
		CSession	* NOTUSED(pUserSession),// User session
		CSXDB_Node	* pNode,		// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the Restart function
//
//---------------------------------------------------------------------------
{
	char cTime[MAX_RDM_TIME_STRING];
	RDM_TIME	time;
	WORD		year,month,day,hour,minute,second;

	int result = 0;

	memset(cTime, 0, MAX_RDM_TIME_STRING);

	// Update the system time and send response

	strncpy(cTime, pNode->GetData(), MAX_RDM_TIME_STRING);
	cTime[MAX_RDM_TIME_STRING-1] = 0;

	result = sscanf (cTime, "%04hd-%02hd-%02hd %02hd:%02hd:%02hd", 
							&year,
							&month,
							&day,
							&hour,
							&minute,
							&second);

	if (result == EOF || result < 6 )
	{
		return pResponse->Printf("<Error> %d </Error>", RDM_ERROR_INVALID_DATA);;
	}

	time.year	= (WORD) year;
	time.day	= (BYTE) day;
	time.month	= (BYTE) month;
	time.hour	= (BYTE) hour;
	time.minute	= (BYTE) minute;
	time.second	= (BYTE) second;

	result = RDM_SetSystemTime(&time);

	if (result != 0)
	{
		pResponse->Printf("<Error> %d </Error>", RDM_ERROR_INVALID_DATA);
	}

	return 0;
}

//---------------------------------------------------------------------------
//
	int						// I/O error on output
	CRDM_System::IPR_StopAllVideo
	(
		CSession	* pUserSession,		// User session
		CSXDB_Node	* NOTUSED(pNode),	// The node of the function
		CSIO		* pResponse		// SIO Response
	)
//
//	Processes the IPR_StopAllVideo function
//	This is a private function for internal IP-Reach use
//	Session must be NULL (super user)
//
//---------------------------------------------------------------------------
{
	if (pUserSession != NULL)
		return PrintError( pResponse, RDM_ERROR_PERMISSION_DENIED );

	return DEP_CloseAllVideoStreams();
}

//---------------------------------------------------------------------------
//				Methods that must be overriden by the project
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	void								
	CRDM_System::DEP_SetVersion
	(
		int NOTUSED(_version)
	)
//
//	Called when the system need to change the version
//
//---------------------------------------------------------------------------
{
}

//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//
	int								// RDM Error
	CRDM_System::DEP_ShutDown
	(
	)
//
//	Called when the system should be shut down
//
//---------------------------------------------------------------------------
{
	return RDM_ShutDown();
}

//---------------------------------------------------------------------------
//
	int								// RDM Error
	CRDM_System::DEP_Restart
	(
	)
//
//	Called when the system should be restarted
//
//---------------------------------------------------------------------------
{
	return RDM_Restart();
}

//---------------------------------------------------------------------------
//
	CSXDB_Node	*					// Ptr to DeviceSettings
	CRDM_System::DEP_GetCCDeviceSettings
	(
	)
//
//	Returns the node ptr /DeviceSettings node where the CC management
//	data will be stored.
//
//---------------------------------------------------------------------------
{
	return RDM_GetDeviceSettings();
}

//---------------------------------------------------------------------------
//
	void
	CRDM_System::DEP_CommitCCDeviceSettings
	(
	)
//
//	Called to save commit the CC management settings
//
//---------------------------------------------------------------------------
{
	RDM_CommitDeviceSetting();
}

//---------------------------------------------------------------------------
//
	int									// RDM_ERROR
	CRDM_System::DEP_CloseAllVideoStreams
	(
	)
//
//	Closes all video streams
//
//---------------------------------------------------------------------------
{
	return RDM_CloseAllVideoStreams();
}

//---------------------------------------------------------------------------
//
	BOOL									// RDM_ERROR
	CRDM_System::DEP_InitializeManagement
	(
	)
//
//	Closes all video streams
//
//---------------------------------------------------------------------------
{
	// Derived class implements this function	

	return FALSE;
}

//---------------------------------------------------------------------------
//
    BOOL
    CRDM_System::DEP_AreWeManaged
    (
    )
//
//  if we are managed by a "manager", such as CC, or PS-A
//
//---------------------------------------------------------------------------
{
	// Derived class implements this function	

    return FALSE;
}

//---------------------------------------------------------------------------
//
    void
    CRDM_System::DEP_StopManagement
    (
    )
//
//  By default close CC management. Overwrite it n derived class if it
//  needs to be different
//
//---------------------------------------------------------------------------
{
	// Derived class implements this function	

	return;
}
