/*--------------------------------------------------------------------------

	Session.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	CSession class - Session management

---------------------------------------------------------------------------*/

#include	<stdio.h>
#include	"pp/RDM.h"
#include	"pp/SXDB_Parse_Table.h"
#include	"pp/Session.h"
#include	"pp/OS_Port.h"

#include	"openssl/evp.h"
#include	"pp/RandomData.h"
#include	"pp/RDM_OS_Dependent.h"

//----------------------------------------
//				Equates
//----------------------------------------

//#define	DEBUG_SESSION	// ONLY DEFINE IF DEBUGGING SESSION CODE !

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------
    // NOTE: This hook function is to be provided by the user of the library.
    //       This function is intended to enable the user to preserve the 
    //       session info in history list before it is removed from the RDM
    extern
    bool								//returns true on success, false otherwise.  
    DEP_UpdateSessionHistory
    (
        const char* pSessionID,
        const char* pLoginTime,
        const char* pLogoutTime,
        const char* pLastResourceUsed,
        const char* pUserID,
        const char* pUserName,
        const char* pGroupID
    );

//----------------------------------------
//				Static Data
//----------------------------------------

//------------ User_Session -------------

/*
	<Session id="Sn1">
		<User UserID="xxx">
			<Name> xxx </Name>
			<GroupID> dfdfdf </GroupID>
		</User>
		<LoginTime> xxx </LoginTime>
		<ResourceUsed PortID="xx">
		<LastResourceUsed> Sn1 </LastResourceUsed> 
	</Session>
*/	

typedef struct
{
	const char	*id;						// 
	const char	*Name;						// 
	const char	*UserID;					// 
	const char	*GroupID;					// 
	const char	*LoginTime;					// 
} Session_Data;

#define	PT_STRUCT	Session_Data
PT_BEGIN	( "Session",		Session_Table,		PT_UNKNOWN_OK )
PT_ATT		( "id",				id,				0,	PT_STRING_PTR )
PT_ELEM		( "LoginTime",		LoginTime,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

typedef struct
{
	const char	*Name;						// 
	const char	*UserID;					// 
	const char	*GroupID;					// 
} Session_User_Data;

#define	PT_STRUCT	Session_User_Data
PT_BEGIN	( "User",			Session_User_Table,		PT_UNKNOWN_OK )
PT_ELEM		( "Name",			Name,			0,	PT_STRING_PTR )
PT_ATT		( "UserID",			UserID,			0,	PT_STRING_PTR )
PT_ELEM		( "GroupID",		GroupID,		0,	PT_STRING_PTR )
PT_END
#undef	PT_STRUCT

//----------------------------------------
//				Code
//----------------------------------------

//---------------------------------------------------------------------------
//									CSessionObject
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CSessionObject::CSessionObject
	(
		int		id,						// The class ID of this object
		const char *	pNewName,		// The name of the object
		CSession_CallBack *pNewCallBack,	// User's Call back
		void *	pNewUserData			// User's defined data returned to callback
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	SetClassID( id );
	SetName( pNewName ? pNewName : "No Name" );
	pCallBack = pNewCallBack;
	pUserData = pNewUserData;
	pNext	= NULL;
}

//---------------------------------------------------------------------------
//
	CSessionObject::~CSessionObject
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
}

//---------------------------------------------------------------------------
//
	void
	CSessionObject::Event
	(
		CSession	* pUserSession,		// The session in question...
		int			event,				// Which SESSION_EVENT_ code
		const void		*pData				// Data associated with the event
	)
//
//	This function is called when something happens to a session.
//	The session manager passes in the ptr to the session object, the event
//	code, and a ptr to data (if applicable).
//	See Session.h to for a definition of the event codes and there meanings.
//	The default behavior of this function is to call the callback function
//	passed to the constructor. 
//	This function is virtual and can be over ridden.
//
//	The Callback...
//		pUserData = the user defined void ptr passed to the constructor
//		pUserSession = The user session the event is for
//		event = the event that is happening to the session
//		pData = data associated with the event (if applicable)
//
//---------------------------------------------------------------------------
{
	if (pCallBack != NULL)
		(pCallBack)( pUserData, pUserSession, event, pData );
}

//---------------------------------------------------------------------------
//									CSessionObjectList
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CSessionObjectList::CSessionObjectList
	(
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	pObjectList = NULL;

	// Initialize the Central CS for new sessions, if not already done

	cs = OS_CreateCriticalSection( OS_CRITICAL_SECTION_NORMAL );
}

//---------------------------------------------------------------------------
//
	CSessionObjectList::~CSessionObjectList
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
	// Delete all of the objects in the list

	DeleteObjects();
	
	if(cs) OS_DeleteCriticalSection(cs);
}

//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::AddSessionObject
	(
		CSessionObject		*pObject		// The object to add
	)
//
//	Adds a Session Object to this session.
//
//---------------------------------------------------------------------------
{
	OS_EnterCriticalSection( cs );

	pObject->pNext = pObjectList;
	pObjectList = pObject;

	OS_LeaveCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::RemoveSessionObject
	(
		CSessionObject		*pObject		// The object to remove
	)
//
//	Removes a Session Object from this session.
//
//---------------------------------------------------------------------------
{
	CSessionObject *pTemp;

	OS_EnterCriticalSection( cs );

	if (pObjectList == pObject)
		pObjectList = pObject->pNext;
	else
	{
		for (pTemp = pObjectList; pTemp != NULL && pTemp->pNext != pObject; pTemp = pTemp->pNext)
			;

		if (pTemp != NULL)
			pTemp->pNext = pObject->pNext;
	}

	pObject->pNext = NULL;

	OS_LeaveCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	CSessionObject *					// Ptr to next object or NULL
	CSessionObjectList::EnumObject
	(
		CSessionObject		*pObject	// NULL to find first, or that last
										// last found to find next.
	)
//
//	Enumerates the objects attached to this session.
//
//---------------------------------------------------------------------------
{
	if (pObject == NULL)
		return pObjectList;
	else
		return pObject->pNext;
}

//---------------------------------------------------------------------------
//
	CSessionObject *					// Ptr to the next object or NULL
	CSessionObjectList::EnumObjectByName
	(
		const char		*pName,		// Ptr to the object name
		CSessionObject		*pObject	// NULL to find first, or that last
										// found to find next.
	)
//
//	Enumerates the objects attached to this session that match the name.
//
//---------------------------------------------------------------------------
{
	if (pObject == NULL)
		pObject = pObjectList;

	for ( ; pObject != NULL; pObject = pObject->pNext)
	{
		if (strcmp(pObject->pName, pName) == 0)
			return pObject;
	}

	return NULL;
}

//---------------------------------------------------------------------------
//
	CSessionObject *					// Ptr to the next object or NULL
	CSessionObjectList::EnumObjectByClass
	(
		int					classID,	// The class ID to search for
		CSessionObject		*pObject	// NULL to find first, or that last
										// found to find next.
	)
//
//	Finds the next object that matches the name
//
//---------------------------------------------------------------------------
{
	if (pObject == NULL)
		pObject = pObjectList;
	else
		pObject = pObject->pNext;

	for ( ; pObject != NULL; pObject = pObject->pNext)
	{
		if (pObject->classID == classID)
			return pObject;
	}

	return NULL;
}


//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::LockList
	(
	)
//
//	Locks the session list
//
//---------------------------------------------------------------------------
{
	OS_EnterCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::UnlockList
	(
	)
//
//	Unlocks the session list
//
//---------------------------------------------------------------------------
{
	OS_LeaveCriticalSection( cs );
}

//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::Event
	(
		CSession	* pUserSession,		// The session in question...
		int			event,				// Which SESSION_EVENT_ code
		const void		*pData				// Data associated with the event
	)
//
//	Sends an event to all the Session Objects in the list
//
//---------------------------------------------------------------------------
{
	CSessionObject	*pObject;

	LockList();

	for ( pObject = pObjectList; pObject != NULL; pObject = pObject->pNext)
	{
		pObject->Event( pUserSession, event, pData );
	}

	UnlockList();
	
}

//---------------------------------------------------------------------------
//
	void
	CSessionObjectList::DeleteObjects
	(
	)
//
//	Closes a session.
//
//---------------------------------------------------------------------------
{
	CSessionObject	*pObject,*pNextObject;;

	LockList();

	for ( pObject = pObjectList; pObject != NULL; pObject = pNextObject)
	{
		pNextObject = pObject->pNext;
		delete pObject;
	}

	pObjectList = NULL;

	UnlockList();
}

//---------------------------------------------------------------------------
//									CSession
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CSession::CSession
	(
		class CRDM	*	pNewRDM,		// Ptr to the RDM object
		CSessionManager	*	pNewMgr			// Ptr to the manager
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	char	key[16];
	int		result;
	EVP_ENCODE_CTX	ctx;
    char    id[RDM_MAX_ID];

	used = 1;
	closed = 0;
    RDM_CreateUniqueID ( "S", pNewMgr->nextSessionID, id );
    strcpy (sessionID, id);

	pNewMgr->nextSessionID ++;
	pMgr = pNewMgr;
	pRDM	= pNewRDM;
	pUserObject = NULL;
	
	// Create the session key

	#ifndef DEBUG_SESSION
		memset(key, 0, 16);
		GetRandomData( key, 16 );
	#else
		// Debug...so that key is always the same
		for (result = 0;result<16;result++)
			key[result] = result;
	#endif

	EVP_EncodeInit(&ctx);
	EVP_EncodeUpdate(&ctx, (u_char *)sessionKey, &result, (u_char *)key, 16);
	EVP_EncodeFinal(&ctx, (u_char *)sessionKey, &result);
    // we don't need a newline at the end, remove it
    {
    int last=strlen(sessionKey)-1;
    if (sessionKey[last] == '\n') sessionKey[last] = 0;
    }

	// Add ourselfs to the session list

	pMgr->AddSession(this);

	// Update RDM with our session info

	WriteSessionData();


	ipAddress = 0;
}

//---------------------------------------------------------------------------
//
    BOOL
	CSession::IsResourceUsed
	(
        const char *pResourceId
	)
//
//	Return true is the input resource id is in use
//
//---------------------------------------------------------------------------
{
    int             result;
	char			xpath[100];
	CSXPath			p;
    CSXDB_Node      *pNode;
	//CSXDB_Element	*pNode;

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]/ResourceUsed",this->sessionID);

    result = FALSE;

	int count = p.Parse( xpath, pRDM->db );

    for (int i=0; i<count; i++)
    {
	    pNode = (CSXDB_Element *) p.Enum(i);

        const char * data = pNode->GetData();

        if (data && strcmp (pResourceId, data)==0)
            return TRUE;
    } 
    
    return result;    
}


//---------------------------------------------------------------------------
//
    const char *
    CSession::GetResourceID
    (
		BOOL    referral 
    )
//
//	Return the resource held by this session
//
//---------------------------------------------------------------------------
{
	char xpath[128];

	if (referral == FALSE)
	{
		sprintf(xpath, "//*[@id=\"%s\"]", sessionID);

		return pRDM->db->GetData(xpath,"ResourceUsed");
	}

	else 
	{
    	CSXPath			p;
    	CSXDB_Node      *pNode;
	
    	sprintf(xpath, "//*[@id=\"%s\"]/ResourceUsed[@Pending=\"%d\"]", sessionID, 1);
    	int count = p.Parse(xpath, pRDM->db);

    	if (count == 0)
        	return NULL;
    
    	pNode = p.Enum(0);

    	return  pNode->GetData();
	}
}

//---------------------------------------------------------------------------
//
    const char *
    CSession::GetElementData
    (
     const char *xpath,
     const char *pElement
    )
//
//
//---------------------------------------------------------------------------
{
    const char            *pElementData;
    CSXPath         p;
    CSXDB_Element   *pNode;
    CSXDB_Element   *pElementID;
    
    pRDM->db->BeginAccess( 1 );

    p.Parse( xpath, pRDM->db );

    pNode = (CSXDB_Element *) p.Enum(0);
    if (pNode == NULL) 
    {
        pRDM->db->EndAccess( );
        return NULL;
    }
    
    pElementID = (CSXDB_Element *) pNode->FindChildOfTypeByName(SXDB_TYPE_ELEMENT, pElement, NULL);
    if(pElementID == NULL)
    {
        pRDM->db->EndAccess( );
        return NULL;
    }
    pElementData = pElementID->GetData();
    pRDM->db->EndAccess();
    return pElementData;
}

//---------------------------------------------------------------------------
//
	const char *
    CSession::GetAttributeData
    (
     const char *xpath,
     const char *pAttribute
    )
//
//
//---------------------------------------------------------------------------
{
    const char            *pAttributeData;
    CSXPath         p;
    CSXDB_Element   *pNode;
    CSXDB_Attribute *pAttributeID;
    
    pRDM->db->BeginAccess( 1 );

    p.Parse( xpath, pRDM->db );

    pNode = (CSXDB_Element *) p.Enum(0);
    if (pNode == NULL) 
    {
        pRDM->db->EndAccess( );
        return NULL;
    }
    
    pAttributeID = (CSXDB_Attribute *) pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, pAttribute, NULL);
    if(pAttributeID == NULL)
    {
        pRDM->db->EndAccess( );
        return NULL;
    }
    pAttributeData = pAttributeID->GetData();
    pRDM->db->EndAccess();
    return pAttributeData;

}

//---------------------------------------------------------------------------
//
	CSession::~CSession
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
    char            xpath[100];
    RDM_TIME        time;
    char            cTime[MAX_RDM_TIME_STRING];

	// Make sure we are closed

	if (!closed)
		Close();

    // Preserve session history

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);
    RDM_GetSystemTime( &time );
    RDM_RDMTimeToString( &time, cTime );

	if (this->pUserObject != NULL)
	{
		DEP_UpdateSessionHistory(this->sessionID,
					 pRDM->db->GetData( xpath, "LoginTime"),
					 cTime,
					 pRDM->db->GetData( xpath, "LastResourceUsed"),
					 this->pUserObject->GetUserID(),
					 this->pUserObject->GetUserName(),
					 this->pUserObject->GetGroupID());
	}

	// Delete the session info from RDM

	pRDM->db->BeginAccess(1);

	pRDM->db->Delete( xpath );

	pRDM->db->EndAccess();

	// Hack Notification ?

	// Delete all of the session objects

	DeleteObjects();

	// Delete the user object

	if (pUserObject != NULL)
		delete pUserObject;
}

//---------------------------------------------------------------------------
//
	void
	CSession::WriteSessionData
	(
	)
//
//	Writes the session info to the RDM database
//
//---------------------------------------------------------------------------
{
	Session_Data	sessionData;
	RDM_TIME		time;
	char			cTime[MAX_RDM_TIME_STRING];
	int				result;

//	DBAssert( used > 0 );

	pRDM->db->BeginAccess( 1 );

	RDM_GetSystemTime( &time );
	RDM_RDMTimeToString( &time, cTime );

	memset( &sessionData,0,sizeof(sessionData) );

	sessionData.id				= this->sessionID;
	sessionData.LoginTime		= cTime;

	result = SXDB_PT_Put(	pRDM->db,
							"/System/Sessions",
							PT_CHILD_APPEND,
							Session_Table,
							&sessionData,
							0
						);

	pRDM->db->EndAccess( );
}

//---------------------------------------------------------------------------
//
	DWORD
	CSession::GetPermission
	(
		const char *SecurityId, 
		const char *uniqueId
	)
//
//	Returns the TR_PERM_xxx value for the permission and node
//
//---------------------------------------------------------------------------
{
	if (this == NULL)
		return TR_PERM_READWRITE;	// Super user

	if (this->pUserObject == NULL)
		return TR_PERM_NONE;		// No user object

	return this->pUserObject->GetPermission(SecurityId, uniqueId);
}

//---------------------------------------------------------------------------
//
	DWORD
	CSession::GetNodePermission
	(
		const char *SecurityId
	)
//
//	Returns the TR_PERM_xxx value for the permission and node
//
//---------------------------------------------------------------------------
{
	if (this == NULL)
		return TR_PERM_READWRITE;	// Super user

	if (this->pUserObject == NULL)
		return TR_PERM_NONE;		// No user object

	return this->pUserObject->GetNodePermission(SecurityId);
}

//---------------------------------------------------------------------------
//
	CUserObject	*
	CSession::GetUserObject
	(
	)
//
//	Returns the associated user object
//
//---------------------------------------------------------------------------
{
//	DBAssert( used > 0 );

	return this->pUserObject;
}

//---------------------------------------------------------------------------
//
	void
	CSession::SetUserObject
	(
		CUserObject	*pNewUserObject	// Ptr to the user object for this session
	)
//
//	Associates a user object with this session.
//
//	This method will also update the RDM database Session node for with the
//	new session data
//
//---------------------------------------------------------------------------
{
//	DBAssert( used > 0 );

	pUserObject = pNewUserObject;
	UpdateUserData();
}

//---------------------------------------------------------------------------
//
	void
	CSession::UpdateUserData
	(
	)
//
//	This method will updates the RDM database Session node for with the
//	new session user data
//
//---------------------------------------------------------------------------
{
	Session_User_Data	sessionData;
	char			xpath[100];
	char			name[128];
	int				result;

//	DBAssert( used > 0 );

	if (pUserObject == NULL)
		return;

	// Update RDM

	pRDM->db->BeginAccess( 1 );

	memset( &sessionData,0,sizeof(sessionData) );

	sessionData.UserID			= this->pUserObject->GetUserID();
	strcpy(name,this->pUserObject->GetUserName());
	for (int x=0;name[x];x++)
		name[x]= toupper(name[x]);
	sessionData.Name			= name;
	sessionData.GroupID			= this->pUserObject->GetGroupID();
	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);

	result = SXDB_PT_Put(	pRDM->db,
							xpath,
							PT_CHILD_UPDATE,
							Session_User_Table,
							&sessionData,
							0
						);

	pRDM->db->EndAccess( );
}

//---------------------------------------------------------------------------
//
    void
    CSession::SetSessionSignature
    (
        const char *new_sessionID,
        const char *new_sessionKey
    )
//
//
//  This method will also update the RDM database Session node for with the
//  new session id and session key
//
//---------------------------------------------------------------------------
{
    char            xpath[100];
    CSXPath         p;
    CSXDB_Element   *pNode;
    CSXDB_Attribute *pNodeID;
    
//  DBAssert( used > 0 );

    if(!new_sessionID) return;
    if(!new_sessionKey) return;

    // update RDM DB
    pRDM->db->BeginAccess( 1 );

    sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);
    p.Parse( xpath, pRDM->db );

    pNode = (CSXDB_Element *) p.Enum(0);
    if (pNode == NULL) 
    {
        pRDM->db->EndAccess( );
        return;
    }

    pNodeID = (CSXDB_Attribute *) pNode->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "id", NULL);
    if(!pNodeID)
    {
        pRDM->db->EndAccess( );
        return;
    }
    
    pNodeID->SetData(new_sessionID);

    pRDM->db->EndAccess( );

    strcpy(sessionKey, new_sessionKey);   
    strcpy(sessionID, new_sessionID);
}

//---------------------------------------New Additions-----------------------------------
//---------------------------------------------------------------------------
//
	void
	CSession::AddUsedResourceWithAttribute
	(
		const char	*pResourceID,			// Ptr to the resource ID
        const char    *pAttr1,                  // Ptr to the Attribute1
        const char    *pData1,                  // Ptr to the Attribute1 Data
        const char    *pAttr2,
        const char    *pData2
	)
//
//	Adds a resource ID as a <ResourceUsed> element with Attribute.
//
//---------------------------------------------------------------------------
{
	char			xpath[100];
	CSXPath			p;
	CSXDB_Element	*pNode;
    CSXDB_Element   *pChild;

//	DBAssert( used > 0 );

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);

    pRDM->db->BeginAccess( 1 );

	p.Parse( xpath, pRDM->db );

	pNode = (CSXDB_Element *) p.Enum(0);

	pChild = NULL;

	if (pNode != NULL)
    {
		pChild = pNode->AddElementWithData( "ResourceUsed",pResourceID);
    }
    if (pChild != NULL)
	{
		if (pAttr1 != NULL)
			pChild->AddAttribute(pAttr1, pData1);  //Add the attribute1
		if (pAttr2 != NULL)
			pChild->AddAttribute(pAttr2, pData2);  //Add the attribute2
	}
    pRDM->db->EndAccess();
 
}
//---------------------------------------------------------------------------
//
    void
    CSession::DeleteResourceAttribute
    (
       const char    *pResource,
       const char    *pAttr
    )
//
//  Deletes an Attribute of the Resource
//
//---------------------------------------------------------------------------
{
    char          xpath[100];
    CSXPath       p;
    CSXDB_Element	*pNode;
    CSXDB_Element   *pChild;
    CSXDB_Attribute *pChildAttr;

    sprintf(xpath, "/System/Sessions/Session[@id=\"%s\"]",this->sessionID);
    
    pRDM->db->BeginAccess( 1 );
    
    p.Parse( xpath, pRDM->db );

    pNode = (CSXDB_Element *) p.Enum(0);

	if (pNode != NULL)
    {
	  pChild = (CSXDB_Element *)pNode->FindChildOfTypeByName(SXDB_TYPE_ELEMENT ,pResource, NULL);
      if (pAttr != NULL) 
      {
        pChildAttr = (CSXDB_Attribute *)pChild->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, pAttr, NULL);
        if(pChildAttr != NULL)
          pChildAttr->Remove();
      }
        
    }
    pRDM->db->EndAccess();

}

//---------------------------------------------------------------------------
//
	void
	CSession::ModifyUsedResourceAndAttribute
	(
		const char	*pResourceID,			// Ptr to the resource ID
        const char    *pAttr1,                  // Ptr to the Attribute1
        const char    *pAttr2,
        const char    *pDeviceID,
        const char    *pData1,                  // Ptr to the Attribute Data
        const char    *pData2
	)
//
//	Adds a resource ID as a <ResourceUsed> element with Attribute.
//
//---------------------------------------------------------------------------
{
	char			xpath[100];
	CSXPath			p;
	CSXDB_Element	*pNode;
    CSXDB_Element   *pChild;
    CSXDB_Attribute *pChildAttr;

//	DBAssert( used > 0 );

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);

    pRDM->db->BeginAccess( 1 );

	p.Parse( xpath, pRDM->db );

	pNode = (CSXDB_Element *) p.Enum(0);

	pChild = NULL;

	if (pNode != NULL)
    {
	  pChild = (CSXDB_Element *)pNode->FindChildOfTypeByName(SXDB_TYPE_ELEMENT ,"ResourceUsed", NULL);
    }
    if (pChild != NULL)
    {
      pChild->SetData(pResourceID);
      
      if (pAttr1 != NULL) 
      {

        pChildAttr = (CSXDB_Attribute *)pChild->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, pAttr1, NULL);
      
        if(pChildAttr != NULL) 

          pChildAttr->SetData(pData1);  //Add the attribute
      
        else
        
          pChild->AddAttribute(pAttr1, pData1);

      }

      if (pAttr2 != NULL) 
      {

        pChildAttr = (CSXDB_Attribute *)pChild->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, pAttr2, NULL);
      
        if(pChildAttr != NULL) 

          pChildAttr->SetData(pData2);  //Add the attribute
      
        else
        
          pChild->AddAttribute(pAttr2, pData2);

      }

    }
   
    if (pNode != NULL)
    {
      pChild = (CSXDB_Element *)pNode->FindChildOfTypeByName(SXDB_TYPE_ELEMENT, "Device", NULL);
    }
    if (pChild != NULL)
    {
      pChildAttr = (CSXDB_Attribute *)pChild->FindChildOfTypeByName(SXDB_TYPE_ATTRIBUTE, "DeviceID", NULL);
      
      if (pChildAttr != NULL)
        pChildAttr->SetData(pDeviceID);
    }
    pRDM->db->EndAccess();
 
}
 

//------------------------------------------New Additions----------------------

//---------------------------------------------------------------------------
//
	void
	CSession::AddUsedResource
	(
		const char	*pResourceID,			// Ptr to the resource ID
        BOOL    referral
	)
//
//	Adds a resource ID as a <ResourceUsed> element.
//
//---------------------------------------------------------------------------
{
	char			xpath[100];
	char			xpath1[100];
	CSXPath			p;
	CSXDB_Element	*pNode;

//	DBAssert( used > 0 );

	pRDM->db->BeginAccess( 1 );

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);

	p.Parse( xpath, pRDM->db );

	pNode = (CSXDB_Element *) p.Enum(0);

	if (pNode != NULL)
 	{
		pNode->AddElementWithData( "ResourceUsed", pResourceID );
 		// Update LastResourceUsed
 		sprintf(xpath1,"/System/Sessions/Session[@id=\"%s\"]/LastResourceUsed",this->sessionID);
 		pRDM->db->Delete( xpath1 );
 		pNode->AddElementWithData( "LastResourceUsed", pResourceID);
 	}
    if (referral == TRUE)
    {
        p.Parse("ResourceUsed", pRDM->db, pNode);

        pNode = (CSXDB_Element *) p.Enum(0);

        pNode->AddAttribute("Pending", "1"); // 1 means ResourceUsed is pending
    }
	
	pRDM->db->EndAccess( );
}

//---------------------------------------------------------------------------
//
    void
    CSession::AddDeviceAndInputPort
    (
       const char *DeviceID,
       const char *InputPort
    )
//
//
//---------------------------------------------------------------------------
{
    char        xpath[100];
    CSXPath     p;
    CSXDB_Element   *pNode;
    CSXDB_Element    *pChild = NULL;

    sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]",this->sessionID);
    pRDM->db->BeginAccess( 1 );
    p.Parse(xpath, pRDM->db);
    pNode = (CSXDB_Element *) p.Enum(0);
    if (pNode != NULL)
       pChild = pNode->AddElement("Device");
    pChild->AddAttribute("DeviceID", DeviceID);
    pChild->AddElementWithData("InputPort",InputPort);
    pRDM->db->EndAccess();

}
    
//---------------------------------------------------------------------------
//
	void
	CSession::SubtractUsedResource
	(
		const char	*pResourceID			// Ptr to the resource ID
	)
//
//	Removes a resource ID as a <ResourceUsed> element.
//
//---------------------------------------------------------------------------
{
	char			xpath[150];

//	DBAssert( used > 0 );

	pRDM->db->BeginAccess( 1 );

	sprintf(xpath,"/System/Sessions/Session[@id=\"%s\"]/ResourceUsed[.=\"%s\"]",this->sessionID,pResourceID);

	pRDM->db->Delete( xpath );

	pRDM->db->EndAccess( );
}

//---------------------------------------------------------------------------
//
	void
	CSession::Aquire
	(
	)
//
//	Increments the session's "used" count. The session cannot be delete as
//	long as it's "used" count in >0. Any code that calls Aquire must call
//	Release when it is done with the session
//
//---------------------------------------------------------------------------
{
	LockList();

//	DBAssert( used > 0 );

	used++;

	UnlockList();
}

//---------------------------------------------------------------------------
//
	void
	CSession::Release
	(
	)
//
//	Decrements the session's "used" count.
//	If the used count goes to zero, the object is deleted
//
//---------------------------------------------------------------------------
{
	int	isUsed;

	LockList();

//	DBAssert( used > 0 );

	used--;

	isUsed = used;

	UnlockList();

	if (isUsed == 0)
		pMgr->QueueGarbageCollect();
}

//---------------------------------------------------------------------------
//
	void
	CSession::Close
	(
	)
//
//	Closes a session
//
//---------------------------------------------------------------------------
{
	// If we are already closed, do nothing

	if (closed)
		return;

	// Mark the session as closed

	closed = 1;

	// Send the close event to all listeners

	Event(this, SESSION_EVENT_CLOSE, NULL);
	pMgr->Event(this, SESSION_EVENT_CLOSE, NULL);

}

//---------------------------------------------------------------------------
//								CSessionManager
//---------------------------------------------------------------------------

//---------------------------------------------------------------------------
//
	CSessionManager::CSessionManager
	(
		class CRDM	*	pNewRDM			// Ptr to the RDM object
	)
//
//	Initialize data items
//
//---------------------------------------------------------------------------
{
	// Initialize Data

	sessionCount	= 0;
	sessionsDisabled= 0;
	nextSessionID	= 1;
	pSessionList	= NULL;
	pRDM		= pNewRDM;

	// Create our Garbage Collection thread

	hGarbageCollectEvent = OS_CreateEvent( OS_EVENT_NORMAL );
	hGarbageCollectThread = OS_CreateThread( (void *) GarbageCollectThread, OS_THREAD_NORMAL, this );
}

//---------------------------------------------------------------------------
//
	CSessionManager::~CSessionManager
	(
	)
//
//	Cleanup
//
//---------------------------------------------------------------------------
{
	Shutdown();
	OS_SetEvent( hGarbageCollectEvent );
	DeleteObjects();
	OS_DeleteEvent( hGarbageCollectEvent );
	OS_DeleteThread( hGarbageCollectThread );
}

//---------------------------------------------------------------------------
//
	CSession *							// Ptr to the new session or NULL
	CSessionManager::NewSession
	(
	)
//
//	Creates a new session and returns the ptr to it.
//	The "used" count of the session is set to 1.
//	The creator of this session must call (CSession)->Release() when it is
//	done with the session.
//
//---------------------------------------------------------------------------
{
	CSession * pSession = NULL;

	if (sessionsDisabled)
		return NULL;

	pSession = new CSession( pRDM, this );


	// Send open event

	if (pSession != NULL)
		Event(pSession, SESSION_EVENT_OPEN, NULL);

	return pSession;
}


//---------------------------------------------------------------------------
//
    void
    CSessionManager::AddSession
    (
        CSession *pSession
    )
//
//  Add the session to session list 
//
//---------------------------------------------------------------------------
{
	if(!pSession)return;
	
	LockList();

	pSession->pNext = pSessionList;
	pSessionList = pSession;
	#ifndef DEBUG_SESSION
		sessionCount ++;
	#endif

	UnlockList();
}


//---------------------------------------------------------------------------
//
    void
    CSessionManager::RemoveSession
    (
        CSession *pSession
    )
//
//  Remove the session to session list 
//
//---------------------------------------------------------------------------
{
	CSession *pTemp;
	
	if(!pSession)return;
	
	LockList();

	if (pSessionList == pSession)
		pSessionList = pSession->pNext;
	else
	{
		for (pTemp = pSessionList; pTemp != NULL && pTemp->pNext != pSession; 
		    pTemp = pTemp->pNext);

		if (pTemp != NULL)
			pTemp->pNext = pSession->pNext;
	}

	pSession->pNext = NULL;

	sessionCount --;

	UnlockList();
	
}



//---------------------------------------------------------------------------
//
    BOOL 
    CSessionManager::IsResourceUsed
    (
        const char *pResourceId
    )
//
//  Check the session list and find out if this resource is used by 
//  any session in the list.
//
//---------------------------------------------------------------------------
{
	CSession *pSession;

	this->LockList();

	pSession = pSessionList;

	while ( pSession != NULL)
	{
		if (pSession->used != 0)
        {
            if (pSession->IsResourceUsed(pResourceId) == TRUE)
            {
            	this->UnlockList();
                return TRUE;
            }
        }

		pSession = pSession->pNext;
	}

	this->UnlockList();

    return FALSE;
}

//---------------------------------------------------------------------------
//
	void
	CSessionManager::QueueGarbageCollect
	(
	)
//
//	Called when a session's used count goes to zero.
//
//---------------------------------------------------------------------------
{
	OS_SetEvent( hGarbageCollectEvent );
}

//---------------------------------------------------------------------------
//
	void
	CSessionManager::GarbageCollect
	(
	)
//
//	Cleans out unused sessions
//
//---------------------------------------------------------------------------
{
	CSession *pSession,*pNext;
	CSession *pSessionDelete = NULL, *pNextDelete;

	this->LockList();

	pSession = pSessionList;
	pNextDelete = pSessionDelete;

	while ( pSession != NULL)
	{
		pNext = pSession->pNext;

		if (pSession->used == 0)
		{
			RemoveSession( pSession );
			
			// add to delete list
			if(pNextDelete == NULL) 
				pSessionDelete = pNextDelete = pSession;
			else
			{ 
				pNextDelete->pNext = pSession;
				pNextDelete = pSession;
			}
		}

		pSession = pNext;
	}

	this->UnlockList();
	
	// delete session from delete list
	pSession = pSessionDelete;
	while (pSession != NULL)
	{
		pNextDelete = pSession->pNext;
		delete pSession;
		
		pSession = pNextDelete;
	}
}

//---------------------------------------------------------------------------
//
	int
	CSessionManager::GetSessionCount
	(
	)
//
//	Returns the number of sessions currently allocated
//
//---------------------------------------------------------------------------
{
	return sessionCount;
}

//---------------------------------------------------------------------------
//
	CSession *							// Returns ptr to session or NULL
	CSessionManager::GetSession
	(
		const char	*pSessionID				// session ID of the session to get
	)
//
//	Returns a ptr to the session that matches <sessionID>.
//	This function will lock the session object. the Caller MUST call the
//	session's Unlock() method when done with the session.
//
//---------------------------------------------------------------------------
{
	CSession *	pSession = NULL;

	if (pSessionID == NULL)
		return NULL;

	while (( pSession = EnumSession( pSession ) ))
	{
		if (strcmp(pSessionID,pSession->sessionID) == 0)
			return pSession;
		else
			pSession->Release();
	}

	return NULL;
}

//---------------------------------------------------------------------------
//
	CSession *							// Returns ptr to session or NULL
	CSessionManager::EnumSession
	(
		CSession *pLastSession			// NULL to get first, last found
										// to get next
	)
//
//	Returns the number of sessions currently allocated
//	This function will lock the session object. the Caller MUST call the
//	session's Unlock() method when done with the session.
//
//---------------------------------------------------------------------------
{
	CSession *	pSession = NULL;

	this->LockList();

	if (pLastSession == NULL)
		pSession = pSessionList;
	else
		pSession = pLastSession->pNext;


	if (pSession != NULL)
		pSession->Aquire();

	this->UnlockList();

	return pSession;
}

//---------------------------------------------------------------------------
//
	void
	CSessionManager::CloseAllOtherSessions
	(
		CSession *pSess
	)
//
//	Closes all other sessions except this one
//
//---------------------------------------------------------------------------
{
	CSession *	pSession;

	sessionsDisabled = 1;

	this->LockList();
	for (pSession = pSessionList; pSession != NULL; pSession = pSession->pNext)
	{
		if (pSession != pSess)
		{
			pSession->Close();
		}
	}
	this->UnlockList();
}

//---------------------------------------------------------------------------
//
	void
	CSessionManager::EnableSessions
	(
	)
//
//	Enable incoming sessions (usually after a previous "disable")
//
//---------------------------------------------------------------------------
{
	sessionsDisabled = 0;
}

//---------------------------------------------------------------------------
//
	void
	CSessionManager::Shutdown
	(
	)
//
//	Closes all sessions and disables new sessions from being created.
//
//---------------------------------------------------------------------------
{
	CSession *	pSession;

	sessionsDisabled = 1;

	while ((pSession = EnumSession( NULL )))
	{
		pSession->Close();
		pSession->Release();
	}
}

//---------------------------------------------------------------------------
//
	DWORD
	GarbageCollectThread
	(
		LPVOID			lpParam			// The thread parameter
	)
//
//	Cleans up after defunct sessions.
//
//---------------------------------------------------------------------------
{
	CSessionManager	*pMgr = (CSessionManager *) lpParam;

	while (!pMgr->sessionsDisabled)
	{
		OS_WaitForEvent( pMgr->hGarbageCollectEvent, INFINITE );

		pMgr->GarbageCollect();
	}
	
	return 0;
}
