/*-------------------------------------------------------------------------------

	SysProgressMgr.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	System progress manager class implementation (CSysProgressMgr)

	This class is used by other components to report begin of progress,
	and end of progress. If first begin report comes, it will notify
	System_Progress_Begin event (event-43). If last end report comes, it will
	notify System_Progress_End (event-44). So that event-43 and event-44 can 
	be sent in serial order.  

--------------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>				// for atoi()
#include "pp/RDM.h"
#include "pp/RDM_Notify.h"
#include "pp/SXDB_Parse_Table.h"
#include "pp/RDM_EventCode.h"
#include "pp/SXDB_Parse.h"

#include "pp/SysProgressMgr.h"




//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------
 
//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

extern CRDM rdm;
static CSysProgressMgr *instance_CSysProgressMgr = NULL;

CSysProgressMgr *GetInstanceCSysProgressMgr()
{
    if(instance_CSysProgressMgr == NULL)
        instance_CSysProgressMgr = new CSysProgressMgr( &rdm );

    return instance_CSysProgressMgr;
}


//--------------------------------------------------------------------------------
//              CSysProgressObj
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
    CSysProgressObj::CSysProgressObj
    (
        int type,   // 1 - database update; 2 - firmware update
        const char *id,   // must be unique. 
        const char *message
    )
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
    m_type = type;
    strncpy(m_id, id, MAX_LEN_PROGRESS_ID);
    if(message)
        strncpy(m_message, message, MAX_LEN_PROGRESS_MSG);
    else
        memset(m_message, 0, MAX_LEN_PROGRESS_MSG);
    
    m_next = NULL;
}
  

//--------------------------------------------------------------------------------
//  
    CSysProgressObj::~CSysProgressObj
    (
    )
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
    
}


//--------------------------------------------------------------------------------
//  
    int                     // 1 -- match; 0 -- no match 
    CSysProgressObj::IfMatch
    (
        const char *id            // progress id
    )
//
//	Check if the id is as same as is id of object
//
//------------------------------------------------------------------------------//
{
    if(id == NULL) return 0;
    return (strncmp(m_id, id, MAX_LEN_PROGRESS_ID) == 0);
}



//--------------------------------------------------------------------------------
//              CSysProgressMgr
//--------------------------------------------------------------------------------


//--------------------------------------------------------------------------------
//  
	CSysProgressMgr::CSysProgressMgr
	(
	    CRDM    *pRDM
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
    m_pRDM = pRDM;
    m_list = NULL;
    m_cs = OS_CreateCriticalSection(OS_CRITICAL_SECTION_NORMAL);
}
	

//--------------------------------------------------------------------------------
//  
	CSysProgressMgr::~CSysProgressMgr
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
    if(m_list)
    {
        // clear all CSysProgressObj
        ClearList();
    }
    
    OS_DeleteCriticalSection(m_cs);
}	


//--------------------------------------------------------------------------------
//  
	void 
	CSysProgressMgr::ClearList
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
    CSysProgressObj *obj;
    
    Lock();
    
    while(m_list)
    {
        obj = m_list;
        m_list = obj->m_next;
        delete obj;
    }
    
    Unlock();
}	
	
	
//--------------------------------------------------------------------------------
//  
	void 
	CSysProgressMgr::Lock
	(
	)
//
//	Lock list
//
//------------------------------------------------------------------------------//
{
    OS_EnterCriticalSection(m_cs);
}	


//--------------------------------------------------------------------------------
//  
	void CSysProgressMgr::Unlock
	(
	)
//
//	Unlock list
//
//------------------------------------------------------------------------------//
{
    OS_LeaveCriticalSection(m_cs);
}	
	

//--------------------------------------------------------------------------------
//  	
	int                     // 1 -- yes, 0 -- no
	CSysProgressMgr::IfSysInProgress
	(
	)
//
//	Check if system is busy in change
//
//------------------------------------------------------------------------------//
{
    return (m_list != NULL);
}	


//--------------------------------------------------------------------------------
//  
    CSysProgressObj *       // NULL if not found
    CSysProgressMgr::GetSysProgressObj
    (
        const char *id            // progress id
    )
//
//	Get CSysProgressObj with the id.
//  Note: caller needs to lock list
//
//------------------------------------------------------------------------------//
{
    CSysProgressObj *obj;
    
    obj = m_list;
    while(obj)
    {
        if(strncmp(obj->m_id, id, MAX_LEN_PROGRESS_ID) == 0)
            break;
        obj = obj->m_next;
    }
    
    return obj;
}	


//--------------------------------------------------------------------------------
//  
    int                     // 1 -- success, 0 -- fail
    CSysProgressMgr::InsertSysProgressObj
    (
       CSysProgressObj *obj 
    )
//
//	Note: caller must make sure there is no duplicated obj in list
//        claler also need to lock list
//
//------------------------------------------------------------------------------//
{
    if(obj == NULL) return 0;
    
    if(m_list == NULL) 
        m_list = obj;
    else
    {
        obj->m_next = m_list;
        m_list = obj;
    }
    
    return 1;
}	


//--------------------------------------------------------------------------------
//      
    int                     // 1 -- success, 0 -- fail
    CSysProgressMgr::RemoveSysProgressObj
    (
        const char *id
    )
//
//  Note: caller needs to lock list
//
//------------------------------------------------------------------------------//
{
    int found = 0;
    CSysProgressObj *pre_obj = NULL, *obj = NULL;
    
    pre_obj = NULL;
    obj = m_list;
    while(obj)
    {
        if(strncmp(obj->m_id, id, MAX_LEN_PROGRESS_ID) == 0)
        {
            // find this obj
            
            if(pre_obj)
            {
                pre_obj->m_next = obj->m_next;
            }
            else
            {
                // this obj is on the head of list
                m_list = obj->m_next;
            }
            
            delete obj;
            found = 1;
            break;
        }
        
        pre_obj = obj;
        obj = obj->m_next;
    }
    
    return found;
}	


//--------------------------------------------------------------------------------
//  	
	void
	CSysProgressMgr::ReportProgressBegin
	(
	    int type, 
	    const char *id, 
	    const char *message
	)
//
//	
//
//------------------------------------------------------------------------------//
{
    int need_notify = 0;
    CSysProgressObj *obj = NULL;
    
    if(id == NULL) return;
    
    Lock();
    
    if(m_list == NULL) need_notify = 1;
    
    obj = GetSysProgressObj(id);
    if(obj == NULL)
    {
        // new obj
        obj = new CSysProgressObj(type, id, message);
        InsertSysProgressObj(obj);
        
    }
    
    Unlock();    
    
    // notify event
    if(need_notify)
    {
        char event_data[RDM_MAX_NOTIFICATION_DATA];
        sprintf(event_data, "<ProgressType> 1 </ProgressType><ProgressID> DB1 </ProgressID><Message> Databse download beginning </Message>");        

        m_pRDM->nm.Notify("System", RDM_EC_System_Progress_Begin, event_data);
    }
}	


//--------------------------------------------------------------------------------
//  	
	void
	CSysProgressMgr::ReportProgressEnd
	(
	    const char *id
	)
//
//	
//
//------------------------------------------------------------------------------//
{
    CSysProgressObj *obj = NULL;
    
    if(id == NULL) return ;
    
    Lock();
    
    obj = GetSysProgressObj(id);
    if(obj)
    {
        RemoveSysProgressObj(id);
    }
    else
    {
        //shouldn't be here
        printf("ReportProgressEnd(): Error - Can not find id[%s] in list\n", id);
    }
    
    Unlock();
    
    // notify event
    if(m_list == NULL)
    {
        char event_data[RDM_MAX_NOTIFICATION_DATA];
        sprintf(event_data, "<ProgressType> 1 </ProgressType><ProgressID> DB1 </ProgressID><Message> Databse download finished </Message>");        

        m_pRDM->nm.Notify("System", RDM_EC_System_Progress_End, event_data);
    }
    
    return;
}	



