/**	
 *	@file	RDM_DDA_Port.h
 *	@brief	Public Device Data API (DDA) for Ports
 *
 *	
 */

#ifndef _RDM_DDA_Port_h_
#define _RDM_DDA_Port_h_

#include "pp/RDM.h"
#include "pp/RDM_BranchManager.h"

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

 class CDDA_Device;

/*----------------------------------------
 *	Port Class
 *--------------------------------------*/

class CDDA_Port : public CRDM_BranchManager
{
protected:
    /**
     * Private minimal constructor, used internally only by derived objects
     * that call initialize themselves.
     * @param  pRDM            Ptr to RDM object
     */
    CDDA_Port (CRDM* pRDM);

public:
	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Constructor for a device
	 *
	 *	Create a new device node in the database.
	 *
	 *	@param	pRDM			Ptr to RDM object
	 *	@param	pParentDevice	Ptr to the parent device object
	 *  @param	pClass			Port Class ("KVM","Serial","Outlet")
	 *	@param	pType			Port Type
	 *	@param	pName			The name of the port
	 *	@param	index			The port index within the device (-1 = no index attribute)
	 *	@param	access			!0 = Add SecurityID="Access" (Defualt = 1)
	 *	@param	internal		!0 = this is an internal port (Defualt = 0)
	 *	@param	capacity		Capacity attribute (# of associated paths)
	 */

	CDDA_Port
	(
		CRDM * pRDM, 
		CDDA_Device *pParentDevice,		// Device object this port belongs to
		const char	*pID,					// id of the device
		const char	*pClass,				// Class ("KVM","Serial","Outlet")
		const char	*pType,					// Type Attribute
		const char	*pName,					// Port name
		int		index,					// >= 0 - port index < 0  = no index attribute
		int		access = 1,				// !0 = Add SecurityID="Access"
		int		internal = 0,			// !0 = this is an internal port
		int		capacity = 1			// Capacity attribute
	);

    ~CDDA_Port();

protected:
    /**
     * Internal code of public constructor, used for decoupled initialization
     * in subclasses. Must be called in public constructors.
     * @see CDDA_Port
     */
    void initialize
    (
        CRDM * pRDM, 
        CDDA_Device *pParentDevice,     // Device object this port belongs to
        const char    *pID,                   // id of the device
        const char    *pClass,                // Class ("KVM","Serial","Outlet")
        const char    *pType,                 // Type Attribute
        const char    *pName,                 // Port name
        int     index,                  // >= 0 - port index < 0  = no index attribute
        int     access = 1,             // !0 = Add SecurityID="Access"
        int     internal = 0,           // !0 = this is an internal port
        int     capacity = 1            // Capacity attribute
    );
    
public:
	/*  --------------------------------------------------------------------*/
	/** 
	 *	Functions to change port attributes.
	 *  Must call BeginUpdate() before changing attributes and EndUpdate()
	 *  when finished changing attributes.
	 *  BeginUpdate() locks the database and EndUpdate() unlocks the database
	 *  and then send the RDM events
	 * 
	 */

	void BeginUpdate();
	void SetStatus( int status );
	void SetName( const char *pName );
	void SetClass( const char *pClass );
	void SetType( const char *pType );
	void SetConnection( const char * pConnection );
	void SetAssociatedPort( const char * pID );
	void SetAssociatedOutlet( const char *pID );
	void EndUpdate();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Utility functions
	 * 
	 */

	const char *GetID();		///< Returns the port ID

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Called when data needs to be saved for the device
	 *
	 *	Called when there is an update to the writable port elements.
	 *  This function can be overridden and so the data can be stored in an
	 *  external config database.
	 *  The default implementation saves the data under the ID key int he 
	 *  RDM DeviceSettings branch.
	 *  CommitData() will be called after all data has been written.
	 * 
	 *  Possible keys: Name, AssociatedPort, AssociatedOutlet[x],
	 * 		DataRate, Parity, StopBits, DataBits, FlowControl
	 * 
	 *	@param	pID				The ID of this port
	 *	@param	pKey			The data key name, such as "Name"
	 *	@param	pData			The 
	 *	@param	pName			The name of the device
	 *	@param	baseDevice		!0 = this is the base device
	 */

	virtual int  WriteData( const char *pID, const char *pKey, char *pData );
	virtual void CommitData();

	/*  --------------------------------------------------------------------*/
	/** 
	 *	@brief	Called to retrieve persistant port data
	 *
	 *	Called when there is an update to the writable port elements.
	 *  This function can be overridden and so the data can be stored in an
	 *  external config database.
	 *  The default implementation saves the data under the ID key int he 
	 *  RDM DeviceSettings branch.
	 *  CommitData() will be called after all data has been written.
	 * 
	 *  Possible keys: Name, AssociatedPort, AssociatedOutlet[x],
	 * 		DataRate, Parity, StopBits, DataBits, FlowControl
	 * 
	 *	@param	pID				The ID of this port
	 *	@param	pKey			The data key name, such as "Name"
	 *	@param	pData			The 
	 *	@param	pName			The name of the device
	 *	@param	baseDevice		!0 = this is the base device
	 */

	virtual int ReadData( const char *pID, const char *pKey, const char *pData );

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Power related virtual hardware dependent functions
	 * 
	 */

	virtual
	int								// 0 or error code
	AssociateOutlet
	(
		const char	*port_id,			// The target that will be controlled

		const char	*outlet_id			// The outlet that controls it
	);

	virtual
	int								// 0 or error code
	SetPowerState
	(
		const char	*outlet_id,			// The outlet to set the power state for
		int		state				// 0 = cycle power, 1 = On, 2 = Off
	);

	virtual
	char *							// Ptr to the allocated power status string
	GetPowerStripStatus
	(

		const char	*strip_id			// The outlet strip to get the status for
	);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Private function that implement the RDM branch manager interface.
	 * 
	 */

	virtual
	int									// 0 or Error code
	Update
	(
		CSession	*pUserSession,	// User session or NULL for super user
		CSXDB_Node	*pNode,			// The Node to be appended too
		CSXDB_Node	*pData			// Ptr to data to append
	);

	/*  --------------------------------------------------------------------*/
	/** 
	 *	Private Data
	 * 
	 */

private:
	CRDM * pRDM;			///< Ptr to the RDM object

	int    sendEvents;		///< !0 = Send events for changes
	char   id[RDM_MAX_ID];	///< Our ID
};

#endif // _RDM_DDA_Port_h_

