/*---------------------------------------------------------------------------

	Session.h

	Copyright (c) 2003, Raritan Computer, Inc.

	Header file for RDMP_Service.cpp and the CRDMP_Service class

---------------------------------------------------------------------------*/

#ifndef _Session_H_
#define	_Session_H_

#include "pp/OS_Port.h"
#include "pp/UserObject.h"
#include "pp/RDM_Definitions.h"

//----------------------------------------
//				Equates
//----------------------------------------

	// Session Events

enum
{
	SESSION_EVENT_PREOPEN = 0,			// A New session is about to open
	SESSION_EVENT_OPEN,					// Session is open
	SESSION_EVENT_CLOSE,				// Session is closing
	SESSION_EVENT_SHUTDOWN,				// Session manager is shutting down
	SESSION_EVENT_CONNECT,				// A connection was made to this session
										// data = ptr to protocol name
	SESSION_EVENT_DISCONNECT,			// A protocol was disconnected from this session
										// data = ptr to protocol name
	SESSION_EVENT_MAX
};

//----------------------------------------
//				Data Types
//----------------------------------------

	// Definition of call back function

typedef void (CSession_CallBack)
			(
				void	 *	pUserData,
				class CSession *	pUserSession,		// The user session or NULL for super user
				int			eventID,			// Numeric event ID
				const void	 *	pData				// Data associated with the event
			);

//----------------------------------------
//				Static Data
//----------------------------------------

	DWORD
	GarbageCollectThread
	(
		LPVOID			lpParam			// The thread parameter
	);

// forward declaration
class CSessionManager;

//----------------------------------------
//				Class CSessionObject
//----------------------------------------

class	CSessionObject
{
public:

	CSessionObject
	(
		int		id = 0,
		const char *	pName = NULL,
		CSession_CallBack *pCallBack = NULL,
		void *	pUserData = NULL
	);

	virtual
	~CSessionObject();

	inline
	void
	SetClassID( int id )
	{
		classID = id;
	}

	inline
	void
	SetName( const char *pNewName )
	{
		pName = pNewName;
	}

	virtual
	void
	Event
	(
		CSession	* pUserSession,		// The session in question...
		int			event,				// Which SESSION_EVENT_ code
		const void		*pData				// Data associated with the event
	);

protected:

	int					classID;		// ID of the class this object is
	const char				*pName;			// Ptr to the name of the object;
	CSession_CallBack	*pCallBack;		// Ptr to the user's call back function
	void				*pUserData;		// User defined data

	class CSessionObject *pNext;		// Linked list

	friend class CSessionObjectList;
};

//----------------------------------------
//				Class CSessionObjectList
//----------------------------------------

class	CSessionObjectList
{
public:
	CSessionObjectList();

	virtual
	~CSessionObjectList();

	void
	AddSessionObject
	(
		CSessionObject		*pObject		// The object to add
	);

	void
	RemoveSessionObject
	(
		CSessionObject		*pObject		// The object to remove
	);

	CSessionObject *					// Ptr to next object or NULL
	EnumObject
	(
		CSessionObject		*pObject	// NULL to find first, or that last
										// last found to find next.
	);

	CSessionObject *					// Ptr to the next object or NULL
	EnumObjectByName
	(
		const char		*pName,		// Ptr to the object name
		CSessionObject		*pObject	// NULL to find first, or that last
										// found to find next.
	);

	CSessionObject *					// Ptr to the next object or NULL
	EnumObjectByClass
	(
		int					classID,	// The class ID to search for
		CSessionObject		*pObject	// NULL to find first, or that last
										// found to find next.
	);

	void
	DeleteObjects
	(
	);

	virtual
	void
	Event
	(
		CSession	* pUserSession,		// The session in question...
		int			event,				// Which SESSION_EVENT_ code
		const void		*pData				// Data associated with the event
	);

protected:

	void
	LockList
	(
	);

	void
	UnlockList
	(
	);

protected:

	OS_CRITICAL_SECTION	cs;				// Critical section for session list

	class CSessionObject *pObjectList;	// Linked list

	friend class CSession;
	friend class CSessionManager;
};

//----------------------------------------
//				Class CSession
//----------------------------------------

class	CSession : public CSessionObjectList
{
protected:
	CSession
	(
		class CRDM	*	pRDM,			// Ptr to the RDM object
		CSessionManager	*pMgr			// Ptr to the manager
	);

	virtual
	~CSession();

public:
	void
	Aquire
	(
	);

	void
	Release
	(
	);

	void
	SetUserObject
	(
		CUserObject	*pUserObject	// Ptr to the user object for this session
	);

    void
    SetSessionSignature
    (
        const char *sessionId,
        const char *sessionKey
    );

	CUserObject *
	GetUserObject();

	DWORD
	GetPermission
	(
		const char *SecurityId, 
		const char *uniqueId
	);

	DWORD
	GetNodePermission
	(
		const char *SecurityId
	);

	inline
	int
	IsClosed()
	{ return closed; }

	inline
	const char *
	GetSessionID( void )
	{ return sessionID; }

	inline
	const char *
	GetSessionKey( void )
	{ return sessionKey; }

	void
	UpdateUserData
	(
	);

    void
    AddUsedResourceWithAttribute
    (
        const char    *pResourceID,
        const char    *pAttr1,
        const char    *pData1,
        const char    *pAttr2,
        const char    *pData2
    );
 
    void
    DeleteResourceAttribute
    (
        const char    *pResourceID,
        const char    *pAttr
    );
   
    void
    ModifyUsedResourceAndAttribute
    (
        const char    *pResourceID,
        const char    *pAttr1,
        const char    *pAttr2,
        const char    *pDeviceID,
        const char    *pData1,
        const char    *pData2
    );
 
    void
    AddDeviceAndInputPort
    (
       const char   *DeviceID,
       const char   *InputPort
    );

	void
	AddUsedResource
	(
		const char	*pResourceID,			// Ptr to the resource ID
        BOOL    referral = FALSE
	);

	void
	SubtractUsedResource
	(
		const char	*pResourceID			// Ptr to the resource ID
	);

	void
	Close
	(
	);

    BOOL 
    IsResourceUsed
    (
        const char *pResourceId
    );

    const char *
    GetResourceID
    (
		BOOL  referral = FALSE 
    );
   
    const char *
    GetElementData
    (
       const char *xpath,
       const char *pElement
    );
   
    const char *
    GetAttributeData
    (
       const char *xpath,
       const char *pAttribute
    );

	inline void SetClientIPAddress ( DWORD _ipAddress) { this->ipAddress = _ipAddress; };

	inline DWORD GetClientIPAddress () { return this->ipAddress; }; 

protected:

	CUserObject		*pUserObject;		// Ptr to the user object for this session

	DWORD	ipAddress;

	char	sessionKey[RDM_MAX_KEY_BASE64];// 128-bit key in base64

	char	sessionID[RDM_MAX_ID];		// The unique session ID
	int		used;						// # of threads/modules that are using this session object
										// if used ever == 0, then the session is deleted
	int		closed;						// !0 = this session is closed
	class CSession		 *pNext;		// Linked list
	class CSessionManager	*pMgr;		// Ptr to the manager
	class CRDM	*	pRDM;				// Ptr to the root RDM object

//	CSXDB_Node	*pSessionNode;			// Ptr to our sessions node

	friend class CSessionManager;

private:

	void
	WriteSessionData
	(
	);

};


//----------------------------------------
//				Class CSessionManager
//----------------------------------------

class	CSessionManager : public CSessionObjectList
{
public:
	CSessionManager
	(
		class CRDM	*	pRDM			// Ptr to the RDM object
	);

	virtual
	~CSessionManager();

	CSession *							// Ptr to the new session or NULL
	NewSession
	(
	);

	void
	AddSession
	(
		CSession *session				// Ptr to the session to add to the session list
	);

	void
	RemoveSession
	(
		CSession *session				// Ptr to the session to remove from the session list
	);

	int
	GetSessionCount
	(
	);

	CSession *							// Returns ptr to session or NULL
	GetSession
	(
		const char	*pSessionID				// session ID of the session to get
	);

	CSession *							// Returns ptr to session or NULL
	EnumSession
	(
		CSession *pSession				// NULL to get first, last found
										// to get next
	);

	void
	CloseAllOtherSessions
	(
		CSession *pSess
	);

	void
	EnableSessions
	(
	);

	void
	Shutdown
	(
	);

	void
	QueueGarbageCollect
	(
	);

	void
	GarbageCollect
	(
	);

    BOOL 
    IsResourceUsed
    (
        const char *pResourceId
    );

protected:

	int	sessionCount;					// # of active sessions
	int	sessionsDisabled;				// 0 = New sessions are allowed
	int	nextSessionID;					// Session ID counter
	CSession * pSessionList;			// Ptr to the first session in list
	class CRDM	*	pRDM;				// Ptr to the root RDM object

	OS_THREAD	hGarbageCollectThread;	// The server's clean up thread
	OS_EVENT	hGarbageCollectEvent;	// Something to clean up when triggered

	friend class CSession;
	friend DWORD GarbageCollectThread
				(
					LPVOID			lpParam			// The thread parameter
				);
};

//----------------------------------------
//			Global Function Prototypes
//----------------------------------------



#endif // _Session_H_

