/**	
 *  @file	RFPUnpack.h
 * 
 *  @brief	CRFPUnpack definition
 * 
 *  Class used to read RFP files
 *	
 */

#ifndef _RFPUnpack_h_
#define _RFPUnpack_h_
#include "pp/SIO.h"
#include "pp/RFPParser.h"

namespace pp
{

/*----------------------------------------
 *	Equates
 *--------------------------------------*/

/*----------------------------------------
 *	Data Types
 *--------------------------------------*/

/*----------------------------------------
 *	Function Prototypes
 *--------------------------------------*/

/*----------------------------------------
 *	Forward References
 *--------------------------------------*/

/*----------------------------------------
 *	Class
 *--------------------------------------*/

class CRFPUnpack : public CRFPParser
{
public:

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Constructor for a CRFPUnpack object
	 *
     *  The CRFP object processes 1 RFP file. Processing includes
     *  parsing the header, decrypting files, checking signatures,
     *  calling virtual functions to run shell scripts and copying
     *  files to destination locations.
	 *
	 */

	CRFPUnpack( );
	virtual ~CRFPUnpack();

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Set the default Key files
	 *
     *  @param	pRSAKeyFile	Ptr to the RSA public key File Name
     *                      (NULL = use default)
     *  @param	pRC4KeyFile	Ptr to the RC4 File Name (NULL = use
     *                      default)
	 *
	 */

	void SetKeyFiles( char * pRSAKeyFile = NULL, char * pRC4KeyFile = NULL  );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Creates the RFP File
     * 
     *  The output of PackRFP is sent to the RFPOutput() function.
     *  The default behavior of RFPOutput() is to write the data to
     *  an SIO object. However, the user can override this method to
     *  send the information someplace else, like across a netowrk
     *  connection.
     * 
     *  NOTE: Only the RFPOutput function writes to the pSIO. If
     *  this function is overriddem, pSIO can be NULL.
	 *
     *  @param	pSIO        Where to output the RFP file.
     *  @return				0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
	 *
	 */

	int UnpackRFP( CSIO * pSIO = NULL );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Called by UnpackRFP to recieve the RFP data.
     * 
     *  The default behavior is to read the data from the pSIO that
     *  was passed in to RFPUnpack().
     * 
     *  @param  pSIO        SIO object to read from (can be ignored
     *                      if overridding this function)
     *  @param  pData       Ptr to buffer containing the data
     *  @param  dataLength  # of bytes to read (1-64k)
     *  @param  pBytesRead  # of bytes read (0 = EOF)
     *  @return				0 = no error, !0 = RFP_ERROR_ @see
     *                      RFPDef.h
	 *
	 */

	virtual int RFPInput( CSIO * pSIO, char * pData, int dataLength, int *pBytesRead );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Resolves A metafile into a realfile name
     * 
     *  This function can be overridden to resolve metafiles.
     *  Te default implmentation returns an error.
     * 
     *  @param  pMetaFileName   Ptr to the Metafile to resolve
     *  @param  pOutRealFileName Buffer to return the real file name
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be resolved
	 *
	 */

	virtual int ResolveMetaFile( char *pMetaFileName, char *pOutRealFileName );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Cleans up after a Metafile.
     * 
     *  This function can be overridden to do any needed cleanup of
     *  resolved metafiles.
     *  The default implementation does nothing.
     * 
     *  @param  pMetaFileName   Ptr to the Metafile to resolve
     *  @param  pResolvedFileName Buffer containing the real file
     *                            name
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be resolved
	 *
	 */

	virtual int ReleaseMetaFile( char *pMetaFileName, char *pResolvedFileName );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Called to Validate the header-version, model & etc.
     * 
     *  The default behavior is to return RFP_OK for all headers.
     * 
     *  @param  pRFP        Ptr to the RFP header structure
     *  @return             RFP_OK, RFP_SKIP_FILE or RFP_ERROR_
	 *
	 */

	virtual int ValidateHeader( RFP * pRFP );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Called to Validate a file-version, model & etc.
     * 
     *  The default behavior is to return RFP_OK for all files.
     * 
     *  @param  pRFP_File   Ptr to the RFP file structure
     *  @return             RFP_OK, RFP_SKIP_FILE or RFP_ERROR_
	 *
	 */

	virtual int ValidateFile( RFP_FILE * pRFP_File );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Called to run a script file
     * 
     *  After a script file has been downloaded, this method is
     *  called to exeute the script. This function should be
     *  overriden.
     * 
     *  @param  pFileName   Ptr to the script file name
     *  @param  pScriptType Ptr to the script type (i.e. "BASH")
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be executed
	 *
	 */

	virtual int RunScript( char *pFileName, char * pScriptType );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Called to create the SIO for a file download.
     * 
     *  This function is called once for each file in the RFP.
     *  The default behavior is to create a CSIO_FP for the file.
     *  NOTE: If an CSIO_FP is used, in must be open in mode "w+".
     * 
     *  @param  pRFP_File   Ptr to the RFP_FILE struct
     *  @param  ppSIO       Ptr to an SIO Ptr. Used to return the
     *                      allocated SIO
     *  @return				Returns RFP_OK, RFP_SKIP_FILE or RFP_ERROR_ if file
     *                      cannot be processed
	 *
	 */

	virtual int AllocateSIO( RFP_FILE * pRFP_File, CSIO **ppSIO );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Called after a file has been written to the SIO
     * 
     *  The default behavior is to close and then delete the SIO
     *  object.
     * 
     *  @param  pRFP_File   Ptr to the RFP_FILE struct
     *  @param  pSIO        Ptr to the SIO for the file
	 *
	 */

	virtual void FreeSIO( RFP_FILE * pRFP_File, CSIO *pSIO );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief  Called after a file has been downloaded.
     * 
     *  The SIO is open and positioned at the begining of the file.
     * 
     *  @param  pRFP_File   Ptr to the RFP_FILE struct
     *  @param  pSIO        Ptr to the SIO for the file
     *  @return				Returns RFP_OK or RFP_ERROR_ if file
     *                      cannot be processed
	 *
	 */

	virtual int PostProcessFile( RFP_FILE * pRFP_File, CSIO *pSIO );

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Returns the state of the RFP processing
	 *
     *  @return				Returns RFP_STATE_ values
	 *
	 */

	inline int GetRFPState() {return state;}

	/*  --------------------------------------------------------------------*/
	/** 
     *  @brief	Returns the file index for the last error (-1 ==
     *         header)
	 *
     *  @return             error file index
	 *
	 */

	inline int GetErrorFile() {return errorFile;}

	/*  --------------------------------------------------------------------*/
	/*  --------------------------------------------------------------------*/
private:


private:
	int		state;						// Current processing state
	char	rsaKeyFile[RFP_MAX_FILES];// Default RSA public key file
	char	rc4KeyFile[RFP_MAX_FILES];	// Default RC4 key file
	char * 	realFileName[RFP_MAX_FILES][RFP_MAX_FILENAME];	// Name of the REAL source file
	int		errorFile;					// Indicates which file caused the last error (-1 == header);
};

} // namespace

#endif	// _RFPUnpack_h_


