/*---------------------------------------------------------------------------

	CircularFile.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Generic Circular file (round robin) file implementation.

	This class takes in variable length entries and stores them in the circular
	file. The entries can be retrieved starting from the last in or the first in.
	The circular file is created with a maximum size.
	This class can work with two files, the work file and the commit file or just
	the work file.

	NOTE: Current implementations is limited to entry sizes of 255.
		  To allow for larger entries, change CG_LENGTH_SIZE and CF_MAX_SIZE.

----------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<string.h>
#include	"pp/CircularFile.h"

//----------------------------------------
//				Equates
//----------------------------------------

#define		CF_FILE_SIGNATURE	0x244b47e3

#define		CircularAdd(a,b) (((a)+(b)) % header.maxSize)
#define		CircularSub(a,b) (((a) - (b)) + ((a) < (b) ? header.maxSize : 0)) 

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//----------------------------------------------------------------------------
//
	CCircularFile::CCircularFile
	(
	)
//
//	Sets the work file. If it is an CSIO_FP, then the file must be open.
//
//--------------------------------------------------------------------------//
{
	writeHeader = 0;
	rdPosition = 0;
	memset(&header,0,sizeof(header));
}

//----------------------------------------------------------------------------
//
	CCircularFile::~CCircularFile
	(
	)
//
//	Clean up
//
//--------------------------------------------------------------------------//
{
	Flush();
}

//----------------------------------------------------------------------------
//
	int									// <0 = error, 0 = initialized, 1 = ok
	CCircularFile::Initialize
	(
		CSIO	*	pNewFile,			// The working file
		int		size				// Desired size of the file
	)
//
//	This method initializes the the file operations. The file is checked
//	for proper form. If it is not well formed, then it is initialized to
//	a null file.
//	pFile must already be open and remains open as through the life
//	of this object.
//
//--------------------------------------------------------------------------//
{
	int				result;
	char			data;

	syncPosition = 0;
	syncLength	 = 0;

	pFile	 = pNewFile;

	// subtract room for the header from the size

	size -= sizeof(header);

	// Check the header

	pFile->Seek(0,SEEK_SET);
	result = pFile->Read( (char *) &header, sizeof(header) );

	if (result < (int)sizeof(header))
		goto INIT;

	if (	header.signature != CF_FILE_SIGNATURE || 
			header.usedSize != header.maxSize || 
			header.lengthSize != CF_LENGTH_SIZE ||
			header.gapPosition > header.maxSize ||
			header.maxSize != size)
	{
		result = 0;
		goto INIT;
	}

	// Check all entries

	Rewind();

	do
	{
		result = Read( &data, 1 );
	} while (result > 0);

	if (result != EOF && result < 0)
		goto INIT;

	// Everything looks good,

	Rewind();

	return 1;

	// File is bad, initialize
INIT:

	header.signature	= CF_FILE_SIGNATURE;
	header.maxSize		= size;
	header.usedSize		= size;
	header.lengthSize	= CF_LENGTH_SIZE;
	header.gapPosition	= 0;
	header.gapLength	= size;
	header.userData		= 0;

	writeHeader			= 1;

	Rewind();

	pFile->Seek(0,SEEK_SET);
	pFile->Write(  (char *) &header, sizeof(header) );

	return result;
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::Write
	(
		char	*	pData			// Ptr to the entry data to write
	)
//
//	Writes an entry into the circular file.
//	Just another form of write.
//
//--------------------------------------------------------------------------//
{
	return Write(pData, strlen(pData));
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::Write
	(
		char	*	pData1,			// Ptr to the entry data to write
		int			length1,		// # of bytes of data to write
		char	*	pData2,			// Second segment
		int			length2,		// 
		char	*	pData3,			// Third segment
		int			length3,		// 
		char	*	pData4,			// Fourth segment
		int			length4			// 
	)
//
//	Writes an entry into the circular file.
//	An entire entry must be written at once. For convinience, this method
//	takes ptrs to 4 segments of data that will all be written as one entry.
//	Segments 2,3,4 default to null.
//
//--------------------------------------------------------------------------//
{
	int		length;
	CF_LENGTH_TYPE	len;
	int		result;

	length = length1 + length2 + length3 + length4 + CF_LENGTH_SIZE*2;

	// Check the max length of the entry

	if (length > CF_MAX_SIZE)
		return -1;

	if (length == 0)
		return 0;
	
	// Make room for the new entry

	while (length > header.gapLength)
	{
		// we need to delete entries to make room

		WrapSeek( header.gapPosition + header.gapLength );
		result = WrapRead( (char *) &len, CF_LENGTH_SIZE );

		if (result < 0) return result;

		if (rdPosition == header.gapPosition + header.gapLength)
			rdPosition = CircularAdd( rdPosition, len+CF_LENGTH_SIZE*2 );

		header.gapLength += len+CF_LENGTH_SIZE*2;
	}

	// Write the entry and the new gap
	
	len = (CF_LENGTH_TYPE) length - CF_LENGTH_SIZE*2;

	WrapSeek( header.gapPosition );
	result = WrapWrite(  (char *) &len, CF_LENGTH_SIZE );
	if (result < 0) return result;

	WrapWrite( pData1, length1 );
	if (length2)
	{
		result = WrapWrite( pData2, length2 );
		if (result < 0) return result;
	}

	if (length3)
	{
		result = WrapWrite( pData3, length3 );
		if (result < 0) return result;
	}

	if (length4)
	{
		result = WrapWrite( pData4, length4 );
		if (result < 0) return result;
	}
	result = WrapWrite(  (char *) &len, CF_LENGTH_SIZE );

	header.gapPosition = (header.gapPosition + length) % header.maxSize;
	header.gapLength -= length;

	writeHeader = 1;

	return 0;
}

//----------------------------------------------------------------------------
//
	int								// # of bytes read or error code
	CCircularFile::Read
	(
		char	*	pData,			// Where to put the data
		int			maxLength,		// Max # of bytes that can be written to pData
									// Default = 0 = no limit
		int			direction		// >=0 = read next entry
									// < 0 = read previous entry
									// 0 = Don't move file position
	)
//
//	Reads at the current read position. Returns ROF if there are no more
//	entries to read. Use Rewind to start reading from the oldest.
//
//--------------------------------------------------------------------------//
{
	CF_LENGTH_TYPE	len;
	int		result;

	if (maxLength == 0)
		maxLength = header.maxSize;

	if (direction >=0)
	{
		if (rdPosition == header.gapPosition)
			return EOF;

		WrapSeek( rdPosition );

		result = WrapRead(  (char *) &len, CF_LENGTH_SIZE );
		if (result < 0) return result;

		if (len < maxLength)
			maxLength = len;

		result = WrapRead( pData, maxLength );

		if (direction > 0)
			rdPosition = CircularAdd( rdPosition, len+CF_LENGTH_SIZE*2 );
	}
	else
	{
		if (rdPosition == (header.gapPosition + header.gapLength) % header.maxSize)
			return EOF;
		
		WrapSeek( rdPosition - CF_LENGTH_SIZE ); // Move to length of last entry

		result = WrapRead(  (char *) &len, CF_LENGTH_SIZE );
		if (result < 0) return result;

		rdPosition = CircularSub( rdPosition, len+CF_LENGTH_SIZE*2 );

		WrapSeek( rdPosition+CF_LENGTH_SIZE );

		if (len < maxLength)
			maxLength = len;

		result = WrapRead( pData, maxLength );
	}

	return result;
}

//----------------------------------------------------------------------------
//
	void
	CCircularFile::Rewind
	(
	)
//
//	Moves the read position to the beginning of the file (oldest entry)
//
//--------------------------------------------------------------------------//
{
	rdPosition = (header.gapPosition + header.gapLength) % header.maxSize;
}

//----------------------------------------------------------------------------
//
	int									// # of entries moved
	CCircularFile::Seek
	(
		int		distance				// # of entries to move
										// negative = move backwards
										// CF_OLDEST = moves to first entry (oldest)
										// CF_NEWEST  = moves ot last entry (newest)
	)
//
//	Moves relative to the current position. Negative moves backwards.
//	Zero is returned if the file cannot seek any farther.
//	NOTE: If CF_OLDEST or CF_NEWEST are used, then 0 is returned.
//
//--------------------------------------------------------------------------//
{
	int		moved = 0;
	int		result;
	CF_LENGTH_TYPE	len;

	if (header.gapLength == header.maxSize)
		return 0; // file is empty

	if (distance == (int)CF_OLDEST)
		rdPosition = (header.gapPosition + header.gapLength) % header.maxSize;

	else if (distance == CF_NEWEST)
	{
		rdPosition = header.gapPosition;
		Seek( -1 );
	}

	else if (distance > 0)
	{
		while (distance > 0)
		{
			if (rdPosition == header.gapPosition)
				break;

			WrapSeek( rdPosition );

			result = WrapRead(  (char *) &len, CF_LENGTH_SIZE );
			if (result < 0) break;;

			rdPosition = CircularAdd( rdPosition, len+CF_LENGTH_SIZE*2 );

			moved++;
			distance--;
		}
	}

	else
	{
		while (distance < 0)
		{
			if (rdPosition == (header.gapPosition + header.gapLength) % header.maxSize)
				break;
			
			// Move to length of last entry

			WrapSeek( rdPosition - CF_LENGTH_SIZE );

			result = WrapRead(  (char *) &len, CF_LENGTH_SIZE );
			if (result < 0) break;

			rdPosition = CircularSub( rdPosition, len+CF_LENGTH_SIZE*2 );

			moved++;
			distance++;
		}
	}

	return moved;
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::Flush
	(
	)
//
//	For SIO_FP files, this calls fflush() for the file
//
//--------------------------------------------------------------------------//
{
	int result;

	if (writeHeader)
	{
		pFile->Seek(0,SEEK_SET);
		result = pFile->Write(  (char *) &header, sizeof(header) );

		if (result <0)
			return result;

		writeHeader = 0;
	}

	return pFile->Flush();
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::Sync
	(
		CSIO	*pDestFile			// File to write to
	)
//
//	Syncronizes the log file SIO (usually memory) with another file (usually
//	on disk).
//
//--------------------------------------------------------------------------//
{
	int	lengthToGo = syncLength;
	int	length;
	int	result = 0;
	int	headerWritten = 0;

	// Make sure the file is up to date

	Flush();

	// See if there is anything to write

	if (!needSync && !writeHeader)
		return 0;

	// See if the whole file needs to be updated

	if (syncLength >= header.maxSize)
	{
		pDestFile->Seek( 0,SEEK_SET );
		result = pFile->CopyTo( pDestFile );
		lengthToGo = 0;
	}

	else if (syncPosition+syncLength > header.maxSize || syncPosition == 0)
	{
		// The updated portion wrapped around to the beginning of the
		// file, so we will first sync the data at the begining of the file.

		if (syncPosition == 0)
			length = syncLength;
		else
			length = syncPosition+syncLength - header.maxSize;

		lengthToGo -= length;

		result = pFile->CopyTo( pDestFile, 0, length + sizeof(header), 0 );

		if (result < 0)
			return result;

		headerWritten = 1;
	}

	// Now write the data near the end of the file

	if (lengthToGo > 0)
	{
		result = pFile->CopyTo( pDestFile, syncPosition + sizeof(header), lengthToGo, syncPosition + sizeof(header) );
		lengthToGo = 0;
	}

	// Write the header if not already done

	if (!headerWritten)
	{
		result = pFile->CopyTo( pDestFile, 0, sizeof(header), 0 );

		if (result != 0)
			return result;
	}

	// Clear the sync flags

	needSync = 0;
	syncPosition = 0;
	syncLength = 0;

	return result;
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::WrapSeek
	(
		int		position
	)
//
//	Seeks to the position, wrapping around if the position is past the end
//
//--------------------------------------------------------------------------//
{
	seekPosition = position = position % header.maxSize;
	return pFile->Seek(position + sizeof(header),SEEK_SET);
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::WrapRead
	(
		char	*	pData,			// Where to put the data
		int			length			// # of bytes to read
	)
//
//	Reads from the file, wrapping around if the read goes past the end
//
//--------------------------------------------------------------------------//
{
	int	count;
	int	numRead = 0;
	int	fPos;
	int	result;

	while (length)
	{
		fPos = pFile->Tell() - sizeof(header);

		if (fPos == header.maxSize)
		{
			fPos = 0;
			pFile->Seek(sizeof(header), SEEK_SET);
		}

		if (fPos + length > header.maxSize)
			count = header.maxSize - fPos;
		else
			count = length;

		result = pFile->Read( pData, count );

		if (result < 0)
			return result;
		if (result == 0)
			return -1;

		length -= result;
		pData += result;
		numRead += result;
	}

	return numRead;
}

//----------------------------------------------------------------------------
//
	int								// 0 or error code
	CCircularFile::WrapWrite
	(
		char	*	pData,			// Where to put the data
		int			length			// # of bytes to read
	)
//
//	Writes to the file, wrapping around if the read goes past the end
//
//--------------------------------------------------------------------------//
{
	int	count;
	int	numWritten = 0;
	int	fPos;
	int	result;

	// Keep track of what part of the file has been updated

	if (needSync == 0)
	{
		needSync = 1;
		syncPosition = seekPosition;
		syncLength = 0;
	}

	syncLength += length;

	// Write the data

	while (length)
	{
		fPos = pFile->Tell() - sizeof(header);

		if (fPos == header.maxSize)
		{
			fPos = 0;
			pFile->Seek(sizeof(header), SEEK_SET);
		}

		if (fPos + length > header.maxSize)
			count = header.maxSize - fPos;
		else
			count = length;

		result = pFile->Write( pData, count );

		if (result < 0)
			return result;
		if (result == 0)
			return -1;

		length -= result;
		pData += result;
		numWritten += result;
	}

	return numWritten;
}



