/*-------------------------------------------------------------------------------

	SXDB_Cmd.cpp

	Copyright (c) 2002, Raritan Computer, Inc.

	Processes database commands to an SXDB database.

	<database Name="DataBaseName")

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>

#include	"pp/SXDB_Cmd.h"

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CSXCMD
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSXCMD::CSXCMD
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	pNext = NULL;
	pDB = NULL;
	pNodes = NULL;
	pSubNodes = NULL;
	depth = 0;
}

//--------------------------------------------------------------------------------
//
	CSXCMD::~CSXCMD
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	if (pDB != NULL)
		delete pDB;
	if (pNodes != NULL)
		delete pNodes;
	if (pSubNodes != NULL)
		delete pSubNodes;
}

//--------------------------------------------------------------------------------
//									CSXCMD_List
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSXCMD_List::CSXCMD_List
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	pCmd = NULL;
	pCmdList = NULL;
}

//--------------------------------------------------------------------------------
//
	CSXCMD_List::~CSXCMD_List
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	CSXCMD	*	p;
	CSXCMD	*	pNext;

	p = pCmdList;

	while (p != NULL)
	{
		pNext = p->pNext;
		delete p;
		p = pNext;
	}

}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::Event
	(
		int			event
	)
//
//	Creates nodes for each part of the xml file.
//
//------------------------------------------------------------------------------//
{
	if (event == SXML_EVENT_START_DOCUMENT)
	{
		if (pCmdList != NULL)
		{
			delete pCmdList;
			pCmdList = NULL;
		}

		state = SXCMD_STATE_NULL;

		return 0;
	}

	switch (state)
	{
		case SXCMD_STATE_NULL:			return State_Null(event);			break;
		case SXCMD_STATE_DATABASE:		return State_Database(event);		break;
		case SXCMD_STATE_GET:			return State_Get(event);			break;
		case SXCMD_STATE_XPATH:			return State_XPath(event);			break;
		case SXCMD_STATE_APPEND_UPDATE:	return State_Append_Update(event);	break;
		case SXCMD_STATE_DATA:			return State_Data(event);			break;

		default:
			return HelperStates(event);
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_Null
	(
		int			event
	)
//
//	Parse the null state
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_Tag_State	( "Database", SXCMD_STATE_DATABASE );

			// Unknown tag...

			SkipTag();

			break;

		case SXML_EVENT_END_TAG:

			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_Database
	(
		int			event
	)
//
//	Parse the <Database> element
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_Tag_State_Action	( "Get",	SXCMD_STATE_GET,			AddCmd( SXCMD_GET ); );
			SXML_Tag_State_Action	( "Count",	SXCMD_STATE_XPATH,			AddCmd( SXCMD_COUNT ); );
			SXML_Tag_State_Action	( "Delete",	SXCMD_STATE_XPATH,			AddCmd( SXCMD_DELETE ); );
			SXML_Tag_State_Action	( "Append",	SXCMD_STATE_APPEND_UPDATE,	AddCmd( SXCMD_APPEND ); );
			SXML_Tag_State_Action	( "Update",	SXCMD_STATE_APPEND_UPDATE,	AddCmd( SXCMD_UPDATE ); );

			// Unknown tag...

			SkipTag();

			break;

		case SXML_EVENT_ATTRIBUTE:

			if (strcmp(GetName(),"Name")==0)
			{
				// Save the data base name

				strncpy( databaseName, GetData(), SXCMD_MAX_DATABASE_NAME-1 );
			}
			
			break;

		case SXML_EVENT_END_TAG:

			state = SXCMD_STATE_NULL;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_Get
	(
		int			event
	)
//
//	Parse the <Get> element
//
//------------------------------------------------------------------------------//
{
	if (pCmd == NULL)
		return SXML_ERROR_OUT_OF_MEMORY;

	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_String			( "Select",		&pCmd->rootPath,	SXCMD_MAX_ROOT_PATH );
			SXML_String_Alloc	( "Nodes",		&pCmd->pNodes,		0 );
			SXML_String_Alloc	( "SubNodes",	&pCmd->pSubNodes,	0 );
			SXML_Int			( "Depth",		&pCmd->depth );

			// Unknown tag...

			SkipTag();

			break;

		case SXML_EVENT_END_TAG:

			state = SXCMD_STATE_DATABASE;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_XPath
	(
		int			event
	)
//
//	Parse the <Count> element
//
//------------------------------------------------------------------------------//
{
	if (pCmd == NULL)
		return SXML_ERROR_OUT_OF_MEMORY;

	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_String			( "Select",		&pCmd->rootPath,	SXCMD_MAX_ROOT_PATH );

			// Unknown tag...

			SkipTag();

			break;

		case SXML_EVENT_END_TAG:

			state = SXCMD_STATE_DATABASE;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_Append_Update
	(
		int			event
	)
//
//	Parse the <Append> or <Update> element
//
//------------------------------------------------------------------------------//
{
	if (pCmd == NULL)
		return SXML_ERROR_OUT_OF_MEMORY;

	switch (event)
	{
		case SXML_EVENT_START_TAG:

			SXML_String				( "Select",		&pCmd->rootPath,	SXCMD_MAX_ROOT_PATH );
			SXML_Tag_State_Action	( "Data",		SXCMD_STATE_DATA,	pCmd->pDB = new CSXDB; pNode = NULL; );

			// Unknown tag...

			SkipTag();

			break;

		case SXML_EVENT_END_TAG:

			state = SXCMD_STATE_DATABASE;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::State_Data
	(
		int			event
	)
//
//	Parse the <Append> or <Update> element
//
//------------------------------------------------------------------------------//
{
	CSXDB_Element *	result;

	if (pCmd->pDB == NULL)
		return SXML_ERROR_OUT_OF_MEMORY;

	if (pNode == NULL)
		pNode = (CSXDB_Element *) pCmd->pDB->Root();

	// Write tags out to file

	switch (event)
	{
		case SXML_EVENT_START_TAG:
			pNode = pNode->AddElement( GetName() );
			break;

		case SXML_EVENT_END_TAG:
			pNode = (CSXDB_Element *) pNode->Parent();
			if (pNode == NULL)
			{
				// We are back to the top...goto previous state

				state = SXCMD_STATE_APPEND_UPDATE;
			}
			break;

		case SXML_EVENT_ATTRIBUTE:
			result = (CSXDB_Element *) pNode->AddAttribute( GetName(), GetData() );
			if (result == NULL)
				return SXML_ERROR_OUT_OF_MEMORY;
			break;

		case SXML_EVENT_DATA:
			result = (CSXDB_Element *) pNode->AddData( GetData() );
			if (result != 0)
				return SXML_ERROR_OUT_OF_MEMORY;
			break;

		case SXML_EVENT_CDATA:
			break;

		default:
			break;
	}


	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXCMD_List::AddCmd
	(
		SXCMD_ID		id
	)
//
//	Adds a command to the command list and sets pCmd to that command
//
//------------------------------------------------------------------------------//
{
	CSXCMD	*p;
	CSXCMD	*pNew;

	pCmd = NULL;

	// Allocate a new cmd

	pNew = new CSXCMD;

	if (pNew == NULL)
		return SXML_ERROR_OUT_OF_MEMORY;

	// Set the cmd ID

	pNew->cmdID = id;

	// Insert it in the list

	if (pCmdList == NULL)
		pCmdList = pNew;
	else
	{
		p = pCmdList;

		while (p->pNext != NULL)
			p = p->pNext;

		p->pNext = pNew;
	}

	pCmd = pNew;

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// SXML_ERROR_
	SXDB_Cmd
	(
		CSIO	*	pIn,				// Input commands
		CSIO	*	pOut,				// Output results
		CSXDB	*	pDB					// The data base
	)
//
//	Process a <Database> block and returns the results
//
//	NOTE: If the return value is SXML_ERROR_OUT_OF_MEMORY, then the output in pOut
//	is in an undetermined state and should not be used.
//
//------------------------------------------------------------------------------//
{
	int			result;
	CSXCMD_List	cmdList;
	CSXCMD		*pCmd;

	// Start the output

	pOut->Printf("<Database>\n");

	// Parse the commands

	result = cmdList.ParseSIO( pIn );

	if (result != 0)
		SXDB_Cmd_Error( pOut, result );
	else
	{
		// Process the commands

		pCmd = cmdList.pCmdList;
		result = 0;

		while (pCmd && result != SXML_ERROR_OUT_OF_MEMORY)
		{
			switch (pCmd->cmdID)
			{
				case SXCMD_GET:
					{
						CSIO_DMEM	tempSio;

						pOut->Printf("<Get>\n");

						result = pDB->Get( pCmd->rootPath, pCmd->pNodes, pCmd->pSubNodes, pCmd->depth, &tempSio );

						if (result != 0)
							SXDB_Cmd_Error( pOut, result );
						else
						{
							pOut->Printf("<Data>\n");
							tempSio.CopyTo( pOut );
							pOut->Printf("</Data>\n");
						}

						pOut->Printf("</Get>\n");
					}
					break;

				case SXCMD_COUNT:
					pOut->Printf("<Count>");

					result = pDB->Count( pCmd->rootPath );

					if (result < 0)
						SXDB_Cmd_Error( pOut, result );
					else
						pOut->Printf("%d",result);

					pOut->Printf("</Count>\n");

					break;

				case SXCMD_APPEND:
					pOut->Printf("<Append>\n");

					result = pDB->Update( pCmd->rootPath, pCmd->pDB, 0 );

					if (result < 0)
						SXDB_Cmd_Error( pOut, result );
					else
						pCmd->pDB = NULL;

					pOut->Printf("</Append>\n");

					break;

				case SXCMD_UPDATE:
					pOut->Printf("<Update>\n");

					result = pDB->Update( pCmd->rootPath, pCmd->pDB, 1 );

					if (result < 0)
						SXDB_Cmd_Error( pOut, result );
					else
						pCmd->pDB = NULL;

					pOut->Printf("</Update>\n");

					break;

				case SXCMD_DELETE:
					pOut->Printf("<Delete>\n");

					result = pDB->Delete( pCmd->rootPath );

					if (result < 0)
						SXDB_Cmd_Error( pOut, result );

					pOut->Printf("</Delete>\n");

					break;

				case SXCMD_LOCK:
				case SXCMD_UNLOCK:
				case SXCMD_UNKNOWN:
					break;
			}

			pCmd = pCmd->pNext;
		}
	}

	result = pOut->Printf("</Database>\n") < 10 ? SXML_ERROR_OUT_OF_MEMORY : 0;

	return result;
}

//--------------------------------------------------------------------------------
//
	void
	SXDB_Cmd_Error
	(
		CSIO	*	pOut,				// Output results
		int			error				// error code
	)
//
//	
//
//------------------------------------------------------------------------------//
{
	pOut->Printf("<Error>%d</Error>",error);
}
