/*-------------------------------------------------------------------------------

	SXDB_Parse_Table.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Table driven parsing of XML database data

	These functions are used to take data from a database and put it in to a 
	structure or to take it from a structure and put it inot the database.

	You create a table that describes the structure and which elements and 
	atributes the fields ofthe structures map to.

	Example:

	Database contains:

	<Test id="xxx">

		This is root data

		<Name> My Name </Name>
		<Age> 25 </Age>
		<Address>
			<Street> 123 main street </Street>
			<Zip> 22033 </Zip>
		</Address>
		<Pet> Dog </Pet>
	</Test>

	// In your code you define a structure
	
	typedef struct
	{
		char	id[4];
		char	name[40];
		int		age;
		char	pet[10];
		char	street[60];
		int		zip;
		char	data[20];
	} PERSON;

	// Then you define the parsing table using the macros

	#define		PT_STRUCT	PERSON
	PT_BEGIN	("Test",	Test_Table,	PT_NO_UNKNOWN)
	PT_ROOT_DATA(			data,	20,	PT_STRING | PT_OPTIONAL)
	PT_ELEM		("Name",	name,	40,	PT_STRING | PT_REQUIRED)
	PT_ATT		("id",		id		4,	PT_STRING | PT_REQUIRED)
	PT_ELEM		("Age",		age,	0,	PT_INT    | PT_REQUIRED)
	PT_MOVE_DOWN("Address",				PT_STRING | PT_REQUIRED)
	PT_ELEM		("Street"	street,	60,	PT_STRING | PT_REQUIRED)
	PT_ELEM		("Zip",		zip,	0,	PT_INT    | PT_REQUIRED)
	PT_MODE_UP	()
	PT_ELEM		("Pet",		pet,	10,	PT_STRING | PT_OPTIONAL | PT_CONDITION1)
	PT_END
	#undef		PT_STRUCT


	// And now in your code:

	void Parse_Person( CSXDB * pDB )
	{
		PERSON	p;
		int		result;

		// To read the data, call

		result = SXDB_PT_Get( pDB, "//Device[1]", Test_Table, &p );

		// To write the data to the database

		result = SXDB_PT_Put( pDB, "//Device[1]", Test_Table, &p );
	}

	After calling SXDB_PT_Get(), the data struture will contain:

		char	id[4];		= "xxx"
		char	name[40];	= "My Name"
		int		age;		= 25
		char	pet[10];	= "Dog"
		char	street[60];	= "123 main street"
		int		zip;		= 22033
		char	data[20];	= "This is root data"


	Rules:

	There is no order dependency, except sub element (like Street and Zip) must
	be between and proper PT_MOVE_DOWN and PT_MOVE_UP.

	Items with the PT_CONDITIONx flag will only be read if the conditionx parameter
	is true. (See function definitions below.

	PT_REQUIRED means the field must be there on a read or an error is returned.

	PT_OPTIONAL means the field is only read if it exist and no error is generated
	if it does not exist.

	The PT_BEGIN macro contains either the PT_UNKNOWN_OK or the PT_NO_UNKNOWN
	flags. If the PT_BEGIN macro contains the PT_NO_UNKNOWN, then any element
	or attribute found in the database that is not contained in the table
	will cause an error PT_UNKNOWN_DATA to be returned.

	Macros:

	All tables start with #define PT_STRUCT your_struct_name

	PT_BEGIN			// Must be first
		(
			name,		// Name of the root element
			table_name,	// Name of this table
			flags		// Either PT_UNKNOWN_OK or PT_NO_UNKNOWN
		)

	PT_ATT				// Parse Attribute
		(
			name,		// Name of the attribute
			field,		// Name of the field in the data structure
			length,		// Length of char fields, including null terminator, unused for PT_STRING_PTR and numbers
			flags		// PT_STRING or PT_INT  (One of these must be present)
						// PT_REQUIRED or PT_OPTIONAL (Default is optional)
						// PT_CONDITIONx - Field is only processed if conditionx
						//   parameter of SXDB_PT_Get and SXDB_PT_Put is true.
		)

	PT_ELEM				// Parse Element
		(
			name,		// Name of the attribute
			field,		// Name of the field in the data structure
			length,		// Length of char fields, including null terminator, unused for PT_STRING_PTR and numbers
			flags		// PT_STRING or PT_INT  (One of these must be present)
						// PT_REQUIRED or PT_OPTIONAL (Default is optional)
						// PT_CONDITIONx - Field is only processed if conditionx
						//   parameter of SXDB_PT_Get and SXDB_PT_Put is true.
		)

	PT_ROOT_DATA		// Parse data from the root element (Test in the example)
		(
			field,		// Name of the field in the data structure
			length,		// Length of char fields, including null terminator, unused for PT_STRING_PTR and numbers
			flags		// PT_STRING or PT_INT  (One of these must be present)
						// PT_REQUIRED or PT_OPTIONAL (Default is optional)
						// PT_CONDITIONx - Field is only processed if conditionx
						//   parameter of SXDB_PT_Get and SXDB_PT_Put is true.
		)
		
	PT_NODE				// Get Node Address - The data type is CSXDB_Node
						// If element is found, the node address is put in the struct.
						// If element is not found, NULL is put in the struct
						// Has no effect on Put operations
		(
			name,		// Name of the attribute
			field,		// Name of the field in the data structure
			flags		// PT_STRING or PT_INT  (One of these must be present)
						// PT_REQUIRED or PT_OPTIONAL (Default is optional)
						// PT_CONDITIONx - Field is only processed if conditionx
						//   parameter of SXDB_PT_Get and SXDB_PT_Put is true.
		)

	PT_MOVE_DOWN		// Navigates down into an element
		(
			name,		// Name of the element to move down into
			flags		// PT_REQUIRED or PT_OPTIONAL
		)
		
	PT_MOVE_UP()		// Navigates up out of an element

	PT_END()			// Must be last
	
	// FIXME: there are more macros than the ones documented here ...

	All tables must #undef PT_STRUCT

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<pp/SXDB_Parse_Table.h>
#include	<pp/CmdLineParse.h>

//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

	int									// 0 or error code
	SXDB_PT_Get_Recursive
	(
		CSXDB_Node	*pRootNode,			// Ptr to the database node
		SXDB_PT	*pTable,				// Ptr to the table
		void	*pStruct,				// Ptr to the structure
		int		conditionFlags,			// condition (Optional, Default is all true)
		int		flags,					// PT_UNKNOWN_OK | PT_NO_UNKNOWN
		int		index					// Starting point for this level in the table
	);

	int									// index of name or -1
	PT_Find_Name
	(
		const char	*pName,					// Name you are looking for (NULL for data node)
		SXDB_PT	*pTable,				// The table to search
		int		startIndex				// index of the item to start searching from
	);

	void
	PT_GetData
	(
		CSXDB_Node	*pNode,				// The node to get from
		SXDB_PT		*pTable,			// Table entry
		void		*pStruct			// The structure base address
	);

	int									// 0 or error code
	SXDB_PT_Put_Recursive
	(
		CSXDB_Node	*pRootNode,			// Ptr to the database node
		SXDB_PT	*pTable,				// Ptr to the table
		void	*pStruct,				// Ptr to the structure
		int		conditionFlags,			// condition (Optional, Default is all true)
		int		flags,					// PT_UNKNOWN_OK | PT_NO_UNKNOWN
		int		index					// Starting point for this level in the table
	);

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Get
	(
		CSXDB		*pDB,				// Ptr to the database
		const char	*pXPath,			// XPath of element to parse
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int		conditionFlags			// condition (Optional, Default is all true)
								// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	)
//
//	Gets data from the database.
//
//------------------------------------------------------------------------------//
{
	CSXPath		p;
	int			result;

	// Check Parameters

	if (pDB == NULL || pXPath == NULL || pTable == NULL /*|| pStruct == NULL*/)
		return -1;

	// Parse xpath

	result = p.Parse( pXPath, pDB );

	if (result < 1)
		return PT_ROOT_NOT_FOUND;

	return SXDB_PT_Get( p.Enum(0), pTable, pStruct, conditionFlags );
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Get
	(
		CSXDB_Node	*pNode,				// Node of the Element to parse
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int			conditionFlags		// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	)
//
//	Gets data from the database.
//
//------------------------------------------------------------------------------//
{
	int	flags;
	int	result;

	// Check Parameters

	if (pNode == NULL || pTable == NULL /*|| pStruct == NULL*/)
		return -1;

	// Get the general flags from PT_BEGIN table item

	if ((pTable[0].flags & PT_TYPE_MASK) != PT_BEGIN_T)
		return PT_BAD_TABLE;

	flags = pTable[0].flags;

	// Make sure the root node matchs the name of the PT_BEGIN item

	if (pNode->GetType() != SXDB_TYPE_ELEMENT)
		return PT_ROOT_NOT_FOUND;

	if ( pTable[0].pName != NULL)
	{
		if (strcmp(pTable[0].pName, pNode->GetName()) != 0)
			return PT_ROOT_NOT_FOUND;
	}

	// parse sub elements

	result = SXDB_PT_Get_Recursive( pNode, pTable, pStruct, conditionFlags, flags, 1 );

	return result < 0 ? result : 0;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Put
	(
		CSXDB		*pDB,				// Ptr to the database
		const char	*pXPath,			// XPath of parent or node element (See mode)
		int		mode,				// PT_NODE_UPDATE, PT_CHILD_UPDATE
										// PT_CHILD_UPDATE_IF_EXISTS, or PT_CHILD_APPEND
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int		conditionFlags		// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	)
//
//	Puts data into the database
//      strings in pStruct will be copied to the XML database
//      overwritten data in DB will be freed (automatically, SXML feature ...)
//      Elements and attributes with string pointer values of NULL are not
//      written. Integers and other types are always valid, use conditions
//      to not write these values.
//
//	mode :
//
//	PT_NODE_UPDATE				= pXPath points to the element that will be updated
//	PT_CHILD_UPDATE				= pXPath points to parent of the element to be updated
//								  The first occurance on the element will be updated
//								  If the element does not exist, it will be created
//	PT_CHILD_UPDATE_IF_EXISTS	= pXPath points to parent of the element to be updated
//								  If the element does not exist, PT_ROOT_NOT_FOUND
//								  will be returned
//	PT_CHILD_APPEND				= pXPath points to parent of the node. a new element
//								  will be appended to the parent
//
//------------------------------------------------------------------------------//
{
	CSXPath		p;
	int			result;

	// Check Parameters

	if (pDB == NULL || pXPath == NULL || pTable == NULL || pStruct == NULL)
		return -1;

	// Parse xpath

	result = p.Parse( pXPath, pDB );

	if (result < 1)
		return PT_ROOT_NOT_FOUND;

	return SXDB_PT_Put( p.Enum(0), mode, pTable, pStruct, conditionFlags );
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Put
	(
		CSXDB_Node	*pNode,				// Ptr to Parent or node element (See mode)
		int		mode,				// PT_NODE_UPDATE, PT_CHILD_UPDATE
										// PT_CHILD_UPDATE_IF_EXISTS, or PT_CHILD_APPEND
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int		conditionFlags		// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	)
//
//	Puts data into the database
//	See documentation of first SXDB_PT_Put() above for details.
//
//	mode :
//
//	PT_NODE_UPDATE				= pNode points to the element that will be updated
//	PT_CHILD_UPDATE				= pNode points to parent of the element to be updated
//								  The first occurance on the element will be updated
//								  If the element does not exist, it will be created
//	PT_CHILD_UPDATE_IF_EXISTS	= pNode points to parent of the element to be updated
//								  If the element does not exist, PT_ROOT_NOT_FOUND
//								  will be returned
//	PT_CHILD_APPEND				= pNode points to perent of the node a new element
//								  will be appended to the parrent
//
//------------------------------------------------------------------------------//
{
	int			flags;
	int			result;
	CSXDB_Node	*pChild;

	// Check Parameters

	if (pNode == NULL || pTable == NULL || pStruct == NULL)
		return -1;

	// Get the general flags from PT_BEGIN table item

	if ((pTable[0].flags & PT_TYPE_MASK) != PT_BEGIN_T)
		return PT_BAD_TABLE;

	flags = pTable[0].flags;

	// Make sure the root node matchs the name of the PT_BEGIN item

	switch (mode)
	{
		case PT_NODE_UPDATE:
			if (pNode->GetType() != SXDB_TYPE_ELEMENT)
				return PT_ROOT_NOT_FOUND;

			if ( pTable[0].pName != NULL)
			{
				if (strcmp(pTable[0].pName, pNode->GetName()) != 0)
					return PT_ROOT_NOT_FOUND;
			}
			break;

		case PT_CHILD_UPDATE:
			pChild = pNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[0].pName, NULL );
			if (pChild == NULL)
			{
				pChild = ((CSXDB_Element *) pNode)->AddElement( pTable[0].pName );
				if (pChild == NULL)
					return SXML_ERROR_OUT_OF_MEMORY;
			}
			pNode = pChild;
			break;

		case PT_CHILD_UPDATE_IF_EXISTS:
			pChild = pNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[0].pName, NULL );
			if (pChild == NULL)
				return SXML_ERROR_OUT_OF_MEMORY;
			pNode = pChild;
			break;

		case PT_CHILD_APPEND:
			pChild = ((CSXDB_Element *) pNode)->AddElement( pTable[0].pName );
			if (pChild == NULL)
				return SXML_ERROR_OUT_OF_MEMORY;
			pNode = pChild;
			break;

		default:
			return -1;
	}			

	// process sub elements

	result = SXDB_PT_Put_Recursive( pNode, pTable, pStruct, conditionFlags, flags, 1 );

	return result < 0 ? result : 0;
}

//--------------------------------------------------------------------------------
//								Private fucntions
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Get_Recursive
	(
		CSXDB_Node	*pRootNode,			// Ptr to the database node
		SXDB_PT	*pTable,				// Ptr to the table
		void	*pStruct,				// Ptr to the structure
		int		conditionFlags,			// condition (Optional, Default is all true)
		int		flags,					// PT_UNKNOWN_OK | PT_NO_UNKNOWN
		int		index					// Starting point for this level in the table
	)
//
//	Gets data from one level of the database
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node	*pNode;
	int			startIndex = index;
	int			endIndex = -1;
	int			*pInt;
	int			result;
	SXDB_PT		*pTempTable;
	void		*pTempStruct;
	int			count;

	// Process each item in the table for this level

	do
	{
		// Check condition flags

		if (pTable[index].flags & PT_CONDITION_MASK)
		{
			if ( (pTable[index].flags & conditionFlags) == 0)
			{
				// Conditions not met, skip this one
				index++;
				continue;
			}
		}

		// Process the instruction

		switch (pTable[index].flags & PT_TYPE_MASK)
		{
			case PT_BEGIN_T:
				break;

			case PT_END_T:
				endIndex = index;
				break;

			case PT_ATT_T:
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ATTRIBUTE, pTable[index].pName, NULL );
				if (pNode == NULL)
				{
					if (pTable[index].flags & PT_REQUIRED)
						return PT_MISSING_FIELD;
				}
				else
					PT_GetData( pNode, &pTable[index], pStruct );
				break;

			case PT_ELEM_T:
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );
				if (pNode == NULL)
				{
					if (pTable[index].flags & PT_REQUIRED)
						return PT_MISSING_FIELD;
				}
				else
					PT_GetData( pNode, &pTable[index], pStruct );
				break;

			case PT_ROOT_T:
				if (pRootNode->GetData() == NULL)
				{
					if (pTable[index].flags & PT_REQUIRED)
						return PT_MISSING_FIELD;
				}
				else
					PT_GetData( pRootNode, &pTable[index], pStruct );
				break;

			case PT_EMPTY_ELEM_T:
				pInt = (int *) (((int) pStruct) + pTable[index].offset);
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );
				*pInt = pNode == NULL ? 0 : 1;
				break;

			case PT_MOVE_DOWN_T:
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );
				if (pNode == NULL)
				{
					if (pTable[index].flags & PT_REQUIRED)
						return PT_MISSING_FIELD;
				}
				else
				{
					index = SXDB_PT_Get_Recursive( pNode, pTable, pStruct,conditionFlags, flags, index+1 );
					if (index < 0)
						return index;
					if ( (pTable[index].flags & PT_TYPE_MASK) == PT_END_T )
						endIndex = index;
				}
				break;

			case PT_MOVE_UP_T:
				endIndex = index;
				break;

			case PT_ARRAY2_T:	// Place holder for extended array info
				break;

			case PT_ARRAY_T:
			case PT_TABLE_T:

				pTempTable = pTable[index].pTable;
				pNode = NULL;
				count = 0;
				do
				{
					pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTempTable->pName, pNode );
					if (pNode == NULL)
					{
						if (count == 0 && (pTable[index].flags & PT_REQUIRED) )
							return PT_MISSING_FIELD;
					}
					else
					{
						if (count > pTable[index].length)
							return PT_TOO_MANY_ELEMENTS;
						pTempStruct = (void *) ((unsigned int) pStruct + pTable[index].offset + pTempTable->length * count);
						result = SXDB_PT_Get_Recursive( pNode, pTempTable, pTempStruct, conditionFlags, flags, 1 );
						if (result < 0)
							return result;
						count++;
					}
				} while (pNode != NULL);

				if ( (pTable[index].flags & PT_TYPE_MASK) == PT_ARRAY_T)
				{
					// Get the count variable location from the next (PT_ARRAY2_T) table entry
					* (int *) (((int) pStruct) + pTable[index+1].offset) = count;
				}

				break;

			default:
				return PT_BAD_TABLE;
				break;
		}

		index++;

	} while (endIndex < 0);

	// Make sure there were no unknown elements

	if (flags & PT_NO_UNKNOWN)
	{
		pNode = pRootNode->Child();

		while (pNode)
		{
			switch (pNode->GetType())
			{
				case SXDB_TYPE_ATTRIBUTE:
				case SXDB_TYPE_ELEMENT:
					// FIXME -> PT_NO_UNKNOWN does not work when PT_TABLE is used !!!
					if (PT_Find_Name( pNode->GetName(), pTable, startIndex ) < 0)
						return PT_UNKNOWN_DATA;
					break;

				case SXDB_TYPE_DATA:
					if (PT_Find_Name( NULL, pTable, startIndex ) < 0)
						return PT_UNKNOWN_DATA;
					break;

				case SXDB_TYPE_NODE:
				case SXDB_TYPE_DOCUMENT:
					break;
			}

			pNode = pNode->Next();
		}
	}

	return endIndex;
}

//--------------------------------------------------------------------------------
//
	int									// index of name or -1
	PT_Find_Name
	(
		const char	*pName,					// Name you are looking for (NULL for data node)
		SXDB_PT	*pTable,				// The table to search
		int		startIndex				// index of the item to start searching from
	)
//
//	Returns the index of the table entry with the same name.
//
//------------------------------------------------------------------------------//
{
	int		index = startIndex;

	while (1)
	{
		switch (pTable[index].flags & PT_TYPE_MASK)
		{
			case PT_ELEM_T:
			case PT_ATT_T:
				if (pName != NULL)
				{
					if (strcmp(pTable[index].pName, pName) == 0)
						return index;
				}
				break;

			case PT_ROOT_T:
				if (pName == NULL)
					return index;
				break;

			case PT_MOVE_DOWN_T:
				// See if the sub element matches

				if (pName != NULL)
				{
					if (strcmp(pTable[index].pName, pName) == 0)
						return index;
				}

				// Skip the sub-element
				do
				{
					if ((pTable[index].flags & PT_TYPE_MASK) == PT_END_T)
						return -1;
					if ((pTable[index].flags & PT_TYPE_MASK) == PT_MOVE_UP_T)
						break;
					index++;
				} while (1);
				break;

			case PT_END_T:
			case PT_MOVE_UP_T:
				return -1;
				break;

			default:
				break;
		}

		index++;
	}

	return -1;
}

//--------------------------------------------------------------------------------
//
	void
	PT_GetData
	(
		CSXDB_Node	*pNode,				// The node to get from
		SXDB_PT		*pTable,			// Table entry
		void		*pStruct			// The structure base address
	)
//
//	Gets data from the node and puts it into the structure.
//
//------------------------------------------------------------------------------//
{
	char		*pString;
	const char	**pStringPtr;
	int		*pInt;
	CSXDB_Node	**ppNode;

	// Check the input parameters

	if (pNode == NULL)
		return;

	// Get the data

	if (pTable->flags & PT_STRING)
	{
		// Get String data

		pString = (char *) (((int) pStruct) + pTable->offset);

		if (pNode == NULL || pNode->GetData() == NULL)
			*pString = 0;
		else
			strncpy( pString, pNode->GetData(), pTable->length-1 );
	}
	if (pTable->flags & PT_STRING_PTR)
	{
		// Get String Ptr data

		pStringPtr = (const char **) (((int) pStruct) + pTable->offset);

		*pStringPtr = pNode->GetData();

		if (*pStringPtr == NULL)
			*pStringPtr = "";
	}
	else if (pTable->flags & PT_INT)
	{
		// Get Int data

		pInt = (int *) (((int) pStruct) + pTable->offset);

		if (pNode == NULL || pNode->GetData() == NULL)
			*pInt = 0;
		else
			*pInt = atoi(pNode->GetData());
	}
	else if (pTable->flags & PT_ULONG)
	{
		// Get Int data

		char	*pStopString;

		pInt = (int *) (((int) pStruct) + pTable->offset);

		if (pNode == NULL || pNode->GetData() == NULL)
			*pInt = 0;
		else
			*pInt = strtoul(pNode->GetData(), &pStopString, 10);
	}
	else if (pTable->flags & PT_IP)
	{
		// Get Int data

		pInt = (int *) (((int) pStruct) + pTable->offset);

		if (pNode == NULL || pNode->GetData() == NULL)
			*pInt = 0;
		else
		{
			GetNextIPAddress( pNode->GetData(), pInt, NULL );
		}
	}
	if (pTable->flags & PT_NODE)
	{
		// Get Node Address

		ppNode = (CSXDB_Node **) (((int) pStruct) + pTable->offset);
		*ppNode = pNode;
	}
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code
	SXDB_PT_Put_Recursive
	(
		CSXDB_Node	*pRootNode,			// Ptr to the database node
		SXDB_PT	*pTable,				// Ptr to the table
		void	*pStruct,				// Ptr to the structure
		int		conditionFlags,			// condition (Optional, Default is all true)
		int		flags,					// PT_UNKNOWN_OK | PT_NO_UNKNOWN
		int		index					// Starting point for this level in the table
	)
//
//	Gets data from one level of the database
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node	*pNode;
	int			endIndex = -1;
	int			result;
	const char		*pString;
	int			*pInt;
	char		temp[20];
	SXDB_PT		*pTempTable;
	void		*pTempStruct;
	int			count;
	int			x;

	// Process each item in the table for this level

	do
	{
		// Check condition flags

		if (pTable[index].flags & PT_CONDITION_MASK)
		{
			if ( (pTable[index].flags & conditionFlags) == 0)
			{
				// Conditions not met, skip this one
				index++;
				continue;
			}
		}

		// Convert the data to a string

		if (pTable[index].flags & PT_STRING)
			pString = (char *) (((int) pStruct) + pTable[index].offset);
		else if (pTable[index].flags & PT_STRING_PTR)
		{
			pString = * (char **) (((int) pStruct) + pTable[index].offset);
			if (pString == NULL)
			{
				index++;
				continue;
			}
		}
		else if (pTable[index].flags & PT_INT)
		{
			pInt = (int *) (((int) pStruct) + pTable[index].offset);
			sprintf(temp,"%d",*pInt);
			pString = temp;
		}
		else if (pTable[index].flags & PT_ULONG)
		{
			pInt = (int *) (((int) pStruct) + pTable[index].offset);
			sprintf(temp,"%u",*pInt);
			pString = temp;
		}
		else if (pTable[index].flags & PT_IP)
		{
			pInt = (int *) (((int) pStruct) + pTable[index].offset);
			sprintf(temp,"%d.%d.%d.%d",
						(*pInt >> 24) & 0xFF,
						(*pInt >> 16) & 0xFF,
						(*pInt >> 8) & 0xFF,
						(*pInt) & 0xFF
					);
			pString = temp;
		}
		else
			pString = "";

		// Process the instruction

		switch (pTable[index].flags & PT_TYPE_MASK)
		{
			case PT_BEGIN_T:
				break;

			case PT_END_T:
				endIndex = index;
				break;

			case PT_ATT_T:
				pNode = ((CSXDB_Element *) pRootNode)->AddAttribute( pTable[index].pName, pString );
				if (pNode == NULL)
					return SXML_ERROR_OUT_OF_MEMORY;
				break;

			case PT_ELEM_T:
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );
				if (pNode == NULL)
				{
					pNode = ((CSXDB_Element *) pRootNode)->AddElementWithData( pTable[index].pName, pString );
					if (pNode == NULL)
						return SXML_ERROR_OUT_OF_MEMORY;
				}
				else
				{
					result = pNode->SetData( pString );
					if (result != 0)
						return result;
				}
				break;

			case PT_EMPTY_ELEM_T:
				pInt = (int *) (((int) pStruct) + pTable[index].offset);
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );

				if (*pInt != 0 && pNode == NULL) // should be there but is not...create
				{
					pNode = ((CSXDB_Element *) pRootNode)->AddElement( pTable[index].pName );
					if (pNode == NULL)
						return SXML_ERROR_OUT_OF_MEMORY;
				}
				else if (*pInt == 0 && pNode != NULL) // should not be there but is...delete
				{
					delete pNode;
				}
				break;

			case PT_ROOT_T:
				result = ((CSXDB_Element *) pRootNode)->SetData( pString );
				if (result != 0)
					return result;
				break;

			case PT_MOVE_DOWN_T:
				pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTable[index].pName, NULL );
				if (pNode == NULL)
				{
					pNode = ((CSXDB_Element *) pRootNode)->AddElement( pTable[index].pName );
					if (pNode == NULL)
						return SXML_ERROR_OUT_OF_MEMORY;
				}
				index = SXDB_PT_Put_Recursive( pNode, pTable, pStruct,conditionFlags, flags, index+1 );
				if (index < 0)
					return index;
				if ( (pTable[index].flags & PT_TYPE_MASK) == PT_END_T )
					endIndex = index;
				break;

			case PT_MOVE_UP_T:
				endIndex = index;
				break;

			case PT_ARRAY2_T:	// Place holder for extended array info
				break;

			case PT_ARRAY_T:
			case PT_TABLE_T:
	
				if ( (pTable[index].flags & PT_TYPE_MASK) == PT_ARRAY_T)
				{
					// Get the count variable location from the next (PT_ARRAY2_T) table entry
					count = * (int *) (((int) pStruct) + pTable[index+1].offset);

					// Himalaya CR 1506:
					// Delete all the child nodes with the matching name (if the update options says delete before update)
					// Let them be added as if they are being appended.
					if ( (pTable[index].flags & PT_UPDATE_TYPE_MASK) == PT_DEL_BEFORE_UPDATE)
					{
						CSXDB_Node* pTempNode1;
						SXDB_PT		*pTempTable1;

						for (;;) 
						{
							pTempTable1 = pTable[index].pTable;
							pTempNode1 = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, 
																			pTempTable1->pName, 
																			NULL);

							if (pTempNode1 == NULL)
								break;

							pTempNode1->Remove();
						}
					}

				}
				else
					count = 1;

				for (x=0;x<count;x++)
				{
					pTempTable = pTable[index].pTable;
					pNode = pRootNode->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pTempTable->pName, NULL, x+1 );
					if (pNode == NULL)
					{
						pNode = ((CSXDB_Element *) pRootNode)->AddElement( pTempTable->pName );
						if (pNode == NULL)
							return SXML_ERROR_OUT_OF_MEMORY;
					}
					pTempStruct = (void *) ((unsigned int) pStruct + pTable[index].offset + pTempTable->length * x );
					result = SXDB_PT_Put_Recursive( pNode, pTempTable, pTempStruct, conditionFlags, flags, 1 );
					if (result < 0)
						return result;
				}
				break;

			default:
				return PT_BAD_TABLE;
				break;
		}

		index++;

	} while (endIndex < 0);

	return endIndex;
}

//--------------------------------------------------------------------------------
//
	int									// 0 or error code: -1 of PT incorrect, -2 if element not found
	SXDB_PT_XPath
	(
		char		*pXPath,		// XPath (only elements and index predicates)
		SXDB_PT		*pTable,		// Ptr to the table
		void		*pStruct,		// Ptr to the structure
		SXDB_PT		**pOutTable,		// Returns the ptr to the table component
		void		**pOutStruct,		// Returns the ptr to the structure component
		int		*pIndex			// Returns the table index location for normalize
	)
/**
 * Uses an xpath to drill down in a parse table and strip unused parts. Requires a parse table
 * description and a target data reference. Removes unnecessary parts of the parse table description
 * that cannot be reached with this xpath.
 * Returns a new parsetable description and an updated target data reference (both being
 * references within the input values). In addition it returns the index of the chosen 
 * node within the returned parsetable if the node selected by the xpath is an
 * element or an attribute (pIndex is 0 if the result is a subbranch).
 * 
 * Note: this assumes that the xPath is well formated and that a matching subtable
 *       exists. Dont use user-supplied paths here.
 * Note: does not support parse tables with PT_MOVE_UP/PT_MOVE_DOWN nodes. Use tables
 *       with PT_TABLE subtables instead.
 * Note: will eventually alter pXPath in case of errors
 */
//	Uses an xpath to drill down into PT tables and computes the structure ptr
//	that represents the table position at the end of the xpath.
// 
//------------------------------------------------------------------------------//
{
	// Implementation notes:
	// pTable and pStruct will always point to working state
	// pName will always point to current xPath segment
	// *pIndex contains the index within the matching SXDB_PT (if you choose an attribute or element)
	
	char		*pName;		// Ptr to the name of the element in the path
	char		*pNext;		// Ptr to the char after the name
	int		index;		// The index predicate value
	int		x;
	int		isAtt;
	int		exit;
	char		saveChar;
	char		*pSaveBracket;
	SXDB_PT		*pTempTable;

	// Initialize to defaults
	pName		= pXPath;
	*pOutTable	= pTable;
	*pOutStruct	= pStruct;
	*pIndex		= 0;

	// Special case a null xpath
	if (pXPath[0] == 0)
		return 0;

	// Process path
	do
	{
		// Get the name and index of the next segment of the xpath
		if (*pName == '/')
			pName++;

		pNext = pName;
		index = 1;
		isAtt = 0;
		pSaveBracket = NULL;

		while (*pNext)
		{
			if (*pNext == '@')
			{
				isAtt = 1;
				continue;
			}
			// this advances over the whole index [x], leaving \0x] and index=x
			if (*pNext == '[')
			{
				pSaveBracket = pNext;
				*pNext = 0;
				pNext++;
				index = atoi(pNext);  // this works because atoi stops at the first non-digit character
				while ( *pNext != ']' && *pNext ) {
					pNext++;
				}
				if (*pNext == ']')
					pNext++;
				break; // exit loop
			}

			pNext++;
		}
		// this stops after the first index [x] or at end of string
		// isAtt==1 if we selected an attribute node /@attribute  (cannot happen together with an indexed node ?)

		saveChar = *pNext;
		*pNext = 0;
		
		// pName now points to the \0 terminated name of the next element, isAttr if it is an attribute, index=x if within array...

		// Search the current parsetable for the named element
		// search all elements as we assume that exactly one element will match
		for (x=0, exit=0; !exit; x++)
		{
			switch ( (pTable[x].flags & PT_TYPE_MASK))
			{
				case PT_MOVE_DOWN_T:
				case PT_MOVE_UP_T:
					// We do not support move up/move down in a table
					return -1;

				case PT_END_T:
					// Element not found
					return -2;
					break;

				case PT_ATT_T:
					if (isAtt)
					{
						if (strcmp(pName,pTable[x].pName)==0)
						{
							*pIndex = x;
							goto HAVE_TABLE;  // no further seach, equal to reaching end of string
							// note: continue should be equivalent here but don't change working code
						}
					}
					break;

				case PT_EMPTY_ELEM_T:
				case PT_ELEM_T:
					if (!isAtt)
					{
						if (strcmp(pName,pTable[x].pName)==0)
						{
							*pIndex = x;
							goto HAVE_TABLE;  // no further search, equal to reaching end of string
							// note: continue should be equivalent here but don't change working code
						}
					}
					break;

				case PT_ARRAY_T:
				case PT_TABLE_T:
					if (strcmp(pName, pTable[x].pTable->pName )==0 && !isAtt)
					{
						if (index < 1 || index > pTable[x].length)
							return -1; // user selected index is out of table range, should not happen at all ?
						// calculate new (temporary) result
						pTempTable = pTable[x].pTable;
						pStruct = (void *) ((unsigned long) pStruct + pTable[x].offset + pTempTable->length * (index-1));
						pTable = pTempTable;
						exit = 1;  // exit this loop to resolve next element from xpath
					}
					break;
			}  // switch(pTable[x].flags & PT_TYPE_MASK)
		} // for

		if (pSaveBracket != NULL)
			*pSaveBracket = '[';
		*pNext = saveChar;
		pName = pNext; // iterate to next element

	} while (saveChar);  // until end of string

HAVE_TABLE:
        // restore xpath
	*pNext = saveChar;
	if (pSaveBracket != NULL)
		*pSaveBracket = '[';
		
	// set result
	*pOutTable = pTable;
	*pOutStruct = pStruct;
	// *pIndex is already set in code ...
	
	return 0;
}

int strins( char * dest, const char * src )
{
	int	ld = strlen (dest);
	int	ls = strlen (src);
	int	x;

	for (x=ld;x>=0;x--)
		dest[x+ls] = dest[x];

	for (x=0;x<ls;x++)
		dest[x] = src[x];

	return ld+ls;
}

//--------------------------------------------------------------------------------
//
	char*						// Ptr to the new xPath or NULL if it fails
	CreateXPath
	(
		CSXDB_Node	*pParent,		// The parent node
		CSXDB_Node	*pChild,		// The child node
		int		insertIndexPredicate	// 0=No indexes, !0 = insert indexes
	)
//
//	Creates the xpath from pParent to pChild.
//	If the pParent is NULL, then the path goes from the DB root to the pChild.
//	If pParent and pChild do not connect, then an NULL is returned.
//	if index is !0 then index predicates will be inserted into the xpath.
//		For example, if there are 3 nodes the same as pChild, and pChild is
//		the 3rd one, then /Name[3]/ is inserted. If insertIndexPredicate is 0,
//		then the [3] predicate is left off.
//
//	The caller must delete the xPath when done
//
//--------------------------------------------------------------------------------
{
	int			length = 0;
	CSXDB_Node *pNode,*pNodeP;
	char		*p = new char[MAX_XPATH_LENGTH];
	char		str[20];
	int			x;
	int			i;
	int			first = 1;

	if (p == NULL)
		goto ERROR_EXIT;
	
	p[0] = 0;

	if (pChild == NULL)
		goto ERROR_EXIT;

	// Special case the document element

	if (pChild->GetType() == SXDB_TYPE_DOCUMENT)
	{
		strcpy(p,"/");
		if (pParent != NULL)
			if (pParent != pChild)
				goto ERROR_EXIT;
		return p;
	}

	// If the node is data, then we skip up to the parent

	if (pChild->GetType() == SXDB_TYPE_DATA)
	{
		pChild = pChild->Parent();
		if (pChild == NULL)
			goto ERROR_EXIT;
		if (pChild == pParent)
			return p;
	}

	// Now add all the parent elements

	do
	{
		// If we are at the parent, then we are done

		if (pChild == pParent)
			return p;

		// If we are at the document, then we insert the final / and we are done

		if (pChild->GetType() == SXDB_TYPE_DOCUMENT)
		{
			strins(p,"/");
			return p;
		}

		// Sanity check the data

		if (pChild->Parent() == NULL)
			goto ERROR_EXIT;

		switch (pChild->GetType())
		{
			case SXDB_TYPE_ELEMENT:
			case SXDB_TYPE_ATTRIBUTE:
				break;
			default:
				goto ERROR_EXIT;
		}

		// Insert the next element into the path

		x = strlen( pChild->GetName() );

		if (length+7 > MAX_XPATH_LENGTH)
		{
			// Path is too long
			goto ERROR_EXIT;
		}

		// Put in the / seperator

		if (!first)
			strins(p,"/");

		first = 0;

		// Insert the predicate

		if (insertIndexPredicate && pChild->GetType() == SXDB_TYPE_ELEMENT)
		{
			pNodeP = pChild->Parent();
			pNode = NULL;
			i = 1;

			while ( (pNode = pNodeP->FindChildOfTypeByName( SXDB_TYPE_ELEMENT, pChild->GetName(), pNode )) != NULL )
			{
				if (pNode == pChild)
				{
					sprintf(str,"[%d]",i);
					strins(p,str);
					break;
				}
			}
		}
		
		// Insert the name

		strins(p,pChild->GetName());

		// Insert the attribute symbol if needed

		if (pChild->GetType() == SXDB_TYPE_ATTRIBUTE)
			strins(p,"@");

		// Move up to the next level

		pChild = pChild->Parent();
	} while (1);

ERROR_EXIT:
	if (p != NULL)
		delete p;

	return NULL;
}

//--------------------------------------------------------------------------------
//
	CSXDB_Node*					// The normalized node ptr or NULL if error
	SXDB_PT_Normalize
	(
		CSXDB_Node	*pNode,			// The input node that will be expanded to fit pTable
		SXDB_PT		*pTable,		// Ptr to the parsetable, the returned node will fit this parsetable
		int		index,			// Index within pTable that describes the content of pNode
							//   -1 = no normalization at all, return pNode immediately
							//    0 = pNode already matches pTable, normalize root node only
							//   >1 = pNode is a subnode of pTable at the specified index
		int		create,			// configure normalization behaviour:
							//    0 = move up nodes only (nodes should exist)
							//    1 = create the needed nodes
		int		include_root		// configure normalization output:
							//    0 = do not include table root node, table data only
							//    1 = include table root (as specified in PT_BEGIN)
	)
//
//	Adds Nodes above pNode to make it match the table hierarchy.
//
/**
 * Alters the passed pNode to fit the specified parsetable by either adding nodes
 * or browsing up the existing node hierarchy. A node for an element or attribute
 * will be created if index is greater than 0. The parse tables root node (as in
 * PT_BEGIN) will be created if root is 1.
 * 
 * If nodes are created, the nodes will be inserted above pNode and its parent.
 * pNodes siblings will be moved to the new element. Thus pNode must have a
 * valid parent node. The newly created node will become a part of pNodes data
 * and it will be freed when pNode is freed. (Reads: don't bother about memory)
 * 
 * If nodes are not created, the function will simply browse up the hierarchy.
 * 
 * This function assumes that normalization works, especially when browsing up
 * the hierarchy instead of creating new nodes.
 */
//------------------------------------------------------------------------------//
{
	CSXDB_Node		*pNew;
	CSXDB_Node		*pParent;

	// If -1, do nothing
	if (index < 0)
		return pNode;

	// See if we pData is for a subnode that has to be created/found:
	if (index > 0)
	{
		if (create)
		{
			// create required object for subnode
			switch (pTable[index].flags & PT_TYPE_MASK)
			{
				case PT_ELEM_T:
				case PT_EMPTY_ELEM_T:
					pNew = (CSXDB_Node *) new CSXDB_Element;
					pNew->SetName( pTable[index].pName );
					break;
				
				case PT_ATT_T:
					pNew = (CSXDB_Node *) new CSXDB_Attribute;
					pNew->SetName( pTable[index].pName );
					break;
				
				case PT_ARRAY_T:
				case PT_TABLE_T:
					// this cannot happen if we use this to normalize SXDB_PT_XPath results
					pNew = (CSXDB_Node *) new CSXDB_Element;
					pNew->SetName( pTable[index].pTable->pName );
					break;
	
				default:
					// Bad type to normalize too
					return NULL;
			}
			
			// inserts the new node between pNode and its parent, move all of pNodes siblings as well.
			pParent = pNode->Parent();
			while ( (pNode = pParent->Child()) != NULL)
			{
				pNode->Remove();
				pNew->AddChild( pNode );
			}
			pParent->AddChild( pNew );
			pNode = pNew;   // move up pNode
		}
		else  // if (create)
		{
			// check target type
			switch (pTable[index].flags & PT_TYPE_MASK)
			{
				// FIXME: is this switch really correct ??
				case PT_ELEM_T:
				case PT_EMPTY_ELEM_T:
				case PT_ATT_T:
					// dont move up for elements an attributes as for them pNode usually points
					// to the correct node (and not the corresponding data node)
					break;
				case PT_ARRAY_T:
				case PT_TABLE_T:
					// move up for tables as pNode in this case references the child of the referenced entry
					// FIXME: right now we assume this, can we enforce/check it ? pNode could point to a data node
					// this cannot happen if we use this to normalize SXDB_PT_XPath results
					pNode = pNode->Parent();
					if (pNode == NULL) return NULL;  // parent must exist for following functions
					break;
	
				default:
					// Bad type to normalize too
					return NULL;
			}
		}
	}

	// Create the root node of the parsetable itself if requested
	if (include_root)
	{
		if (create)
		{
			// create the new node
			pNew = (CSXDB_Node *) new CSXDB_Element;
			pNew->SetName( pTable[0].pName );
			// inserts the new node between pNode and its parent, move all of pNodes siblings as well.
			pParent = pNode->Parent();
			while ( (pNode = pParent->Child()) != NULL)
			{
				pNode->Remove();
				pNew->AddChild( pNode );
			}
			pParent->AddChild( pNew );
			pNode = pNew;
		}
		else
		{
			pNode = pNode->Parent();
			//if (pNode == NULL) return NULL;
		}		
	}

	return pNode;
}

