/*--------------------------------------------------------------------------------

	SXML.cpp

	Copyright (c) 2002, Raritan Computer, Inc.

	This is a VERY simple event based parser for XML data.

	The paradigm is simular to the SAX xml parser, but MUCH simpler. Read more about
	SAX here http://www.saxproject.org/


	To understand how an event-based API can work, consider the following 
	sample document:

	<?xml version="1.0"?>
	<doc>
	<para>Hello, world!</para>
	</doc>

	An event-based interface will break the structure of this document down into
	a series of linear events, such as these:

	start document
	start element: doc
	start element: para
	characters: Hello, world!
	end element: para
	end element: doc
	end document

	An application handles these events just as it would handle events from a
	graphical user interface: there is no need to cache the entire document in
	memory or secondary storage.

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	"pp/SXML.h"

//----------------------------------------
//				Equates
//----------------------------------------

#define	MAX_ENTITY_NAME		10

#define	isNameFirstChar(a) ( isalpha(a) || a == '_' || a ==':' )
#define	isNameChar(a) ( isalpha(a) || a == '_' || a ==':' || a == '.' || a == '-' || isdigit(a) )

#define	UNUSED(a)					// For unused parameters

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

static SXML_TOKEN entityTokenList[] =
{
	{	"lt",		'<'	},
	{	"gt",		'>'	},
	{	"amp",		'&'	},
	{	"apos",		39	},
	{	"quot",		34	},
	{	NULL,		0	}
};

#ifdef	SXML_TEST
int	verbose = 0;
#endif

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//						Constructor/Deconstructor Methods
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSXML::CSXML
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	SetBufferLimit(SXML_BUFFER_ID_ALL,0);
	SetOption(SXML_OPTION_ID_ALL_DEFAULT,0);
	
	skipTag				= 0;
	skipData			= 0;
	state				= 0;

#ifdef SXML_TEST
	tagStarted			= 0;
	fpOut				= stdout;
#endif
}

//--------------------------------------------------------------------------------
//
	CSXML::~CSXML
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::Parse
	(
		const char	*	pXMLData,			// Ptr to XML data or NULL to use Read()
		int			length				// Length of the XML data, if pXMLData != NULL
	)
//
//	The Parse() method parses the entire XML document. The document can be in 
//	memory or in a file accessed via the Read() virtual method. If the XML data 
//	is in memory, <pXMLData> points to the data and <length> is the length of the 
//	data in bytes. If the XML data is in a file or comes from another source, 
//	<pXMLData> should be set to NULL. In this case, Paser() will call the virtual 
//	method Read() to access the data.
//	
//	As the XML data is parsed, the Parse() method sends event codes to the virtual
//	method Event(). See Event() below for more details.
//	
//	SXML_NO_ERROR (0) is returned if the file was parsed successfully. If a syntax 
//	error was found, one of the SXML_ERROR_ error numbers will be returned. These 
//	are all negitive numbers. 
//	
//	NOTE: Changes to buffer sizes and options should be made by calling 
//	SetBufferLimit() and SetOption() before calling parse.
//	
//	NOTE: A NULL character (0) in the data is interpreted as the end of file.
//
//------------------------------------------------------------------------------//
{
	int				result;

	// Allocate the buffers

	result = AllocBuffers( pXMLData, length );

	if (result != 0)
		goto EXIT;

	// Parse the prolog

	result = ParseProlog();

	if (result != 0)
		goto EXIT;

	// Parse the Document

	result = ParseDocument();

	if (result != 0)
		goto EXIT;

	// Parse the Epilog

//	result = ParseEpilog();

//	if (result != 0)
//		goto EXIT;

	// Clean up and return
EXIT:

	FreeBuffers();

	return result;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::Event
	(
		int			UNUSED(event)
	)
//
//	The virtual method Event() is called whenever an element has been successfully
//	parsed from the XML data. "event" indicates the parsing event code. The event 
//	codes are described below.
//	If Event() returns a non-zero value, parsing will stop and the value will be 
//	returned from the Parse() call. By convention, the overridden Event() function 
//	should return positive numbers to indicate errors generated at the application 
//	level.
//	
//	SXML_EVENT_START_DOCUMENT
//	This event is sent at the start of an XML document. There is no data associated
//	with this event.
//
//	SXML_EVENT_END_DOCUMENT
//	This event is sent at the end of an XML document. There is no data associated
//	with this event.
//
//	SXML_EVENT_START_TAG
//	This event is sent whenever a start tag is found, such as "<tag_name> ". Event() 
//	can call the GetName() method to get the name of the new tag. In this example, 
//	GetName() would returned a pointer to the string "tag_name".
//	
//	Each SXML_EVENT_START_TAG represents a new nested level in the hierarchal 
//	structure of the XML document.
//	
//	SXML_EVENT_END_TAG
//	This event is sent whenever an end tag is found, such as "</tag_name>". Event() 
//	can call the GetName() method to get the name of the new tag. In this example, 
//	GetName() would returned a pointer to the string "tag_name".
//	XML also allows for a special syntax for empty tags, "<tag_name/>". In this 
//	case, this one tag string generates an SXML_EVENT_START_TAG immediately 
//	followed by an SXML_EVENT_END_TAG.
//	Each SXML_EVENT_END_TAG represents the end of a nested level in the hierarchal 
//	structure of the XML document.
//	
//	SXML_EVENT_DATA
//	This event is sent whenever parsed character data is found. The Event() method 
//	can call GetData() to get a pointer to the data. GetDataLenght() returns the 
//	length of the data.
//	Multiple SXML_EVENT_DATA events can be sent for a single tag for two reasons. 
//	If the data is larger than the data buffer limit value, then the data will be 
//	broken into segments and sent to Event() in successive calls. See SetBufferLimit() 
//	for details about data buffer limit.
//	The second reason is if a sub tag or other XML element interrupts the data.
//	Consider:
//	
//	<tag>
//		start of data
//		<sub-tag>
//		</sub-tag>
//		rest of data
//	</tag>
//	
//	In this example, two SXML_EVENT_DATA events will be sent for <tag>. The first 
//	will represent "start of data" and the second will represent "rest of data"
//	Parse() can optionally skip leading and trailing white space around data. See 
//	SetOption() method for details.
//	SXML_EVENT_ATTRIBUTE
//	This event is sent for each attribute found in a start tag. The Event() method 
//	can call the GetName() method to get the name of the attribute, GetData() to get 
//	the data of the attribute and GetDataLength() to get the length of the attribute 
//	data. Example:
//	<start_tag color="red" size = "small">
//	
//	This tag generates two SXML_EVENT_ATTRIBUTE events, one for the "color" attribute 
//	and one for the "size" attributes. For the first attribute, the call to GetName() 
//	returns a pointer to the string "color" and the call to GetData() returns a 
//	pointer to the string "red".
//	If the attribute data is larger than the data buffer limit, then the remaining 
//	attribute data will be passed to Event() with subsequent 
//	SXML_EVENT_ATTRIBUTE_CONTINUED events. See SetBufferLimit() for more details.
//	
//	SXML_EVENT_ATTRIBUTE_CONTINUED
//	See SXML_EVENT_ATTRIBUTE.
//	
//	SXML_EVENT_CDATA
//	This tag is sent whenever an XML CDATA section is encountered. The Event() 
//	method can call GetData() to get a pointer to the CDATA data. GetDataLenght() 
//	returns the length of the data.
//	If the CDATA section data is larger than the data buffer limit, then the remaining 
//	CDATA data will be passed to Event() with subsequent SXML_EVENT_CDATA events.
//
//	Other functions useful in processing events are:
//
//	GetCurrentTagName()		Returns a ptr to the current open tag
//	GetRootTagName()		Returns a ptr to the top level open tag name
//	GetNextTagName()		Returns a ptr to the name of the tag on the next level down
//	GetPreviousTagName()	Returns a ptr to the name of the tag on the next level up
//
//------------------------------------------------------------------------------//
{
#ifdef SXML_TEST
	int		x;
	int		indent = 1;
	char	quote = 0x22;

	if (verbose)
	{
		if (event == SXML_EVENT_START_DOCUMENT)
		{
			fprintf(fpOut,"<?xml verision=%c1.0%c standalone=%cyes/%c encoding=%cUTF-8%c?>",quote,quote,quote,quote,quote,quote);
			tagStarted = 0;
			return 0;
		}

		// Write tags out to file

		if (tagStarted && event != SXML_EVENT_ATTRIBUTE && event != SXML_EVENT_ATTRIBUTE_CONTINUED)
		{
			tagStarted = 0;
			fprintf(fpOut,">\n");
		}
		else if (tagStarted)
			indent = 0;

		if (indent)
		{
			for (x=0;x<GetLevel()-1;x++)
				fprintf(fpOut,"  ");
		}

		switch (event)
		{
			case SXML_EVENT_START_TAG:
				fprintf(fpOut,"<%s",GetName());
				tagStarted = 1;
				break;

			case SXML_EVENT_END_TAG:
				fprintf(fpOut,"</%s>\n",GetName());
				break;

			case SXML_EVENT_ATTRIBUTE:
			case SXML_EVENT_ATTRIBUTE_CONTINUED:
				fprintf(fpOut," %s = %c%s%c",GetName(),quote,GetData(),quote);
				break;

			case SXML_EVENT_DATA:
				fprintf(fpOut,"  %s\n",GetData());
				break;

			case SXML_EVENT_CDATA:
				fprintf(fpOut,"  <![CDATA[%s]]>",GetData());
				break;

			default:
				break;
		}

	}
#endif

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// # of bytes read or 0 for EOF
	CSXML::Read
	(
		char	*	UNUSED(pData),		// Ptr to the buffer
		int			UNUSED(count)		// # of bytes to read
	)
//
//	The virtual method Read() should be overridden if the XML data is to come from 
//	a file or other source. Parse() calls read each time it needs more data from 
//	the XML file. The "pBuffer" parameter points to where to place the data. The 
//	"count" parameter is the desired number of bytes to read. Read() returns the 
//	number of bytes actually read. A return value of 0 indicates an end of file 
//	result(EOF).
//	
//------------------------------------------------------------------------------//
{
	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// Old Limit
	CSXML::SetBufferLimit
	(
		int			bufferID,			// Which buffer limit to set (SMXL_BUFFER_ID_)
		int			limit				// The new limit, or 0 for default
	)
//
//	The Parse() method uses several internal buffers to parse the data. There is 
//	a default buffer limit for each buffer. However, the application can override 
//	these limits by calling SetBufferLimit() with the ID of the buffer and the new 
//	limit value.
//	SetBufferLimit() must be called before calling the Parse() method.
//	SetBufferLimit() returns the old limit value.
//	Calling SetBufferLimit() with a buffer ID and a limit value of 0 causes the 
//	limit to be set back to it's default value.
//	Calling SetBufferLimit() with the buffer ID SXML_BUFFER_ID_ALL causes all buffer 
//	limits to be set to their defaults.
//	SetBufferLimit() only needs to be called if the application desires a limit other 
//	than the default.
//	
//	BuffreID				Default	Purpose
//	SXML_BUFFER_ID_NAME		80		Controls the maximum size of a tag or attribute name. 
//									If a name is longer than this limit, the additional
//									characters are ignored.
//	SXML_BUFFER_ID_DATA		2048	Controls how much data can be passed in a 
//									single data event. Data items larger than this 
//									limit are sent using multiple events.
//	SXML_BUFFER_ID_NESTING	16		Controls the tag nesting depth.This value times 
//									SXML_BUFFER_ID_NAME is allocated for the nesting 
//									buffer. The nesting buffer holds the name of the 
//									current tag and all higher level tags in the hierarchy. 
//									Once the string length of all of these names 
//									reaches the limit of the nesting buffer, then no 
//									more sub tags can be parsed and the lower level 
//									sub tags are ignored.Using the default of 16, 
//									there will be room for nesting of AT LEAST 16 tags. 
//									More tags can be nested if the tag names are shorter 
//									than SXML_BUFFER_ID_NAME.
//	SXML_DEFAULT_READ_SIZE	2048	Controls the read buffer used to read in data from the Read() method.
//	SXML_BUFFER_ID_ALL				Using this ID sets all buffer limits to their defaults.
//	
//
//------------------------------------------------------------------------------//
{
	int	oldLimit = 0;

	switch (bufferID)
	{
		case SXML_BUFFER_ID_ALL:
			nameBufferSize = SXML_DEFAULT_NAME_SIZE;
			dataBufferSize = SXML_DEFAULT_DATA_SIZE;
			readBufferSize = SXML_DEFAULT_READ_SIZE;
			nestBufferDepth= SXML_DEFAULT_NESTING;
			break;

		case SXML_BUFFER_ID_NAME:
			oldLimit = nameBufferSize;
			if (limit == 0)
				nameBufferSize = SXML_DEFAULT_NAME_SIZE;
			else
				nameBufferSize = limit;
			break;

		case SXML_BUFFER_ID_DATA:
			oldLimit = dataBufferSize;
			if (limit == 0)
				dataBufferSize = SXML_DEFAULT_DATA_SIZE;
			else
				dataBufferSize = limit;
			break;

		case SXML_BUFFER_ID_NESTING:
			oldLimit = nestBufferDepth;
			if (limit == 0)
				nestBufferDepth = SXML_DEFAULT_NESTING;
			else
				nestBufferDepth = limit;
			break;

		case SXML_BUFFER_ID_READ:
			oldLimit = readBufferSize;
			if (limit == 0)
				readBufferSize = SXML_DEFAULT_READ_SIZE;
			else
				readBufferSize = limit;
			break;
	}
	
	return oldLimit;
}	

//--------------------------------------------------------------------------------
//
	int									// Old value
	CSXML::SetOption
	(
		int			optionID,			// Which buffer limit to set (SMXL_OPTION_ID_)
		int			value				// TRUE or FALSE
	)
//
//	Int SetOption( int optionID, int value )
//	
//	The SetOption() method turns an option on or off. "optionID" indicates with 
//	option. These Ids are defined in SXML.h. The "value" controls if the options 
//	is enabled or disabled. 
//	SetOption() must be called before calling the Parse() method.
//	SetOption() returns the old option value.
//	Calling SetOption() with a buffer limit ID and a limit value of 0 causes the 
//	limit to be set back to it's default value.
//	Calling SetOption() with the buffer ID SXML_OPTION_ID_ALL_DEFAULT causes all 
//	options to be set to their defaults.
//	SetOption() only needs to be called if the application wants to set an option 
//	to something other than their default.
//	
//	optionID							Default	Purpose
//	SXML_OPTION_ID_INCLUDE_WHITE_SPACE	off		Enable this option to include all 
//												leading and trailing while space 
//												(space, tab, return) around data 
//												reported by SXML_EVENT_DATA. When 
//												disabled (default), the leading and 
//												trailing white space is stripped from 
//												the data before it is passed to Event().			
//	SXML_OPTION_ID_ALL_DEFAULT					Using this ID sets all options to their defaults.
//
//------------------------------------------------------------------------------//
{
	int	oldValue = 0;

	switch (optionID)
	{
		case SXML_OPTION_ID_ALL_DEFAULT:
			includeWhiteSpace = 0;
			break;
		case SXML_OPTION_ID_INCLUDE_WHITE_SPACE:
			oldValue = includeWhiteSpace;
			includeWhiteSpace = value;
			break;
	}

	return oldValue;
}

//--------------------------------------------------------------------------------
//
	const char *
	CSXML::GetName
	(
	)
//
//	The GetName() method returns a pointer to the name associated with the current
//	event. The meaning of the name depends on the type of event.
//
//	Event					Meaning of GetName()
//	SXML_EVENT_START_TAG	GetName() returns a pointer to the name of the tag starting.
//	SXML_EVENT_END_TAG		GetName() returns a pointer to the name of the tag ending.
//	SXML_EVENT_ATTRIBUTE	GetName() returns a pointer to the name of the attribute.
//	All other events		GetName()'s return value is undefined.
//
//------------------------------------------------------------------------------//
{
	return pNameBuffer;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::GetNameLength
	(
	)
//
//	Returns the length of the name
//
//------------------------------------------------------------------------------//
{
	return nameBufferCount;
}

//--------------------------------------------------------------------------------
//
	const char *
	CSXML::GetData
	(
	)
//
//	The GetData() method returns a pointer to the data associated with the current 
//	event. The meaning of the data depends on the type of event.
//	
//	Event					Meaning of GetData()
//	SXML_EVENT_DATA			GetData() returns a pointer to the data for the current tag.
//	SXML_EVENT_ATTRIBUTE	GetData() returns a pointer to the data for the current attribute.
//	SXML_EVENT_CDATA		GetData() returns a pointer to the data for the current CDATA section.
//	All other events		GetData ()'s return value is undefined.
//
//------------------------------------------------------------------------------//
{
	return pDataBuffer;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::GetDataLength
	(
	)
//
//	Returns the length of the data in the data buffer
//
//------------------------------------------------------------------------------//
{
	return dataBufferCount;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::IsMore
	(
	)
//
//	As described previously, XML data items can be larger than SXML's internal 
//	buffer. In this case the data item is sent in segments with multiple calls 
//	to the Event() method. The Event() method can call IsMore() to find out if 
//	subsequent data events will be sent with more data. IsMore() returns non-zero 
//	if there is more data to send. IsMore() is only meaningful for SXML_EVENT_DATA, 
//	SXML_EVENT_ATTRIBUTE, SXML_EVENT_ATTRIBUTE_CONTINUED, and SXML_EVENT_CDATA.
//	
//------------------------------------------------------------------------------//
{
	return more;
}

//--------------------------------------------------------------------------------
//
	void
	CSXML::SkipTag
	(
	)
//
//	The SkipTag() method causes all data, CDATA, sub tags, and attributes to be 
//	ignored until the end of the current tag. The next event sent to Event() will 
//	be an SXML_EVENT_END_TAG for the current tag. SkipTag() can be called at any 
//	point between a start and end tag.
//	SkipTag() is very important for handling unknown tags. Consider the following example:
//	
//	<employee>
//		<spouse>
//			<name>Judy smith</name>
//		</spouse>
//		<name>Ken Smith</name>
//	</employee>
//	
//	Let's assume an application parsing this file expects <employee> and the sub
//	tag <name>, however, the application does not recognize <spouse>. If the 
//	application does nothing when it receives the <spouse> start tag, the next 
//	tag event it will receive will be <name> and "Judy Smith" as data. The 
//	application might confuse the <name> tag under <spouse> as the <name> tag for <employee>.
//	Instead, the application should call SkipTag() when it receives the 
//	unrecognized tag of <spouse>. This would cause the <name>Judy Smith</name> 
//	to all be ignored. After calling SkipTag(), the next event the application 
//	will receive is the end tag of </spouse>.
//
//------------------------------------------------------------------------------//
{
	skipTag = 1;
}

//--------------------------------------------------------------------------------
//
	void
	CSXML::SkipData
	(
	)
//
//	The SkipData() method can be used to ignore the remaining data in a tag, 
//	attribute, or CDATA section. SkipData() is only meaningful for SXML_EVENT_DATA, 
//	SXML_EVENT_ATTRIBUTE, SXML_EVENT_ATTRIBUTE_CONTINUED, and SXML_EVENT_CDATA.
//	
//------------------------------------------------------------------------------//
{
	skipData = 1;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::GetLevel
	(
	)
//
//	Returns the current tag nesting level.
//	
//------------------------------------------------------------------------------//
{
	return level;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::GetFilePosition
	(
	)
//
//	Returns the current character index into the xml file.
//	
//------------------------------------------------------------------------------//
{
	return filePosition;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::GetTagPosition
	(
	)
//
//	This method is only useful for SXML_EVENT_START_TAG and SXML_EVENT_END_TAG.
//	For SXML_EVENT_START_TAG, the Tag Position is the file position of the '<'.
//	For SXML_EVENT_END_TAG, the tag position is the file position of the character
//	after the '>'
//	
//------------------------------------------------------------------------------//
{
	return tagPosition;
}

#ifdef SXML_TEST
//--------------------------------------------------------------------------------
//
	void
	CSXML::SetOutputFile
	(
		FILE	*fp						// New file to output to
	)
//
//	Sets the output file for hte default Event handler
//	
//------------------------------------------------------------------------------//
{
	fpOut = fp;
}
#endif

#ifdef SXML_TEST
//--------------------------------------------------------------------------------
//
	FILE *
	CSXML::GetOutputFile
	(
	)
//
//	Returns output file used by the default handler
//	
//------------------------------------------------------------------------------//
{
	return fpOut;
}
#endif

//--------------------------------------------------------------------------------
//
	const char *								// Returns ptr to root name
	CSXML::GetRootTagName
	(
	)
//
//	Returns a ptr to the top level tag name.
//
//	<top>
//		<middle>
//			<bottom>
//			</bottom>
//		</middle>
//	<top>
//
//	Assuming SXML has parsed to the <bottom> start tag and has sent the 
//	SXML_START_TAG with the name "bottom" has been sent to Event(), then:
//
//	char* root = GetRootTagName() - returns "top"
//	GetNextTagName( root ) - returns "middle"
//	char* cur = GetCurrentTagName() - returns "bottom"
//	GetPreviousTagName( cur ) - returns "middle"
//
//------------------------------------------------------------------------------//
{
	return pNestBuffer;
}

//--------------------------------------------------------------------------------
//
	const char *								// Returns ptr to current name
	CSXML::GetCurrentTagName
	(
	)
//
//	Returns a ptr to the currently open tag
//
//------------------------------------------------------------------------------//
{
	int	nestNameIndex;

	nestNameIndex = nestBufferCount-2;

	while (nestNameIndex > 0)
	{
		if (pNestBuffer[nestNameIndex-1] == 0)
			break;

		nestNameIndex--;
	}

	return &pNestBuffer[nestNameIndex];
}

//--------------------------------------------------------------------------------
//
	const char *								// Returns ptr to next name or NULL if no more names
	CSXML::GetNextTagName
	(
		const char	*pCurrentName			// Ptr to the current name
	)
//
//	Returns a ptr to the name of the tag one level down from the specified 
//	tag name <pCurrentName>
//
//------------------------------------------------------------------------------//
{
	if (pCurrentName < pNestBuffer || pCurrentName > &pNestBuffer[nestBufferCount])
	{
		// bad input
		return NULL;
	}

	pCurrentName += strlen(pCurrentName);

	if (*pCurrentName == 0)
		return NULL;

	return pCurrentName;
}

//--------------------------------------------------------------------------------
//
	const char *								// Returns ptr to previous name or NULL if no more names
	CSXML::GetPreviousTagName
	(
		const char	*pCurrentName			// Ptr to the current name
	)
//
//	Returns a ptr to the name of the tag one level up from the specified 
//	tag name <pCurrentName>
//
//------------------------------------------------------------------------------//
{
	--pCurrentName;

	if (pCurrentName <= pNestBuffer || pCurrentName > &pNestBuffer[nestBufferCount])
	{
		// bad input
		return NULL;
	}

	while (*pCurrentName && pCurrentName != pNestBuffer)
	{
		pCurrentName--;
	}

	if (pCurrentName <= pNestBuffer)
		return NULL;

	return pCurrentName;
}




//--------------------------------------------------------------------------------
//							Parser Helpers
//--------------------------------------------------------------------------------


//--------------------------------------------------------------------------------
//
	int
	CSXML::HelperStates
	(
		int			event
	)
//
//	Handles parser helper states
//
//------------------------------------------------------------------------------//
{
	switch (state)
	{
		case SXML_STATE_STRING:			return HelperString(event);			break;
		case SXML_STATE_STRING_ALLOC:	return HelperString(event);			break;
		case SXML_STATE_INT:			return HelperInt(event);			break;
		default:
			return SXML_ERROR_UNKNOWN_STATE;
			break;
	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::HelperString
	(
		int			event
	)
//
//	Parses a string.
//	Uses strPtr, strLen, and prevState
//
//------------------------------------------------------------------------------//
{
	int		length;
	char	*p;

	switch (event)
	{
		case SXML_EVENT_START_TAG:
			SkipTag();
			break;

		case SXML_EVENT_DATA:

			if (state == SXML_STATE_STRING_ALLOC)
			{
				length = GetDataLength() + 1;

				if (strLength == 0)
					strLength = length;
				else if (length < strLength)
					strLength = length;

				/* FIXME: who deletes this? */
				p = new char[strLength];

				if (p == NULL)
					return SXML_ERROR_OUT_OF_MEMORY;

				/* FIXME: (rwa) What the hack does this strange code? */
				* (char **) strPtr = p;
				strPtr = p;
			}
					
			strncpy( strPtr, GetData(), strLength-1 );
			strPtr[strLength-1] = 0;
			state = prevState;
			SkipTag();
			// Fall through to end tag

		case SXML_EVENT_END_TAG:
			state = prevState;
			break;
	}
	
	return 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::HelperInt
	(
		int			event
	)
//
//	Parses an int.
//	Uses intPtr, and prevState
//
//------------------------------------------------------------------------------//
{
	switch (event)
	{
		case SXML_EVENT_START_TAG:
			SkipTag();
			break;

		case SXML_EVENT_DATA:
			*intPtr = atoi(GetData());
			state = prevState;
			SkipTag();
			// Fall through to end tag

		case SXML_EVENT_END_TAG:
			state = prevState;
			break;
	}
	
	return 0;
}








//--------------------------------------------------------------------------------
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	PRIVATE CODE	
//--------------------------------------------------------------------------------


//--------------------------------------------------------------------------------
//
	int									// error code
	CSXML::AllocBuffers
	(
		const char	*	pXMLData,			// Ptr to XML data or NULL to use Read()
		int			length				// Length of the XML data, if pXMLData != NULL
	)
//
//	PRIVATE CODE
//
//	Allocates all of the buffers need by Parse().
//
//------------------------------------------------------------------------------//
{
	// Clear all of the pointers

	pNameBuffer = NULL;
	pDataBuffer = NULL;
	pNestBuffer = NULL;
	pReadBuffer = NULL;

	// Allocate the name buffer

	pNameBuffer = (char *) malloc( nameBufferSize );

	if (pNameBuffer == NULL)
		return -1;

	// Allocate the data buffer

	pDataBuffer = (char *) malloc( dataBufferSize+1 );

	if (pDataBuffer == NULL)
		return -1;

	// Allocate Nesting buffer

	nestBufferSize = (nestBufferDepth+1) * nameBufferSize +1;

	pNestBuffer = (char *) malloc( nestBufferSize );

	if (pNestBuffer == NULL)
		return -1;

	nestBufferCount = 0;
	pNestBuffer[0] = pNestBuffer[1] = 0;

	// Allocate the read buffer (or use the user's buffer)

	if (pXMLData == NULL)
	{
		// Using internal buffer and Read()'ing

		pReadBuffer = (char *) malloc( readBufferSize );

		if (pReadBuffer == NULL)
			return -1;

		readBufferCount	= 0;
		userBuffer		= 0;
	}
	else
	{
		// Using the caller's buffer, Read() will never be called

		pReadBufferConst	= pXMLData;
		readBufferCount = length;
		userBuffer		= 1;
	}

	filePosition = 0;

	readPtr = pReadBuffer;

	return 0;
}

//--------------------------------------------------------------------------------
//
	void
	CSXML::FreeBuffers
	(
	)
//
//	PRIVATE CODE
//
//	Frees all  of the buffers used by parse
//
//------------------------------------------------------------------------------//
{
	// Free the buffers

	if (pNameBuffer != NULL)
		free(pNameBuffer);

	if (pDataBuffer != NULL)
		free(pDataBuffer);

	if (pNestBuffer != NULL)
		free(pNestBuffer);

	if (!userBuffer)
		if (pReadBuffer != NULL)
			free(pReadBuffer);

	pNameBuffer = NULL;
	pDataBuffer = NULL;
	pNestBuffer = NULL;
	pReadBuffer = NULL;
}


//--------------------------------------------------------------------------------
// Parsing Functions
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	void
	CSXML::SkipWhiteSpace
	(
	)
//
//	PRIVATE CODE
//
//	Skips past all white space characters.
//
//------------------------------------------------------------------------------//
{
	char	ch;

	while (1)
	{
		ch = NextChar();
		if (isspace(ch))
			AdvancePtr(1);
		else
			return;
	}

	return;
}
	
//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseComment
	(
	)
//
//	PRIVATE CODE
//
//	Parse from the current position until it finds EOF or "-->". All comment
//	data is thrown away.
//
//------------------------------------------------------------------------------//
{
	char		ch;
	const char	*	ptr;

	// See if there is a comment start

	ptr = LookAhead(4);
	
	if (ptr == NULL)
		return SXML_ERROR_NOT_FOUND;

	if (ptr[0] != '<' || ptr[1] != '!' || ptr[2] != '-' || ptr[3] !='-')
		return SXML_ERROR_NOT_FOUND;

	AdvancePtr(4);

	// Parse the rest of the comment

	do
	{
		ch = GetChar();

		if (ch == '-')
		{
			ptr = LookAhead(2);

			if (ptr == NULL)
				return SXML_ERROR_EOF;

			if (ptr[0] == '-' && ptr[1] == '>')
			{
				AdvancePtr(2);
				return 0;
			}
		}

	} while (ch != 0);

	return SXML_ERROR_EOF;
}

//--------------------------------------------------------------------------------
//
	int
	CSXML::ParseName
	(
	)
//
//	PRIVATE CODE
//
//	Parses a tag, attribute, or processing name
//	The name is placed in pNameBuffer, and the length of the name is in nameBufferCount
//
//------------------------------------------------------------------------------//
{
	char		ch;
	int			truncated = 0;

	// Parse the first character

	nameBufferCount = 0;

	ch = GetChar();

	if (ch == 0)
		return SXML_ERROR_EOF;

	if ( !isNameFirstChar(ch) )
		return SXML_ERROR_SYNTAX;
	
	pNameBuffer[nameBufferCount++] = ch;

	// Parse Remaining characters

	while (1)
	{
		ch = NextChar();

		if (ch == 0)
			return SXML_ERROR_EOF;

		if ( !isNameChar(ch) )
			break;

		if (nameBufferCount < nameBufferSize - 1)
			pNameBuffer[nameBufferCount++] = ch;
		else
			truncated = 1;

		AdvancePtr(1);
	}

	pNameBuffer[nameBufferCount] = 0;

	return SXML_NO_ERROR;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseEntity
	(
	)
//
//	PRIVATE CODE
//
//	Parses a an entity macro and inserts data into the data buffer.
//	NOTE: This function only recognizes pre-defined entities and character references.
//
//------------------------------------------------------------------------------//
{
	char	name[MAX_ENTITY_NAME];
	int		length = 0;
	int		value;
	int		base = 10;
	char	ch;

	// See if there is an entity start

	if (NextChar() != '&')
		return SXML_ERROR_NOT_FOUND;

	AdvancePtr(1);

	// See if this is character reference

	if (NextChar() == '#')
	{
		// it is...

		AdvancePtr(1);
		
		value = 0;

		while ( (ch = GetChar()) != ';' )
		{
			if (ch == 0)
				return SXML_ERROR_EOF;

			if (ch == 'x')
			{
				base = 16;
				continue;
			}

			value *= base;

			if (isdigit(ch))
				value += ch-'0';
			else if (base == 16)
			{
				if (ch >= 'A' && ch <= 'F')
					value += ch - 'A' + 10;
				else if (ch >= 'a' && ch <= 'f')
					value += ch - 'a' + 10;
				else
					return SXML_ERROR_BAD_ENTITY;
			}
		}
	}
	else
	{

		// Get the entity name

		while (1)
		{
			ch = GetChar();

			if (ch == 0)
				return SXML_ERROR_EOF;

			if (ch == ';')
				break;

			if (length == MAX_ENTITY_NAME-1)
				return SXML_ERROR_BAD_ENTITY;

			name[length++] = ch;
		}

		name[length] = 0;

		// Find the value for the entity

		value = FindToken(name,entityTokenList);

		if (value == -1)
			return SXML_ERROR_BAD_ENTITY;

	}

	// put the data in the data buffer

	pDataBuffer[dataBufferCount++] = (char) value;

	return SXML_NO_ERROR;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseQuotedData
	(
		int		*_more					// On input,  0  = First parse of string
										//            !0 = continue parsing
										// On Output, 0  = finished with data,
										//			  !0 = there is more data
	)
//
//	PRIVATE CODE
//
//	The data is placed in pDataBuffer, the length is in dataBufferCount.
//	If there is more data to be parsed, *more = !0 on return.
//
//------------------------------------------------------------------------------//
{
	char		ch;
	int		result;

	// If this is the beginning of the string, see if there is a quote

	if (!*_more)
	{
		ch = NextChar();

		if (ch != 0x22 && ch != 0x27)
			return SXML_ERROR_NOT_FOUND;

		AdvancePtr(1);
	}

	// Init counters

	*_more = 0;
	dataBufferCount = 0;

	// Parse Remaining characters

	while (1)
	{
		ch = NextChar();

		if (ch == 0)
			return SXML_ERROR_EOF;

		if (ch == 0x22 || ch == 0x27)
		{
			AdvancePtr(1);
			break;
		}

		if (ch == '&')
		{
			result = ParseEntity();
			if(result != 0)
			   return SXML_ERROR_BAD_ENTITY;

			continue;
		}

		if (dataBufferCount == dataBufferSize)
		{
			*_more = 1;
			break;
		}

		pDataBuffer[dataBufferCount++] = GetChar();
	}

	pDataBuffer[dataBufferCount] = 0;

	return SXML_NO_ERROR;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseData
	(
	)
//
//	PRIVATE CODE
//
//	Parses a data inside a tag
//
//------------------------------------------------------------------------------//
{
	char		ch;
	int			result = 0;

	// Init counters

	skipData = 0;

	// Skip leading white space

	if (!includeWhiteSpace)
		SkipWhiteSpace();

	// Make sure there is some data

	if (NextChar() == '<')
		return SXML_ERROR_NOT_FOUND;

	// Parse data

	do
	{
		dataBufferCount = 0;

		while (1)
		{
			ch = NextChar();

			if (ch == 0)
				return SXML_ERROR_EOF;

			if (ch == '<')
			{
				more = 0;
				break;
			}

			if (dataBufferCount == dataBufferSize)
			{
				more = 1;
				break;
			}

			if (ch == '&')
			{
				result = ParseEntity();
				if (result != 0)
					return result;

				continue;
			}

			pDataBuffer[dataBufferCount++] = GetChar();
		}

		// If this is the end of the data block, strip trailing white space

		if (!includeWhiteSpace && !more)
		{
			while (dataBufferCount > 0)
			{
				if (!isspace(pDataBuffer[dataBufferCount-1]))
					break;

				dataBufferCount--;
			}
		}
						
		pDataBuffer[dataBufferCount] = 0;

		// Send the data to the user

		if (!skipData && dataBufferCount != 0 && !skipTag)
		{
			result = Event(SXML_EVENT_DATA);

			if (result != 0)
				return result;
		}

	} while (more);

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseCData
	(
	)
//
//	PRIVATE CODE
//
//	Parses a CDATA block
//
//------------------------------------------------------------------------------//
{
	char		ch;
	const char	*	ptr;
	int			result = 0;

	// See if there is a CDATA start

	ptr = LookAhead(9);
	
	if (ptr == NULL)
		return SXML_ERROR_NOT_FOUND;

	if (strncmp(ptr,"<![CDATA[",9))
		return SXML_ERROR_NOT_FOUND;

	AdvancePtr(9);

	// Init counters

	more = 0;
	skipData = 0;

	// Parse Remaining characters

	do
	{
		dataBufferCount = 0;

		while (1)
		{
			ch = NextChar();

			if (ch == 0)
				return SXML_ERROR_EOF;

			if (ch == ']')
			{
				ptr = LookAhead(3);

				if (ptr == NULL)
					return SXML_ERROR_EOF;

				if (ptr[1] == ']' && ptr[2] == '>')
				{
					// End of the CDATA Section

					AdvancePtr(3);
					more = 0;
					break;
				}
			}

			if (dataBufferCount == dataBufferSize)
			{
				more = 1;
				break;
			}

			pDataBuffer[dataBufferCount++] = GetChar();
		}

		pDataBuffer[dataBufferCount] = 0;

		// Send the data to the user

		if (!skipData && dataBufferCount != 0 && !skipTag)
		{
			result = Event(SXML_EVENT_DATA);

			if (result != 0)
				return result;
		}

	} while (more);

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseAttribute
	(
	)
//
//	PRIVATE CODE
//
//	This function will parse attributes and call Event() if they are found.
//	There can be 0,1, or more attributes.
//
//------------------------------------------------------------------------------//
{
	int		result = 0;
	int		event;
	char	ch;
	
	while (1)
	{
		SkipWhiteSpace();

		// See if this is the end of the tag

		ch = NextChar();

		if ( !isNameFirstChar(ch) )
			return 0;

		// Get the name

		result = ParseName();

		if (result != SXML_NO_ERROR)
			return result;

		// Skip past the '='

		SkipWhiteSpace();

		ch = GetChar();

		if (ch == 0)
			return SXML_ERROR_EOF;

		if (ch != '=')
			return SXML_ERROR_BAD_ATTRIBUTE;

		SkipWhiteSpace();

		// Get the Data
	
		more = 0;
		event = SXML_EVENT_ATTRIBUTE;
		skipData = 0;

		do
		{
			result = ParseQuotedData(&more);

			if (result != 0)
				return SXML_ERROR_SYNTAX;

			// Send the Event

			if (!skipData && !skipTag)
			{
				result = Event(event);

				if (result != 0)
					return result;
			}

			event = SXML_EVENT_ATTRIBUTE_CONTINUED;

		} while (more);

	}
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseStartTag
	(
		int			*empty				// Return !0 if the start tag is an empty tag
										// ( <tag/> ... no end tag expected )
	)
//
//	PRIVATE CODE
//
//	Parses a start tag and it's attributes,
//	Events are sent for the start of the tag and for each attribute
//
//------------------------------------------------------------------------------//
{
	int		result = 0;
	char	ch;
	const char	*ptr;
	int		skipEndTag = 0;
	int		tempNameBufferCount;

	*empty = 0;

	// Make sure this is a start tag

	ptr = LookAhead(2);

	if (ptr == NULL)
		return SXML_ERROR_NOT_FOUND;

	if (ptr[0] != '<')
		return SXML_ERROR_NOT_FOUND;

	if ( !isNameFirstChar(ptr[1]) )
		return SXML_ERROR_NOT_FOUND;

	tagPosition = filePosition;

	AdvancePtr(1);

	// Parse the name

	result = ParseName();

	if (result != 0)
		return result;

	// If we are skipping a tag, do not remember the name of hte sub tag

	if (!skipTag)
	{
		// See if we have space in the nesting buffer for a new tag

		if ( (nestBufferSize - nestBufferCount) >= nameBufferSize + 2 )
		{
			// Add the name to the nesting list

			strcpy(&pNestBuffer[nestBufferCount],pNameBuffer);

			nestBufferCount += nameBufferCount + 1;
			pNestBuffer[nestBufferCount] = 0;

			// Send start tag event

			level++;

			result =  Event( SXML_EVENT_START_TAG );

			if ( result != 0 )
				return result;
		}
		else
		{
			// We have reached our max nesting level...skip this and all sub tags

			SkipTag( );
		}
	}
	else
		skipTag++;

	// Skip Trailing white space

	SkipWhiteSpace();

	// Parse the attributes

	tempNameBufferCount = nameBufferCount;

	result = ParseAttribute();

	if (result != 0)
		return result;

	nameBufferCount = tempNameBufferCount;

	// Ok, close the tag

	ch = GetChar();

	if (ch == 0)
		return SXML_ERROR_EOF;

	if (ch == '/')
	{
		// Empty Tag...send end event

		*empty = 1;
		
		if ( (ch = GetChar()) == 0)
			return SXML_ERROR_EOF;

		tagPosition = filePosition;

		if (skipTag)
		{
			skipTag--;
			skipEndTag = 1;
		}

		if (!skipTag)
		{
			if (!skipEndTag)
				result =  Event( SXML_EVENT_END_TAG );
			else
				result = 0;

			level--;

			if (result != 0)
				return result;

			// Remove the tag from the nesting list

			nestBufferCount -= nameBufferCount + 1;
		}

	}
	else if (!skipTag)
	{
		// Tell handler that we are now at a data position

		Event(SXML_EVENT_DATA_START);
	}
	
	if (ch != '>')
		return SXML_ERROR_SYNTAX;


	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseEndTag
	(
	)
//
//	PRIVATE CODE
//
//	Parses a end tag and sents an end tag event
//
//------------------------------------------------------------------------------//
{
	int		result = 0;
	char	ch;
	const char	*ptr;
	int		nestNameIndex;
	int		skipEndTag = 0;

	// Make sure this is a end tag

	ptr = LookAhead(3);

	if (ptr == NULL)
		return SXML_ERROR_NOT_FOUND;

	if (ptr[0] != '<')
		return SXML_ERROR_NOT_FOUND;

	if (ptr[1] != '/')
		return SXML_ERROR_NOT_FOUND;

	if ( !isNameFirstChar(ptr[2]) )
		return SXML_ERROR_NOT_FOUND;

	AdvancePtr(2);

	// Parse the name

	result = ParseName();

	if (result != 0)
		return result;

	// Make sure we are expecting an end tag

	if (nestBufferCount == 0)
		return SXML_ERROR_SYNTAX;

	// Remove the closing bracket

	SkipWhiteSpace();

	ch = GetChar();

	if (ch == 0)
		return SXML_ERROR_EOF;

	if (ch != '>')
		return SXML_ERROR_BAD_TAG;

	// If we are skipping a tag, do not check it's name

	if (skipTag)
	{
		skipTag--;
		skipEndTag = 1;
	}

	if (!skipTag)
	{
		// Make sure the name matches the last name on the nesting list

		nestNameIndex = nestBufferCount-2;

		while (nestNameIndex > 0)
		{
			if (pNestBuffer[nestNameIndex-1] == 0)
				break;

			nestNameIndex--;
		}

		result = strcmp(&pNestBuffer[nestNameIndex],pNameBuffer);

		if (result != 0)
			return SXML_ERROR_BAD_TAG;

		// Remove the name from the nest list

		nestBufferCount = nestNameIndex;

		// Send event

		tagPosition = filePosition;

		if (!skipEndTag)
			result =  Event(SXML_EVENT_END_TAG);
		else
			result = 0;

		if (result != 0)
			return result;

		level--;

	}

	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseDocument
	(
	)
//
//	PRIVATE CODE
//
//	Parses a CDATA block
//
//	The document body contains start tags, end tags, parse data (PCDATA),
//	unparsed data (CDATA) and comments.
//
//------------------------------------------------------------------------------//
{
	int		result;
	int		empty;

	skipTag = 0;
	level = 0;

	// Signal the start of the document

	Event(SXML_EVENT_START_DOCUMENT);

	// A document must start with a tag

	result = ParseStartTag(&empty);

	if (result != 0)
		return result;

	if (!empty)
	{
		// Now parse the elements of the document body until the final end tag is found
		
		while (1)
		{
			// ----------------------------------------
			// Parse Data until we find something of interest

			result = ParseData();

			if (result != 0 && result != SXML_ERROR_NOT_FOUND)
				return result;

			// we reached the end of the data, let's see what's next

			// ----------------------------------------
			// Check for an end tag

			result = ParseEndTag();

			if (result == 0)
			{
				// We have an end tag

				if (level == 0)
				{
					// Document is finished

					break;
				}
				continue;
			}
			else if (result != SXML_ERROR_NOT_FOUND)
				return result;

			// ----------------------------------------
			// Check for a start tag

			result = ParseStartTag(&empty);

			if (result == 0)
			{
				// We have a start tag

				continue;
			}
			else if (result != SXML_ERROR_NOT_FOUND)
				return result;

			// ----------------------------------------
			// Check for a comment

			result = ParseComment();

			if (result == 0)
			{
				// We have a comment...ignore it

				continue;
			}
			else if (result != SXML_ERROR_NOT_FOUND)
				return result;

			// ----------------------------------------
			// Check for CDATA

			result = ParseCData();

			if (result == 0)
			{
				// CData is done,

				continue;
			}
			else if (result != SXML_ERROR_NOT_FOUND)
				return result;

			// ----------------------------------------
			// Unrecognized...error

			if (NextChar() == 0)
				return SXML_ERROR_EOF;
			else
				return SXML_ERROR_SYNTAX;

		}
	}

	result = Event(SXML_EVENT_END_DOCUMENT);

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// Error code
	CSXML::ParseProlog
	(
	)
//
//	PRIVATE CODE
//
//	Parses the Prolog.
//	This function is not implemented. It simply skips the prolog until we reach
//	the start of the document.
//
//------------------------------------------------------------------------------//
{
	char	ch;
	const char *	ptr;

	// Scan until we see a start tag

	do
	{
		ch = NextChar();

		if (ch == '<')
		{
			ptr = LookAhead(2);

			if (ptr == NULL)
				break;

			if (isNameFirstChar(ptr[1]))
			{
				// We have a start tag

				return 0;
			}
		}

		AdvancePtr(1);

	} while (ch != 0);

	return SXML_ERROR_EOF;
}

//--------------------------------------------------------------------------------
// Buffer Management functions
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	char
	CSXML::GetChar
	(
	)
//
//	Returns the next char from the read buffer. If the read buffer is empty,
//	then more data is read into the buffer and then the next char is returned.
//	If there is no more data, then \0 is returned.
//
//------------------------------------------------------------------------------//
{
	if (readBufferCount > 0)
	{
		readBufferCount--;
		filePosition++;
		return *readPtr++;
	}
	else
	{
		ReadMoreData();
		if (readBufferCount == 0)
			return 0;
		else
		{
			readBufferCount--;
			filePosition++;
			return *readPtr++;
		}
	}
}

//--------------------------------------------------------------------------------
//
	char
	CSXML::NextChar
	(
	)
//
//	Returns the next char from the read buffer, but does not advance the ptr.
//
//------------------------------------------------------------------------------//
{
	if (readBufferCount > 0)
		return *readPtr;
	else
	{
		ReadMoreData();
		if (readBufferCount == 0)
			return 0;
		else
			return *readPtr;
	}
}

//--------------------------------------------------------------------------------
//
	const char	*							// Ptr to the look ahead data
	CSXML::LookAhead
	(
		int			count				// # of bytes needed
	)
//
//	Returns a ptr to <count> bytes. The caller can index the returned ptr
//	to access the look ahead data. If there is not enough data, then null is
//	returned
//
//------------------------------------------------------------------------------//
{
	if (readBufferCount >= count)
		return readPtr;
	else
	{
		ReadMoreData();
		if (readBufferCount >= count)
			return readPtr;
		else
			return NULL;
	}
}

//--------------------------------------------------------------------------------
//
	int									// error code
	CSXML::AdvancePtr
	(
		int			count				// # of bytes to advance
	)
//
//	Advances (consumes) <count> bytes of the buffer.
//	usually used in conjunction with LookAhead().
//
//------------------------------------------------------------------------------//
{
	if (readBufferCount < count)
	{
		ReadMoreData();
		if (readBufferCount < count)
			return -1;
	}

	filePosition += count;
	readPtr += count;
	readBufferCount -= count;

	return 0;
}

//--------------------------------------------------------------------------------
//
	void
	CSXML::ReadMoreData
	(
	)
//
//	Reads more data into the read buffer and adjusts the ptrs and counters.
//
//------------------------------------------------------------------------------//
{
	int		residualCount;
	int		result;

	// First, copy any residual data to the start of the buffer

	residualCount = readBufferCount;

	if (residualCount > 0)
		memcpy(pReadBuffer,readPtr,residualCount);

	// Read more data

	result = Read(&pReadBuffer[residualCount], readBufferSize - residualCount );

	readBufferCount = residualCount + result;

	readPtr = pReadBuffer;
}

//--------------------------------------------------------------------------------
//
	int									// Token value or -1 if not found
	CSXML::FindToken
	(
		const char	*	pString,			// Ptr to the string to match
		SXML_TOKEN	*pTokenList			// Ptr to an array of SXML_TOKENs
										// List is terminated with a token that
										// has a null name ptr
	)
//
//	Searches a list of token names for one that matches the <pString>. Returns
//	the SXML_TOKEN.value or -1 if no tokens matched.
//
//------------------------------------------------------------------------------//
{
	int	x;

	for (x=0; pTokenList[x].name != NULL; x++)
	{
		if (strcmp(pTokenList[x].name,pString) == 0)
			return pTokenList[x].value;
	}

	return -1;
}

//--------------------------------------------------------------------------------
//								CSXML_FILE
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	int
	CSXML_FILE::ParseFile
	(
		const char	*	pFileName
	)
//
//	Prases the specified file
//	
//------------------------------------------------------------------------------//
{
	int		result = SXML_ERROR_EOF;

	fp = fopen(pFileName,"rb");

	if (fp != NULL)
	{
		result = Parse( NULL, 0 );

		fclose(fp);
	}

	return result;
}

//--------------------------------------------------------------------------------
//
	int									// # of bytes read or 0 for EOF
	CSXML_FILE::Read
	(
		char	*	pData,				// Ptr to the buffer
		int			count				// # of bytes to read
	)
//
//	Reads from the file provided to ParseFile
//	
//------------------------------------------------------------------------------//
{
	int	result;

	result = fread( pData, 1,count, fp );

	if (result < 0)
		result = 0;

	return result;
}


//--------------------------------------------------------------------------------
//								CSXML_SIO
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	int
	CSXML_SIO::ParseSIO
	(
		CSIO	*	_pSIO
	)
//
//	Prases the specified file
//	
//------------------------------------------------------------------------------//
{
	pSIO = _pSIO;
	
	return Parse( NULL, 0 );
}

//--------------------------------------------------------------------------------
//
	int									// # of bytes read or 0 for EOF
	CSXML_SIO::Read
	(
		char	*	pData,				// Ptr to the buffer
		int			count				// # of bytes to read
	)
//
//	Reads from the SIO provided to ParseSIO
//	
//------------------------------------------------------------------------------//
{
	int	result;

	if (this->userBuffer)
		return 0;

	result = pSIO->Read( pData, count );

	if (result < 0)
		result = 0;

	return result;
}


//--------------------------------------------------------------------------------
//								Test Code
//--------------------------------------------------------------------------------


#ifdef SXML_TEST


int	main(int argc, char * argv[])
{
	FILE	*fp;
	char	*buffer;
	int		length;
	CSXML	*pXml;
	int		result;

	if (argc < 2)
	{
		printf("Usage: SXML file -v\n  -v = verbose mode.\n");
		return 0;
	}

	if (argc >= 2)
		verbose = 1;
	else
		verbose = 0;

	fp = fopen(argv[1],"r");

	if (fp == NULL)
	{
		printf("Cannot open %s\n",argv[1]);
		return 0;
	}

	buffer = (char *) malloc(65534);

	length = fread(buffer,1,65534,fp);

	pXml = new CSXML;

	pXml->SetBufferLimit( SXML_BUFFER_ID_DATA, 512 );

	result = pXml->Parse( buffer, length );

	printf("result = %d\n");

	fclose(fp);

	free(buffer);

	return 0;
}

#endif

