/*-------------------------------------------------------------------------------

	SXPath.cpp

	Copyright (c) 2002, Raritan Computer, Inc.

	This a VERY small subset of XPath. This class can only handle the following
	XPath syntax:

	/ / *[@id="xxx"]		- Where xxx is the unique ID of an element node
	/element/child-element	- Any number of child desendents with no predicates.
	/element/child-el[1]	- An index is that only allow predicate
	/element/child/@attr	- specifies an attribute node
	
	// and * are not support except as stated in the first example
	no math or logic symbols are supported 
	no axes are supported

	This class represents on XPath and it's resultant node set.
	First call the Parse() method with the XPath.
	Next you can call Enum() to get each node in the node set.
	When the node set is no longer needed, delete the object.

--------------------------------------------------------------------------------*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<ctype.h>
#include	"pp/SXPath.h"
#include	"pp/SXDB.h"
#include	"pp/SXML_Errors.h"

//----------------------------------------
//				Equates
//----------------------------------------

#define	isNameFirstChar(a) ( isalpha(a) || a == '_' || a ==':' )
#define	isNameChar(a) ( isalpha(a) || a == '_' || a ==':' || a == '.' || a == '-' || isdigit(a) )

	// Token IDs

enum
{
	SXP_EOF,							// End of input
	SXP_ERROR,							// Unknown token
	SXP_SLASH,							// "/"
	SXP_DOUBLE_SLASH,					// "//"
	SXP_DOT	,							// "."
	SXP_DOT_DOT,						// ".."
	SXP_AT,								// @
	SXP_ASTERISK,						// *
	SXP_LT_BRACKET,						// [
	SXP_RT_BRACKET,						// ]
	SXP_EQUAL,							// =
//	SXP_NUMBER,							// Parsed number
	SXP_STRING,							// Parsed string

	SXP_MAX
};

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

//----------------------------------------
//				Code
//----------------------------------------

//--------------------------------------------------------------------------------
//									CSXPath
//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------
//
	CSXPath::CSXPath
	(
	)
//
//	Initialize data items
//
//------------------------------------------------------------------------------//
{
	count = 0;
	firstList.pNext = NULL;
}

//--------------------------------------------------------------------------------
//
	CSXPath::~CSXPath
	(
	)
//
//	Cleanup
//
//------------------------------------------------------------------------------//
{
	DeleteAllNodes();
}

//--------------------------------------------------------------------------------
//
	int									// returns error code or # of nodes found
	CSXPath::Parse
	(
		const char	*	_pXPath,			// Ptr to the XPath string
		CSXDB		*	pDB,			// Ptr to the database to search
		CSXDB_Node	*	pContextNode,	// The current context node
										// Set to NULL to use root as the context node
		int			_maxNodes,		// maxNodes to parse (0 - no limit)
		void		*	pUser			// User account
	)
//
//	Parses the xpath and creates the node set.
//	Call EnumNode() to enumerate nodes from the set.
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node		*	pNode;
	int					result;
	NODE_PARSE_INFO		info;

	DeleteAllNodes();

	// Init the parse context

	pXPath = _pXPath;
	pXPathStart = pXPath;
	tokenPosition = 0;
	processPredicate = 0;

	if (_maxNodes != 0)
		maxNodes = _maxNodes;
	else
		maxNodes = 0x7FFFFFFF;

	result = Lex();

	// See if this is a //*[@id= thing

	if (result == SXP_DOUBLE_SLASH)
	{
		pNode = NULL;
		result = SXML_ERROR_XPATH_SYNTAX;

		if (Lex() == SXP_ASTERISK)
		{
			if (Lex() == SXP_LT_BRACKET)
			{
				if (Lex() == SXP_AT)
				{
					if (Lex() == SXP_STRING && strcmp(string,"id")==0)
					{
						if (Lex() == SXP_EQUAL)
						{
							if (Lex() == SXP_STRING)
							{
								if (Lex() == SXP_RT_BRACKET)
								{
									pNode = Find(pDB,SXDB_TYPE_ATTRIBUTE,"id",string);
									result = SXML_ERROR_XPATH_NULL_SET;
								}
							}
						}
					}
				}
			}
		}

		if (pNode == NULL)
			return result;

		pContextNode = pNode;
		result = Lex();

		if (result == SXP_EOF)
		{
			if (pDB->CheckAccess( pNode, pUser, SXDB_ACCESS_READ ) )	// SECURITY
				AddNode(pNode);
			return count;
		}
		else if (result == SXP_SLASH)
			result = Lex();
		else
			return SXML_ERROR_XPATH_SYNTAX;
	}

	// See if we are starting from context or root

	if (result == SXP_SLASH)
	{
		// Special case root "/"

		pNode = pDB->Root();
		result = Lex();

	}
	else
	{
		if (pContextNode == NULL)
			pNode = pDB->Root();
		else
			pNode = pContextNode;
	}

	if (result == SXP_EOF)
	{
		AddNode( pNode );
		return count;
	}

	// Start parsing

	info.pDB = pDB;
	info.pUser = pUser;

	result = ParseNode(pNode, &info);

	return result == 0 ? count : result;

}

//--------------------------------------------------------------------------------
//
	int									// returns error code or # of nodes found
	CSXPath::ParseNode
	(
		CSXDB_Node	*	pContextNode,
		NODE_PARSE_INFO * pInfo
	)
//
//	Parses the xpath and creates the node set.
//	Call EnumNode() to enumerate nodes from the set.
//
//------------------------------------------------------------------------------//
{
	const char	*pNextChar;
	const char	*pPredicateStart;
	int			isAttribute = 0;
	CSXDB_Node	*pChild;
	int			index = 0;
	int			occurance = 0;
	int			leaf;
	int			result;
	int			step;
	int			token_save;
	char		name[SXP_MAX_STRING];

	// See if this is an attribute node

	if (token == SXP_AT)
	{
		isAttribute = 1;

		Lex();
	}

	// Check for ..

	if (token == SXP_DOT_DOT)
	{
		token = SXP_DOT;
		pContextNode = pContextNode->Parent();
		if (pContextNode == NULL)
			return 0;
	}

	// Parse the step

	step = token;

	switch (step)
	{
		case SXP_ASTERISK:
		case SXP_DOT:
			break;

		case SXP_DOT_DOT:
			pContextNode = pContextNode->Parent();
			if (pContextNode == NULL)
				return 0;
			break;

		case SXP_STRING:
			strcpy(name,string);
			break;

		default:
			return SXML_ERROR_XPATH_SYNTAX;
	}

	// See if there is an index predicate

	pPredicateStart = NULL;

	if (Lex() == SXP_LT_BRACKET)
	{
		pPredicateStart = pXPath;

		if (Lex() == SXP_STRING && isNumber)
		{
			index = number;

			if (index < 1)
				return SXML_ERROR_XPATH_SYNTAX;
		
			if (Lex() != SXP_RT_BRACKET)
				return SXML_ERROR_XPATH_SYNTAX;

			Lex();

			pPredicateStart = NULL;
		}
	}

	// See if this is a leaf node of the path

	pNextChar = pXPath;

	if (token == SXP_SLASH)
	{
		Lex();
		leaf = 0;
	}
	else
		leaf = 1;

	// Search for the node
	token_save = token;
	pChild = NULL;

	do
	{
		switch (step)
		{
			case SXP_ASTERISK:
				pChild = pContextNode->FindChildOfType( isAttribute ? SXDB_TYPE_ATTRIBUTE : SXDB_TYPE_ELEMENT, pChild );
				break;

			case SXP_DOT:
			case SXP_DOT_DOT:
				if (pChild == NULL)
					pChild = pContextNode;
				else
					pChild = NULL;
				break;

			case SXP_STRING:
				pChild = pContextNode->FindChildOfType( isAttribute ? SXDB_TYPE_ATTRIBUTE : SXDB_TYPE_ELEMENT, pChild );
				if (pChild != NULL  && pChild->GetName()!= NULL && strcmp(pChild->GetName(), name) != 0)
					continue;
				break;

			default:
				return SXML_ERROR_XPATH_SYNTAX;
		}

		if (pChild == NULL)
			break;

		// Evaluate the index predicate

		occurance ++;

		if (index && index != occurance)
			continue;

		// Evaluate the <path> and <path>="" predicates

		if (pPredicateStart != NULL)
		{
			processPredicate++;
			pXPath = pPredicateStart;
			Lex();
			result = ParseNode( pChild, pInfo );
			processPredicate--;
			
			if (result < 0)
				return result;

			if (result == 0)
				continue;

			Lex(); // Get the token after the predicate

			// See if this is a leaf node of the path

			pNextChar = pXPath;

			if (token == SXP_SLASH)
			{
				Lex();
				leaf = 0;
			}
			else
				leaf = 1;

		}

		// Start next step

		result = 0;
		
		if (leaf)
		{
			if (!processPredicate)
			{
				// Normal processing
				if (pInfo->pUser == NULL || pInfo->pDB->CheckAccess( pChild, pInfo->pUser, SXDB_ACCESS_READ ) )	// SECURITY
					result = AddNode( pChild );
			}
			else
			{
				// Predicate processing
				if (token == SXP_EQUAL)
				{
					// Node Data Comparison
					if (Lex() != SXP_STRING)
						return SXML_ERROR_XPATH_SYNTAX;

					if (pChild->GetData() != NULL)
						result = strcmp(pChild->GetData(),string) == 0 ? 1 : 0;

					if (Lex() != SXP_RT_BRACKET)
						return SXML_ERROR_XPATH_SYNTAX;

					if (result == 1)
						return 1;

					token = token_save;
					pXPath = pNextChar;
				}
				else if (token == SXP_RT_BRACKET)
				{
					return 1;
				}
				else
					return SXML_ERROR_XPATH_SYNTAX;
			}
		}
		else
		{
			result = ParseNode( pChild, pInfo );
			pXPath = pNextChar;
			Lex();
		}

		if (count >= maxNodes)
			return 0;

		if (result < 0)
			return result;

		if (index && index == occurance)
			break;
			
	} while (1);
			
	return 0;
}

//--------------------------------------------------------------------------------
//
	int									// Token type parsed
	CSXPath::Lex
	(
	)
//
//	Parses the next item in the string
//
//------------------------------------------------------------------------------//
{
	char	ch;
	int		x;
	char	quote = 0;

	isNumber = 0;

	// Parse special characters

	while (*pXPath && (*pXPath == ' ' || *pXPath == '\t' || *pXPath == 10 || *pXPath == 13) )
		pXPath++;

	tokenPosition = pXPath - pXPathStart;

	ch = *pXPath++;

	switch ( ch )
	{
		case 0:
			return token = SXP_EOF;
			break;
			
		case '/':
			if (*pXPath == '/')
			{
				token = SXP_DOUBLE_SLASH;
				pXPath++;
			}
			else
				token = SXP_SLASH;

			return token;
		
		case '.':
			if (*pXPath == '.')
			{
				token = SXP_DOT_DOT;
				pXPath++;
			}
			else
				token = SXP_DOT;

			return token;
		
		case '@':	token = SXP_AT;			return token;
		case '[':	token = SXP_LT_BRACKET;	return token;
		case ']':	token = SXP_RT_BRACKET;	return token;
		case '=':	token = SXP_EQUAL;		return token;
		case '*':	token = SXP_ASTERISK;	return token;
	}
	
	pXPath--;

	// See if this is a number

/*
	if ( isdigit(*pXPath) )
	{
		number = 0;
		for (x=0; ;x++)
		{
			if (!isdigit(*pXPath))
				break;

			number *= 10;
			number += (*pXPath++) - '0';
		}

		token = SXP_NUMBER;
		return token;
	}
*/

	// Parse the string
	
	if (*pXPath == '\'' || *pXPath == '\"')
		quote = *pXPath++;


	for (x=0; *pXPath; x++)
	{
		if (quote)
		{
			if (*pXPath == quote)
				break;
		}
		else
		{
			if (!isNameChar(*pXPath))
				break;
		}

		if (x<SXP_MAX_STRING-1)
			string[x] = *pXPath++;
	}

	if (quote)
	{
		if (*pXPath == quote)
			pXPath++;
		else
			token = SXP_ERROR;
	}

	string[x] = 0;

	token = SXP_STRING;

	if (!quote && x == 0)
		token = SXP_ERROR;

	for (x=0; string[x]; x++)
	{
		if (!isdigit(string[x]))
			break;
	}
	
	if (string[x] == 0 && x > 0)
	{
		isNumber = 1;
		number = atoi(string);
	}

	return token;
}

//--------------------------------------------------------------------------------
//
	int									// Token type parsed
	CSXPath::AddNode
	(
		CSXDB_Node	*pNode				// The node to add
	)
//
//	Adds a node to the node list
//
//------------------------------------------------------------------------------//
{
	int				list = count / SXP_NODES_PER_LIST;
	int				index = count % SXP_NODES_PER_LIST;
	CSX_NODE_LIST *	pList = &firstList;

	// Add it

	while (list--)
	{
		if (pList->pNext == NULL)
		{
			// Allocate a new list

			pList->pNext = new CSX_NODE_LIST;

			if (pList->pNext == NULL)
				return SXML_ERROR_OUT_OF_MEMORY;

			pList->pNext->pNext = NULL;
		}

		pList = pList->pNext;
	}

	pList->pNodes[index] = pNode;

	count++;

	return 0;
}

//--------------------------------------------------------------------------------
//
	CSXDB_Node	*						// The node or NULL if index is out of range
	CSXPath::Enum
	(
		int			index				// Which node to return
	)
//
//	Returns a ptr to the nth node in the node set
//
//------------------------------------------------------------------------------//
{
	int				list = (index) / SXP_NODES_PER_LIST;
	int				listIndex = (index) % SXP_NODES_PER_LIST;
	CSX_NODE_LIST *	pList = &firstList;

	if (index >= count)
		return NULL;

	while (list--)
	{
		if (pList->pNext == NULL)
				return NULL;

		pList = pList->pNext;
	}

	return pList->pNodes[listIndex];
}

//--------------------------------------------------------------------------------
//
	void
	CSXPath::DeleteAllNodes
	(
	)
//
//	Removes all nodes from the node list
//
//------------------------------------------------------------------------------//
{
	CSX_NODE_LIST *	pList = firstList.pNext;
	CSX_NODE_LIST *	pNext;
	
	while (pList != NULL)
	{
		pNext = pList->pNext;
		delete pList;
		pList = pNext;
	}

	firstList.pNext = NULL;

	count = 0;
}

//--------------------------------------------------------------------------------
//
	int
	CSXPath::GetTokenPosition
	(
	)
//
//	Returns the character offset to the last token parsed. Useful for displaying
//	error message about bad XPath statements.
//
//------------------------------------------------------------------------------//
{
	return tokenPosition;
}

//--------------------------------------------------------------------------------
//
	CSXDB_Node *						// the found node or NULL
	CSXPath::Find
	(
		CSXDB		*	pDB,			// The data base
		int				type,			// SXDB_TYPE_
		const char	*	pName,			// Name of the attribute
		const char	*	pData			// ID to find
	)
//
//	Searches to tree for an element with the attribute <pName> = pData
//
//------------------------------------------------------------------------------//
{
	pIDNode = NULL;
	FindRecursive( pDB->Root(), type, pName, pData );

	return pIDNode;
}

//--------------------------------------------------------------------------------
//
	int									// Token type parsed
	CSXPath::FindRecursive
	(
		CSXDB_Node	*	pNode,			// The data base
		int				type,			// SXDB_TYPE_
		const char	*	pName,			// Name of the attribute
		const char	*	pData			// ID to find
	)
//
//	Searches to tree for an element with the attribute <pName> = pData
//
//------------------------------------------------------------------------------//
{
	CSXDB_Node			* pChild;
	CSXDB_Data			* pDataNode;
	int					result;

	// See if this node matches

	pChild = pNode->FindChildOfTypeByName((SXDB_TYPE) type, pName, NULL);

	if (pChild)
	{
		pDataNode = (CSXDB_Data *) pChild->FindChildOfType(SXDB_TYPE_DATA, NULL);

		if (pDataNode)
		{
			if (strcmp(pDataNode->GetData(),pData) == 0)
			{
				pIDNode = pNode;
				return 1;
			}
		}
	}
				
	// Enum each child element of this node

	pChild = NULL;

	do
	{
		pChild = pNode->FindChildOfType( SXDB_TYPE_ELEMENT, pChild );

		if (pChild == NULL)
			break;

		result = FindRecursive( pChild, type, pName, pData );

		if (result != 0)
			return result;
	} while (1);

	return 0;
}
