/*--------------------------------------------------------------------------------

	SXDB_Parse_Table.h

	Copyright (c) 2003, Raritan Computer, Inc.

	Header file for SXDB_Parse_Table.cpp

--------------------------------------------------------------------------------*/

#ifndef _SXDB_Parse_Table_H_
#define	_SXDB_Parse_Table_H_

#include "pp/SXML_Errors.h"
#include "pp/SXDB.h"
#include "pp/SXML.h"

//----------------------------------------
//				Data Types
//----------------------------------------

typedef struct SXDB_PT_s SXDB_PT;

struct SXDB_PT_s {
    union {
	const char*	pName;		// Name of the Element or Attribute (in XML)
	SXDB_PT* 	pTable;		// Another parsetable description array (nested)
    };
    int			offset;		// Offset of the field in the struct. Parsed data will be written here.
    int			length;		// Length of the string FIXME: array ?
    int			flags;		// Type, processing
};

//----------------------------------------
//				Macros
//----------------------------------------

/**
 * Parse Table Macros
 * 
 * These macros can be used to define a XML parser description (parse
 * table). This parse table can then be used to parse an XML tree to a
 * struct with a generic parser function.
 * Internally a generated parse table is a variable with predefined values.
 * 
 * Attention: the define PT_STRUCT must contain the name (type) of
 * the target struct. Please undefine PT_STRUCT after completing
 * a PT definition to allow different parse tables.
 * 
 * Please read the detailed documentation in SX_DB_Parse_Table.cpp
 */

/* FIXME: These macros are evil - they circumvent type check! */
#define	PT_BEGIN(      name, table_name,    flags)         SXDB_PT table_name[] = { \
                                                               { name,                0,                          sizeof(PT_STRUCT),  flags | PT_BEGIN_T },
#define PT_ATT(        name, field, length, flags)             { name,                (int) ( & ((PT_STRUCT*) 0)->field),    length,  flags | PT_ATT_T },
#define PT_ELEM(       name, field, length, flags)             { name,                (int) ( & ((PT_STRUCT*) 0)->field),    length,  flags | PT_ELEM_T },
#define PT_ROOT_DATA(        field, length, flags)             { NULL,                (int) ( & ((PT_STRUCT*) 0)->field),    length,  flags | PT_ROOT_T },
#define	PT_EMPTY_ELEM( name, field,         flags)             { name,                (int) ( & ((PT_STRUCT*) 0)->field),    0,       flags | PT_EMPTY_ELEM_T },
#define	PT_ELEM_EXIST( name, field,         flags)             { name,                (int) ( & ((PT_STRUCT*) 0)->field),    0,       flags | PT_EMPTY_ELEM_T },
#define	PT_MOVE_DOWN(  name,                flags)             { name,                0,                                     0,       flags | PT_MOVE_DOWN_T },
#define	PT_MOVE_UP()                                           { NULL,                0,                                     0,       PT_MOVE_UP_T },

#define	PT_TABLE(      table, field,        flags)             { (const char*) table, (int) ( & ((PT_STRUCT*) 0)->field),    1,       flags | PT_TABLE_T },
#define	PT_ARRAY(      table, field, count, flags, countVar)   { (const char*) table, (int) ( & ((PT_STRUCT*) 0)->field),    count,   flags | PT_ARRAY_T}, \
                                                               { NULL,                (int) ( & ((PT_STRUCT*) 0)->countVar), 0,       PT_ARRAY2_T },
#define	PT_END                                                 { NULL,                0,                                     0,       PT_END_T} };



// Parse Table Flags
enum
{
    PT_STRING             = 0x00100,			// char[] data type, length is size of array
    PT_STRING_PTR         = 0x00200,			// char* data type
    PT_INT                = 0x00400,			// int data type
    PT_NODE               = 0x00800,			// Node address, can only be read from XML, can not be written to XML
    PT_ULONG              = 0x01000,			// Unsigned long
    PT_IP                 = 0x02000,			// Dotted IP address in XML, int in struct
    PT_REQUIRED           = 0x08000,			// If this item is not in the DB,
                                   			// SXD_PT_ERROR_MISSING_ITEM is returned
    PT_OPTIONAL	          = 0x00000,			// Data item is option for reads
    
    PT_CONDITION_1        = 0x00001,			// Only read/write if condition 1 is true
    PT_CONDITION_2        = 0x00002,			// Only read/write if condition 2 is true
    PT_CONDITION_3        = 0x00004,			// Only read/write if condition 3 is true
    PT_CONDITION_4        = 0x00008,			// Only read/write if condition 4 is true

    PT_UNKNOWN_OK         = 0x00000,			// Unknown elements and attributes are ok
    PT_NO_UNKNOWN         = 0x08000,			// Unknown elements and attributes cause error

    PT_DEL_BEFORE_UPDATE  = 0x010000			// Delete node(s) in the database before updating with user supplied entries.
};


// Parse Table Type Flags (Internal Use)
enum
{
	PT_TYPE_MASK	= 0xFF000000,		// Used for internal processing
	PT_CONDITION_MASK = 0x000F,			// 
	PT_UPDATE_TYPE_MASK = 0x00FF0000,	// (Internal) way to udpate the entries

	PT_ROOT_T		= 0x01000000,		// Use the root node's data
	PT_MOVE_DOWN_T	= 0x02000000,		// Move down into an element
	PT_MOVE_UP_T	= 0x03000000,		// Move up out of an element
	PT_ATT_T		= 0x04000000,		// Attribute
	PT_ELEM_T		= 0x05000000,		// Element
	PT_EMPTY_ELEM_T	= 0x06000000,		// Empty element type
	PT_BEGIN_T		= 0x07000000,		// Start of table
	PT_END_T		= 0x08000000,		// End of table
	PT_TABLE_T		= 0x09000000,		// Sub Table declaration
	PT_ARRAY_T		= 0x0A000000,		// Sub Table Array declaration
	PT_ARRAY2_T		= 0x0B000000		// Used as private extended array info
};

	// SXDB_Put modes

enum
{
	PT_NODE_UPDATE,				// pNode points to the element that will be updated
	PT_CHILD_UPDATE,			// pNode points to parent of the element to be updated
								// The first occurance on the element will be updated
								// If the element does not exist, it will be created
	PT_CHILD_UPDATE_IF_EXISTS,	// pNode points to parent of the element to be updated
								// If the element does not exist, PT_ROOT_NOT_FOUND
								// will be returned
	PT_CHILD_APPEND				// pNode points to perent of the node a new element
								// will be appended to the parrent
};

	// Error codes

enum
{
	PT_ERROR_BASE	= -1310,			// Base of our error codes
	PT_MISSING_FIELD,					// A PT_REQUIRED field was missing
	PT_UNKNOWN_DATA,					// Unknown data was found and the PT_NO_UNKNOWN
										// flag was set in the PT_BEGIN macro.
	PT_ROOT_NOT_FOUND,
	PT_BAD_TABLE,
	PT_TOO_MANY_ELEMENTS				// Number of elements exceeded the length in 
										// a table array (PT_TABLE, name, LENGTH, 0)
};

#define	MAX_XPATH_LENGTH		2048

//----------------------------------------
//				Function Prototypes
//----------------------------------------

	int									// 0 or error code
	SXDB_PT_Get
	(
		CSXDB		*pDB,				// Ptr to the database
		const char		*pXPath,			// XPath of element to parse
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int			conditionFlags = 0xFFFFFFFF
										// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	);

	int									// 0 or error code
	// Gets references to the data in the db. PT_STRING copies into fixed buffer
	SXDB_PT_Get
	(
		CSXDB_Node	*pNode,				// Node of the Element to parse
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int			conditionFlags = 0xFFFFFFFF
										// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	);

	int									// 0 or error code
        // data will be copied in the database (strings in the XML-db will be allocated)
        // overwritten data will be freed
	SXDB_PT_Put
	(
		CSXDB		*pDB,				// Ptr to the database
		const char	*pXPath,			// XPath of parent or node element (See mode)
		int		mode,				// PT_NODE_UPDATE, PT_CHILD_UPDATE
										// PT_CHILD_UPDATE_IF_EXISTS, or PT_CHILD_APPEND
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int		conditionFlags = 0xFFFFFFFF	// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	);

	int									// 0 or error code
        // data will be copied in the database (strings in the XML-db will be allocated)
        // overwritten data will be freed
	SXDB_PT_Put
	(
		CSXDB_Node	*pNode,				// Ptr to Parent or node element (See mode)
		int		mode,				// PT_NODE_UPDATE, PT_CHILD_UPDATE
										// PT_CHILD_UPDATE_IF_EXISTS, or PT_CHILD_APPEND
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		int		conditionFlags = 0xFFFFFFFF	// condition (Optional, Default is all true)
										// PT_CONDITION1 | PT_CONDITION2 | PT_CONDITION3
	);

	int									// 0 or error code
        // data will be copied in the database (strings in the XML-db will be allocated)
        // overwritten data will be freed
	SXDB_PT_XPath
	(
		char	*pXPath,			// XPath (only elements and index predicates)
		SXDB_PT		*pTable,			// Ptr to the table
		void		*pStruct,			// Ptr to the structure
		SXDB_PT		**pOutTable,		// Returns the ptr to the table component
		void		**pOutStruct,		// Returns the ptr to the structure component
		int			*pIndex				// Returns the table index location for normalize
	);

	char *								// Ptr to the new xPath or NULL if it fails
	CreateXPath
	(
		CSXDB_Node	*pParent,			// The parent node
		CSXDB_Node	*pChild,			// The child node
		int			insertIndexPredicate// 0=No indexes, !0 = insert indexes
	);

	CSXDB_Node	*						// The new node ptr or NULL if error
	SXDB_PT_Normalize
	(
		CSXDB_Node		*pNode,			// The root node to be normalized to the table
		SXDB_PT			*pTable,		// Ptr to the table
		int				index,			// index of the table entry to normalize to
										// -1 = No normalization
										//  0 = Normalize to the Table Root
										// >1 = Normalize to a sub element of the table
		int				create,			// 0 = Move up nodes only
										// 1 = create the needed nodes
		int				root			// 0 = Do not include table root
										// 1 = include table root			
	);

//----------------------------------------
//				Static Data
//----------------------------------------



#endif // _RDMP_H_


