/*-------------------------------------------------------------------------------

	CmdLineParse.cpp

	Copyright (c) 2003, Raritan Computer, Inc.

	Utility functions to parse tokens, values, and what not out of a string.
	Real handy for simple command line parsing

	Command lines are parsed from left to right. Each parse function takes
	in a ptr to the current position in the command line and returns a ptr
	to the position in the string after the next token. Use the return ptr
	in the next function call. Such as:

	char *p = "Command 123";
	char token[30];
	int		x,found;

	p = GetNextToken( p, token, 30 ); // token == "COMMAND"
	p = GetNextValue( p, &x, &found); // x = 123, found = 1
	p = GetNextValue( p, &x, &found); // x = 0,   found = 0 because there were no
						            more values in the string

--------------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include <pp/CmdLineParse.h>


//----------------------------------------
//				Equates
//----------------------------------------

//----------------------------------------
//				Data Types
//----------------------------------------

//----------------------------------------
//				Function Prototypes
//----------------------------------------

//----------------------------------------
//				Static Data
//----------------------------------------

static const char * defaultDelimiters = " ,;:.";

//----------------------------------------
//				Code
//----------------------------------------

// Returns ptr to character after white space
const char *					// Returns the ptr to the the to character in the string after the white space
SkipDelimiters(const char * string,	// String that contains the token
	       const char * delimiters)	// NULL for default delimiters or ptr to delimiter string
{
    if (delimiters == NULL) delimiters = defaultDelimiters;

    while (*string && (strchr(delimiters, *string) != NULL || *string < 32)) {
	++string;
    }

    return string;
}

// Gets a numeric value from a string
const char *				// Returns the ptr to the the to character in the string after the token
GetNextValue(const char * string,	// String that contains the token
	     int * value,		// where the value is stored
	     int * bFound,		// returns true if a number was found
	     const char * delimiters)	// NULL for default delimiters or ptr to delimiter string
{
    int hex = 0;
    int neg = 0;
    int mul = 10;

    if (delimiters == NULL) delimiters = defaultDelimiters;
    if (bFound != NULL) *bFound = 0;
    *value = 0;
    string = SkipDelimiters(string,delimiters);

    while (1) {
	char ch = toupper(*string);
	if (ch == 'X') {
	    hex = 1;
	    mul = 0x10;
	    string++;
	    continue;
	}

	if (ch == '-') {
	    neg = !neg;
	    string++;
	    continue;
	}

	if (ch >= '0' && ch <= '9') {
	    *value = (*value * mul) + ch - '0';
	} else if (hex && ch >= 'A' && ch <= 'F') {
	    *value = (*value * mul) + ch -'A' + 10;
	} else {
	    break;
	}

	if (bFound != NULL) *bFound = 1;
	string++;
    }

    if (neg) *value = -*value;

    return string;
}

// Gets a numeric value in the form of (xxx.xxx.xxx.xxx) from a string
// if there is a number, but no "." then the number is considered an ip address
// in unsigned inte form
const char *					// Returns the ptr to the the to character in the string after the token
GetNextIPAddress(const char * string,	// String that contains the token
		int * value,		// where the ip address value is stored
		int * bFound)		// returns true if a number was found
{
    int shift = 24;
    int temp, found;
    int ip = 0;

    if (bFound != NULL) *bFound = 0;

    do {
	string = GetNextValue( string, &temp, &found );
	if (!found) break;

	if (bFound != NULL) ++(*bFound);

	if (shift == 24 && *string != '.') {
	    // Integer form of IP address
	    ip = temp;
	    break;
	}

	ip |= (temp & 0xFF) << shift;
	shift -= 8;

	if (*string != '.') break;

	++string;
    } while (shift >= 0);

    *value = ip;

    return string;
}

// Gets a token from a string
const char *				// Returns the ptr to the the to character in the string after the token
GetNextToken(const char * string,	// String that contains the token
	     char * buffer,		// where the token is stored
	     int bufferSize,		// size of the buffer
	     int caseMode,		// -1 = lower case, 0 = no case conversion, 1 for upper case; (Defult is upper)
	     const char * delimiters)	// NULL for default delimiters or ptr to delimiter string
{
    int length = 0;
    int inQuote = 0;
	
    if (delimiters == NULL) delimiters = defaultDelimiters;

    string = SkipDelimiters(string, delimiters);

    while (*string && *string > 31 ) {
	if (*string == '\"' || *string == '\'')	{
	    inQuote = inQuote ? 0 : 1;
	}

	if (!inQuote) {
	    if (strchr(delimiters, *string) != NULL) break;
	}

	if (length++ < bufferSize - 1) {
	    if (caseMode > 0) {
		*buffer++ = toupper(*string++);
	    } else if (caseMode < 0) {
		*buffer++ = toupper(*string++);
	    } else {
		*buffer++ = *string++;
	    }
	} else {
	    ++string;
	}
    }

    *buffer = 0;

    return string;
}

// Returns the value of the token 'string' or 0 if 'string' does not
// match any of the names in the tokenList.
int
GetTokenValue(const char * string,	// String that contains the token
	      TOKEN * tokenList)	// list of TOKEN structures
{
    int value = TOKENNOTFOUND;

    while (tokenList->name != NULL) {
	if (strcmp( string, tokenList->name) == 0) {
	    value = tokenList->value;
	    break;
	}
	tokenList++;
    }

    return value;
}

// Gets a token from a string
int					// number of tokens in the string
CountTokens(const char * string,	// String that contains the token
	    const char * delimiters	// NULL for default delimiters or ptr to delimiter string
	   )
{
    int count = 0;
	
    if (delimiters == NULL) delimiters = defaultDelimiters;

    do {
	string = SkipDelimiters(string, delimiters);
	if (*string) {
	    count++;
	    while (*string && strchr(delimiters, *string) == NULL && *string > 31) {
		string++;
	    }
	}
    } while (*string);

    return count;
}
