/*
 * balloc.c -- Block allocation module
 *
 * Copyright (c) GoAhead Software Inc., 1995-2000. All Rights Reserved.
 *
 * See the file "license.txt" for usage and redistribution license requirements
 *
 * 2004 Peppercon AG: adapted to use pp_mallocator_t
 *   (I also removed part of the PP_B_STAT interface - all the fkts
 *    were possibly taking __File__, __line__ parameters,
 *    so the client code needed to be written accordingly.
 *    I don't wonna clog the client code with this, so currently 
 *    statistics of balloc has a global scope only.
 *    We may improve this by an automatic way to determine the call
 *    stack, for instance by getting the caller's PC, or something like that)
 * tbr@peppercon.de
 */

/******************************* Description *********************************/

/*
 * This module implements a very fast block allocation scheme suitable for
 * ROMed environments. It maintains block class queues for rapid allocation
 * and minimal fragmentation. This module does not coalesce blocks. The 
 * storage space may be populated statically or via the traditional malloc 
 * mechanisms. Large blocks greater than the maximum class size may be 
 * allocated from the O/S or run-time system via malloc. To permit the use 
 * of malloc, call bopen with flags set to PP_B_USE_MALLOC (this is the default).
 * It is recommended that bopen be called first thing in the application. 
 * If it is not, it will be called with default values on the first call to 
 * balloc(). Note that this code is not designed for multi-threading purposes
 * and it depends on newly declared variables being initialized to zero.
 */  

/******************************** Includes ***********************************/

#include <stdarg.h>
#include <stdlib.h>
#include <pthread.h>

#ifdef PP_B_STATS
#include <ctype.h>
#include <string.h>
#include <stdio.h>
#endif

#include <pp/base.h>

#include <pp/balloc.h>

/******************************** Defines ************************************/

#include "balloc_intern.h"

/********************************* Locals ************************************/

/************************** Forward Declarations *****************************/

#ifdef PP_B_STATS
static void bStatsAlloc(pp_mallocator_t* m, unsigned long id, void *ptr,
			int q, int size);
static void bStatsFree(pp_mallocator_t* m, unsigned long id, void *ptr,
		       int q, int size);
static void bstatsWrite(char *fmt, ...);
static int  bStatsFileSort(const void *cp1, const void *cp2);
#endif /* PP_B_STATS */

#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
static void bFillBlock(void *buf, int bufsize);
#endif

#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
static void verifyUsedBlock(bType *bp, int q);
static void verifyFreeBlock(bType *bp, int q);
static void verifyBallocSpace(pp_mallocator_t* a);
#endif /* PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD */

static void* bAlloc(pp_mallocator_t* m, size_t size);
static void  bFree(pp_mallocator_t* m, void *mp);
static void* bRealloc(pp_mallocator_t* m, void *mp, size_t newsize);
static void  bLock(pp_mallocator_t* m);
static void  bUnlock(pp_mallocator_t* m);
static void  bMutexInit(pp_mallocator_t* a, pp_mutex_t* m, pp_mutex_kind_t k);
static void  bDestroy(pp_mallocator_t* m);
static int ballocGetSize(int size, int *q);


/********************************* Code **************************************/

/*
 * Initialize the balloc module. bopen should be called the very first thing
 * after the application starts and bclose should be called the last thing 
 * before exiting. If bopen is not called, it will be called on the first 
 * allocation with default values. "buf" points to memory to use of size 
 * "bufsize". If buf is NULL, memory is allocated using malloc. flags may 
 * be set to PP_B_USE_MALLOC if using malloc is okay. This routine will allocate
 * an initial buffer of size bufsize for use by the application.
 */

pp_mallocator_t*
pp_bnew(void* buf, size_t bufsize, int flags) {
    return pp_bnew2(buf, bufsize, flags, bLock, bUnlock, bMutexInit);
}
    
/*
 * allows specification of lock and unlock function
 */
pp_mallocator_t*
pp_bnew2(void* buf, size_t bufsize, int flags,
	 void  (*lock)(pp_mallocator_t* a),
	 void  (*unlock)(pp_mallocator_t* a),
	 void  (*mutex_init)(pp_mallocator_t* a, pp_mutex_t* m,
			     pp_mutex_kind_t k))
{
    bmallocator_t* a = malloc(sizeof(bmallocator_t));
    
    pp_binit(a, buf, bufsize, flags, lock, unlock, mutex_init, 0);

    return (pp_mallocator_t*)a;
}

void
pp_binit(bmallocator_t* a, void* buf, size_t bufsize, int flags,
	 void  (*lock)(pp_mallocator_t* a),
	 void  (*unlock)(pp_mallocator_t* a),
	 void  (*mutex_init)(pp_mallocator_t* a, pp_mutex_t* m,
			     pp_mutex_kind_t k),
	 int is_proc_shared)
{
    assert((lock == NULL && unlock == NULL) ||
	   (lock != NULL && unlock != NULL));
    
    a->bFlags = flags;
    
#ifdef PP_B_STATS
    a->bStatsBlksMax = 0;
    a->bStatsFilesMax = 0;
    a->bStatsMemInUse = 0;
    a->bStatsBallocInUse = 0;
    a->bStatsMemMax = 0;
    a->bStatsBallocMax = 0;
    a->bStackMin = (void*) -1;
    a->bStatsMemMalloc = 0;
#endif /* PP_B_STATS */

    if (buf == NULL) {
	if (bufsize == 0) {
	    bufsize = PP_B_DEFAULT_MEM;
	}
	buf = malloc(bufsize);
#ifdef PP_B_STATS
	a->bStatsMemMalloc += bufsize;
#endif
    } else {
	a->bFlags |= PP_B_USER_BUF;
    }

    a->bFreeSize = a->bFreeLeft = bufsize;
    a->bFreeBuf = a->bFreeNext = buf;
    memset(a->bQhead, 0, sizeof(a->bQhead));
    
#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
    bFillBlock(buf, bufsize);
#endif
    
#ifdef PP_B_STATS
    a->bStackStart = &buf;
#endif
    
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    verifyFreeBlock(buf, bufsize);
#endif

    a->base.alloc      = bAlloc;
    a->base.realloc    = bRealloc;
    a->base.free       = bFree;
    a->base.lock       = lock;
    a->base.unlock     = unlock;
    a->base.mutex_init = mutex_init;
    a->base.destroy    = bDestroy;
    a->base.is_proc_shared = is_proc_shared;

    MUTEX_CREATE_ERRORCHECKING(&a->mtx);
}

/*
 * Close down the balloc module and free all malloced memory.
 */

static void
bDestroy(pp_mallocator_t* m)
{
    bmallocator_t* a = (bmallocator_t*)m;
    assert(m);
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    verifyBallocSpace();
#endif
    if (!(a->bFlags & PP_B_USER_BUF)) {
	free(a->bFreeBuf);
    }
    free(a);
}

/*
 * Allocate a block of the requested size. First check the block 
 * queues for a suitable one.
 */

static void*
bAlloc(pp_mallocator_t* m, size_t size)
{
    bmallocator_t* a = (bmallocator_t*)m;
    bType * bp;
    int     q, memSize;
    void *  ret = NULL;

    assert(m);
    m->lock(m);

    assert(a->bFreeBuf != NULL);
    assert(size > 0);
    
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    verifyBallocSpace();
#endif
    
    memSize = ballocGetSize(size, &q);

    if (q >= PP_B_MAX_CLASS) {
	/*
	 * Size if bigger than the maximum class. Malloc if use has been okayed
	 */
	if (a->bFlags & PP_B_USE_MALLOC) {
#ifdef PP_B_STATS
	    pp_bstats(m, NULL);
#endif
	    bp = (bType*) malloc(memSize);
#ifdef PP_B_STATS
	    a->bStatsMemMalloc += memSize;
#endif
#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
	    bFillBlock(bp, memSize);
#endif
	} else {
	    bp = NULL;
	    pp_log("bAlloc: block too big (was 0x%zx, max is 0x%x)\n",
		   size, PP_B_MAX_CLASS);
	    abort();
	}

	/*
	 * the u.size is the actual size allocated for data
	 */
	bp->u.size = memSize - sizeof(bType);
	bp->flags = PP_B_MALLOCED;

    } else if ((bp = a->bQhead[q]) != NULL) {
	/*
	 * Take first block off the relevant q if non-empty
	 */
	a->bQhead[q] = bp->u.next;
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
	verifyFreeBlock(bp, q);
#endif
#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
	bFillBlock(bp, memSize);
#endif
	bp->u.size = memSize - sizeof(bType);
	bp->flags = 0;

    } else {
	if (a->bFreeLeft > memSize) {
	    /*
	     * The q was empty, and the free list has spare memory so 
	     * create a new block out of the primary free block
	     */
	    bp = (bType*) a->bFreeNext;
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
	    verifyFreeBlock(bp, q);
#endif
	    a->bFreeNext += memSize;
	    a->bFreeLeft -= memSize;
#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
	    bFillBlock(bp, memSize);
#endif
	    bp->u.size = memSize - sizeof(bType);
	    bp->flags = 0;

	} else if (a->bFlags & PP_B_USE_MALLOC) {
#ifdef PP_B_STATS
	    static int once = 0;
	    if (once++ == 0) {
		pp_bstats(m, NULL);
	    }
#endif
	    /*
	     * Nothing left on the primary free list, so malloc a new block
	     */
	    bp = (bType*) malloc(memSize);
#ifdef PP_B_STATS
	    a->bStatsMemMalloc += memSize;
#endif
#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
	    bFillBlock(bp, memSize);
#endif
	    bp->u.size = memSize - sizeof(bType);
	    bp->flags = PP_B_MALLOCED;
	} else {
	    pp_log("bAlloc: out of memory (%d bytes)\n", a->bFreeSize);
	    abort();
	}
    }

#ifdef PP_B_STATS
    bStatsAlloc(m, 0, bp, q, memSize);
#endif
    bp->flags |= PP_B_INTEGRITY;

    ret = ((char*) bp + sizeof(bType));

    m->unlock(m);
    return ret;
}

/*
 * Free a block back to the relevant free q. We don't free back to the O/S
 * or run time system unless the block is greater than the maximum class size.
 * We also do not coalesce blocks.
 */

static void
bFree(pp_mallocator_t* m, void *mp)
{
    bmallocator_t* a = (bmallocator_t*)m;
    bType	*bp;
    int		q, memSize;

    assert(m);
    if (mp == NULL) return;
    m->lock(m);

#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    verifyBallocSpace();
#endif
    bp = (bType*) ((char*) mp - sizeof(bType));

    assert((bp->flags & PP_B_INTEGRITY_MASK) == PP_B_INTEGRITY);
    
    memSize = ballocGetSize(bp->u.size, &q);

#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    verifyUsedBlock(bp, q);
#endif
#ifdef PP_B_STATS
    bStatsFree(m, 0, bp, q, bp->u.size+sizeof(bType));
#endif
    if (bp->flags & PP_B_MALLOCED) {
	free(bp);
	goto end;
    }
		
#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD
    bFillBlock(bp, memSize);
#endif

    /*
     *	Simply link onto the head of the relevant q
     */
    bp->u.next = a->bQhead[q];
    a->bQhead[q] = bp;

    bp->flags = PP_B_FILL_WORD;
 end:
    m->unlock(m);
}

/*
 * Reallocate a block. Allow NULL pointers and just do a malloc.
 * Note: if the realloc fails, we return NULL and the previous buffer is 
 * preserved.
 */

static void*
bRealloc(pp_mallocator_t* m, void *mp, size_t newsize)
{
    bType	*bp;
    void	*newbuf;

    assert(m);
    if (mp == NULL) {
	return bAlloc(m, newsize);
    }
    bp = (bType*) ((char*) mp - sizeof(bType));
    assert((bp->flags & PP_B_INTEGRITY_MASK) == PP_B_INTEGRITY);

    /*
     * If the allocated memory already has enough room just return the
     * previously allocated address.
     */
    if (bp->u.size >= newsize) {
	return mp;
    }
    newbuf = bAlloc(m, newsize);
    memcpy(newbuf, mp, bp->u.size);
    bFree(m, mp);

    return newbuf;
}

/*
 * Find the size of the block to be balloc'ed.  It takes in a size, finds the 
 * smallest binary block it fits into, adds an overhead amount and returns.
 * q is the binary size used to keep track of block sizes in use.  Called
 * from both balloc and bfree.
 */

static int
ballocGetSize(int size, int *q)
{
    int	mask;

    mask = (size == 0) ? 0 : (size-1) >> PP_B_SHIFT;
    for (*q = 0; mask; mask >>= 1) {
	*q = *q + 1;
    }
    return ((1 << (PP_B_SHIFT + *q)) + sizeof(bType));
}

static void
bLock(pp_mallocator_t* m) {
    bmallocator_t* a = (bmallocator_t*)m;
    MUTEX_LOCK(&a->mtx);
}

static void
bUnlock(pp_mallocator_t* m) {
    bmallocator_t* a = (bmallocator_t*)m;
    MUTEX_UNLOCK(&a->mtx);
}

static void
bMutexInit(pp_mallocator_t* a UNUSED, pp_mutex_t* m, pp_mutex_kind_t k) {
    pp_mutex_init_proclocal(m, k);
}

#if defined(PP_B_FILL) || defined(PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD)
/*
 * Fill the block (useful during development to catch zero fill assumptions)
 */

static void
bFillBlock(void *buf, int bufsize)
{
    memset(buf, PP_B_FILL_CHAR, bufsize);
}
#endif

#ifdef PP_B_STATS
/*
 * Statistics. Do output via calling the writefn callback function with 
 * "handle" as the output file handle. 
 */

void pp_bstats(pp_mallocator_t* m, void (*writefn)(char *fmt, ...))
{
    bmallocator_t*    a = (bmallocator_t*)m;
    bStatsFileType	*fp, *files;
    bStatsBlkType	*blkp;
    bType		*bp;
    char		*cp;
    int			q, count, mem, total, len;
    static int		recurseProtect = 0;

    assert(m);
    
    if (recurseProtect++ > 0) {
	recurseProtect--;
	return;
    }

    if (writefn == NULL) {
	writefn = bstatsWrite;
    }

    /*
     *	Print stats for each memory block
     */
    (*writefn)("\nMemory Stats\n");

    /*
     *	The following tabular format is now used for the output.
     *   Q  Size  Free Bytes Inuse Bytes Allocs
     *	dd ddddd   ddd ddddd  dddd ddddd   dddd
     */
    (*writefn)(" Q  Size   Free  Bytes Inuse Bytes Allocs\n");

    total = 0;
    for (q = 0; q < PP_B_MAX_CLASS; q++) {
	count = 0;
	for (bp = a->bQhead[q]; bp; bp = bp->u.next) {
	    count++;
	}
	mem = count * (1 << (q + PP_B_SHIFT));
	total += mem;
	(*writefn)("%2d %5d   %4d %6d  %4d %5d   %4d\n",
		   q, 1 << (q + PP_B_SHIFT), count, mem, a->bStats[q].inuse, 
		   a->bStats[q].inuse * (1 << (q + PP_B_SHIFT)),
		   a->bStats[q].alloc);
    }

    (*writefn)("\n");

    /*
     *	Print summary stats
     *
     *	bFreeSize		Initial memory reserved with bopen call
     *	bStatsMemMalloc		memory from calls to system MALLOC
     *	bStatsMemMax		
     *	bStatsBallocMax		largest amount of memory from balloc calls
     *	bStatsMemInUse
     *	bStatsBallocInUse	present balloced memory being used
     *	bStatsBlksMax);
     *	bStackStart
     *	bStackMin);
     *	total);
     *	bFreeLeft);
     *
     */
    (*writefn)("Initial free list size    %7d\n", a->bFreeSize);
    (*writefn)("Max memory malloced       %7d\n", a->bStatsMemMalloc);
    (*writefn)("Max memory ever used      %7d\n", a->bStatsMemMax);
    (*writefn)("Max memory ever balloced  %7d\n", a->bStatsBallocMax);
    (*writefn)("Memory currently in use   %7d\n", a->bStatsMemInUse);
    (*writefn)("Memory currently balloced %7d\n", a->bStatsBallocInUse);
    (*writefn)("Max blocks allocated      %7d\n", a->bStatsBlksMax);
    (*writefn)("Maximum stack used        %7d\n",
	       (int) a->bStackStart - (int) a->bStackMin);
    (*writefn)("Free memory on all queues %7d\n", total);
    (*writefn)("Free list buffer left     %7d\n", a->bFreeLeft);
    (*writefn)("Total free memory         %7d\n", a->bFreeLeft + total);

    /*
     *	Print per file allocation stats. Sort the copied table.
     */
    len = sizeof(bStatsFileType) * PP_B_MAX_FILES;
    files = malloc(len);
    memcpy(files, a->bStatsFiles, len);
    qsort(files, a->bStatsFilesMax, sizeof(bStatsFileType), bStatsFileSort);
	
    (*writefn)("\nMemory Currently Allocated\n");
    total = 0;
    (*writefn)("                      bytes, blocks in use, total times, largest,   q\n");

    for (fp = files; fp < &files[a->bStatsFilesMax]; fp++) {
	if (fp->file[0]) {
	    (*writefn)("%18s, %7d,         %5d,      %6d, %7d,%4d\n",
		       fp->file, fp->allocated, fp->count, fp->times, fp->largest, 
		       fp->q);
	    total += fp->allocated;
	}
    }
    (*writefn)("\nTotal allocated %7d\n\n", total);

    /*
     *	Dump the actual strings
     */
    (*writefn)("\nStrings\n");
    for (blkp = &a->bStatsBlks[a->bStatsBlksMax - 1];
	 blkp >= a->bStatsBlks; blkp--) {
	if (blkp->ptr) {
	    cp = (char*) ((char*) blkp->ptr + sizeof(bType));
	    fp = blkp->who;
	    if (isalnum(*cp)) {
		(*writefn)("%-50s allocated by %s\n", cp, 
			   fp->file);
	    }
	}
    }
    free(files);
    recurseProtect--;
}

/*
 * File sort function. Used to sort per file stats
 */

static int
bStatsFileSort(const void *cp1, const void *cp2)
{
    bStatsFileType	*s1, *s2;

    s1 = (bStatsFileType*) cp1;
    s2 = (bStatsFileType*) cp2;

    if (s1->allocated < s2->allocated)
	return -1;
    else if (s1->allocated == s2->allocated)
	return 0;
    return 1;
}

/*
 * Accumulate allocation statistics
 */

static void bStatsAlloc(pp_mallocator_t* m, unsigned long id,
			void *ptr, int q, int size)
{
    bmallocator_t*      a = (bmallocator_t*)m;
    int			memSize;
    bStatsFileType	*fp;
    bStatsBlkType	*bp;
    char		name[FNAMESIZE + 10];

    sprintf(name, "%ld", id);

    a->bStats[q].alloc++;
    a->bStats[q].inuse++;
    a->bStatsMemInUse += size;
    if (a->bStatsMemInUse > a->bStatsMemMax) {
	a->bStatsMemMax = a->bStatsMemInUse;
    }
    memSize = (1 << (PP_B_SHIFT + q)) + sizeof(bType);
    a->bStatsBallocInUse += memSize;
    if (a->bStatsBallocInUse > a->bStatsBallocMax) {
	a->bStatsBallocMax = a->bStatsBallocInUse;
    }

    /*
     *	Track maximum stack usage. Assumes a stack growth down. Approximate as
     *	we only measure this on block allocation.
     */
    if ((void*) &m < a->bStackMin) {
	a->bStackMin = (void*) &m;
    }

    /*
     *	Find the file and adjust the stats for this file
     */
    for (fp = a->bStatsFiles; fp < &a->bStatsFiles[a->bStatsFilesMax]; fp++) {
	if (fp->file[0] == name[0] && strcmp(fp->file, name) == 0) {
	    fp->allocated += size;
	    fp->count++;
	    fp->times++;
	    if (fp->largest < size) {
		fp->largest = size;
		fp->q = q;
	    }
	    break;
	}
    }

    /*
     *	New entry: find the first free slot and create a new entry
     */
    if (fp >= &a->bStatsFiles[a->bStatsFilesMax]) {
	for (fp = a->bStatsFiles; fp < &a->bStatsFiles[PP_B_MAX_FILES]; fp++) {
	    if (fp->file[0] == '\0') {
		strncpy(fp->file, name, sizeof(fp->file));
		fp->allocated += size;
		fp->count++;
		fp->times++;
		fp->largest = size;
		fp->q = q;
		if ((fp - a->bStatsFiles) >= a->bStatsFilesMax) {
		    a->bStatsFilesMax = (fp - a->bStatsFiles) + 1;
		}
		break;
	    }
	}
    }

    /*
     *	Update the per block stats. Allocate a new slot.
     */
    for (bp = a->bStatsBlks; bp < &a->bStatsBlks[PP_B_MAX_BLOCKS]; bp++) {
	if (bp->ptr == NULL) {
	    bp->ptr = ptr;
	    bp->who = fp;
	    if ((bp - a->bStatsBlks) >= a->bStatsBlksMax) {
		a->bStatsBlksMax = (bp - a->bStatsBlks) + 1;
	    }
	    break;
	}
    }
}

/*
 * Free statistics
 */

static void
bStatsFree(pp_mallocator_t* m, unsigned long id, void *ptr, int q, int size)
{
    bmallocator_t*      a = (bmallocator_t*)m;
    int			memSize;
    bStatsFileType	*fp;
    bStatsBlkType	*bp;

    memSize = (1 << (PP_B_SHIFT + q)) + sizeof(bType);
    a->bStatsMemInUse -= size;
    a->bStatsBallocInUse -= memSize;
    a->bStats[q].inuse--;

    /*
     *	Update the per block stats. Try from the end first 
     */
    for (bp = &a->bStatsBlks[a->bStatsBlksMax - 1]; bp >= a->bStatsBlks;
	 bp--) {
	if (bp->ptr == ptr) {
	    bp->ptr = NULL;
	    fp = bp->who;
	    bp->who = NULL;
	    fp->allocated -= size;
	    fp->count--;
	    return;
	}
    }
}

/*
 * Default output function. Just send to trace channel.
 */

#undef sprintf
static void bstatsWrite(char *fmt, ...)
{
    va_list	args;

    va_start(args, fmt);
    vprintf(fmt, args);
    va_end(args);
}

#endif /* PP_B_STATS */

#ifdef PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD

/*
 * The following routines verify the integrity of the balloc memory space.
 * These functions use the PP_B_FILL feature.  Corruption is defined
 * as bad integrity flags in allocated blocks or data other than PP_B_FILL_CHAR
 * being found anywhere in the space which is unallocated and that is not a
 * next pointer in the free queues. a_assert is called if any corruption is
 * found.  CAUTION:  These functions add severe processing overhead and should
 * only be used when searching for a tough corruption problem.
 */

/*
 * verifyUsedBlock verifies that a block which was previously allocated is
 * still uncorrupted.  
 */

static void
verifyUsedBlock(bType *bp, int q)
{
    int		memSize, size;
    char	*p;

    memSize = (1 << (PP_B_SHIFT + q)) + sizeof(bType);
    assert((bp->flags & ~PP_B_MALLOCED) == PP_B_INTEGRITY);
    size = bp->u.size;
    for (p = ((char *)bp)+sizeof(bType)+size; p < ((char*)bp)+memSize; p++) {
	assert(*p == PP_B_FILL_CHAR);
    }
}

/*
 * verifyFreeBlock verifies that a previously free'd block in one of the queues
 * is still uncorrupted.
 */

static void
verifyFreeBlock(bType *bp, int q)
{
    int		memSize;
    char	*p;

    memSize = (1 << (PP_B_SHIFT + q)) + sizeof(bType);
    for (p = ((char *)bp)+sizeof(void*); p < ((char*)bp)+memSize; p++) {
	assert(*p == PP_B_FILL_CHAR);
    }
    bp = (bType *)p;
    assert((bp->flags & ~PP_B_MALLOCED) == PP_B_INTEGRITY ||
	     bp->flags == PP_B_FILL_WORD);
}

/*
 * verifyBallocSpace reads through the entire balloc memory space and
 * verifies that all allocated blocks are uncorrupted and that, with the
 * exception of free list next pointers, all other unallocated space is
 * filled with PP_B_FILL_CHAR.
 */

static void
verifyBallocSpace(pp_mallocator_t* m)
{
    bmallocator_t* a = (bmallocator_t*)m
    int	  	q;
    char	*p;
    bType	*bp;

    /*
     *	First verify all the free blocks.
     */
    for (q = 0; q < PP_B_MAX_CLASS; q++) {	
	for (bp = a->bQhead[q]; bp != NULL; bp = bp->u.next) {
	    verifyFreeBlock(bp, q);
	}
    }

    /*
     *	Now verify other space
     */
    p = a->bFreeBuf;
    while (p < (a->bFreeBuf + a->bFreeSize)) {
	bp = (bType *)p;
	if (bp->u.size > 0xFFFFF) {
	    p += sizeof(bp->u);
	    while (p < (a->bFreeBuf + a->bFreeSize) && *p == PP_B_FILL_CHAR) {
		p++;
	    }
	} else {
	    assert(((bp->flags & ~PP_B_MALLOCED) == PP_B_INTEGRITY) ||
		     bp->flags == PP_B_FILL_WORD);
	    p += (sizeof(bType) + bp->u.size);
	    while (p < (a->bFreeBuf + a->bFreeSize) && *p == PP_B_FILL_CHAR) {
		p++;
	    }
	}
    }
}

#endif /* PP_B_VERIFY_CAUSES_SEVERE_OVERHEAD */
