#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <arpa/inet.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <sys/sem.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <pp/base.h>

#include "base_intern.h"

/*
 * The name of the interface that is used to create the euid of the device.
 * The euid is created from it's mac address. It should not change during
 * device lifetime.
 */
#define EUID_INTERFACE_NAME "eth0"

static struct sembuf semop_lock[2] = {
    {
	sem_num: 0,
	sem_op: 0,
	sem_flg: 0
    },
    {
	sem_num: 0,
	sem_op: 1,
	sem_flg: SEM_UNDO
    }
};

static struct sembuf semop_lock_nb[2] = {
    {
	sem_num: 0,
	sem_op: 0,
	sem_flg: IPC_NOWAIT
    },
    {
	sem_num: 0,
	sem_op: 1,
	sem_flg: SEM_UNDO
    }
};

static struct sembuf semop_lock_nu[2] = {
    {
	sem_num: 0,
	sem_op: 0,
	sem_flg: 0
    },
    {
	sem_num: 0,
	sem_op: 1,
	sem_flg: 0
    }
};

static struct sembuf semop_lock_nb_nu[2] = {
    {
	sem_num: 0,
	sem_op: 0,
	sem_flg: IPC_NOWAIT
    },
    {
	sem_num: 0,
	sem_op: 1,
	sem_flg: 0
    }
};

static struct sembuf semop_unlock[1] = {
    {
	sem_num: 0,
	sem_op: -1,
	sem_flg: (IPC_NOWAIT | SEM_UNDO)
    }
};


static struct sembuf semop_unlock_nu[1] = {
    {
	sem_num: 0,
	sem_op: -1,
	sem_flg: IPC_NOWAIT
    }
};

static struct sembuf semop_unlock_bl[1] = {
    {
	sem_num: 0,
	sem_op: -1,
	sem_flg: 0
    }
};

char * name = NULL;
log_silent_t log_is_silent = 0;
int pp_i_am_eric = 0;
char pp_eui64_str[24];
pp_eui64_t pp_eui64;
static int initialized = 0;
extern int propchange_initialized;

#if defined(PP_FEAT_DEVICE)
int eric_fd = -1;
#endif

extern int propchange_init(void);
extern void propchange_cleanup(void);
static int get_eui64(const char* if_name, pp_eui64_t* buf,
		     char* strbuf, size_t buflen);

int
pp_base_init(const char * prog_name, log_silent_t log_silent)
{
    const char * fn = ___F;
    
    if (!initialized) {
	name = strdup(prog_name);
	log_is_silent = log_silent;

#if defined(PP_FEAT_DEVICE)
	if ((eric_fd = open(ERIC_DEV_PATH, O_RDWR)) == -1) {
	    pp_log_err("%s(): Opening %s failed", fn, ERIC_DEV_PATH);
	    return -1;
	}

        /* i2c and gpio services are lazy initialized,
         * to reduce dependencies from cores/drivers at startup
	 */
#endif /* PP_FEAT_DEVICE */

#if defined(PP_FEAT_PROPCHANGE)
	if (propchange_init() == -1) {
	    pp_log_err("%s(): Could not initialize propchange services", fn);
	    return -1;
	}
#endif /* PP_FEAT_PROPCHANGE */
	if (pp_hrtime_init() == -1) {
	    pp_log_err("%s(): Could not initialize hires timer", fn);
	    return -1;
	}

	if (get_eui64(EUID_INTERFACE_NAME, &pp_eui64, pp_eui64_str,
		      sizeof(pp_eui64_str))) {
	    memset(&pp_eui64, 0, sizeof(pp_eui64));
	    pp_eui64_str[0] = '\0';
	}

	initialized = 1;
    }

    return 0;
}

void
pp_base_cleanup(void)
{
#if defined(PP_FEAT_PROPCHANGE)
    if (propchange_initialized) {
	propchange_cleanup();
    }
#endif /* PP_FEAT_PROPCHANGE */

#if defined(PP_FEAT_DEVICE)
    if (eric_fd != -1) {
	close(eric_fd);
	eric_fd = -1;
    }
    pp_gpio_cleanup();
    pp_i2c_cleanup();
#endif /* PP_FEAT_DEVICE */
    free(name);
    name = NULL;
    initialized = 0;
}

int
pp_check_ip_syntax(const char * ip)
{
    unsigned int p1, p2, p3, p4;
    char dummy[2];

    if (ip == NULL ||
	sscanf(ip, "%u.%u.%u.%u%1[^\n]", &p1, &p2, &p3, &p4, dummy) != 4 ||
	p1 > 255 || p2 > 255 || p3 > 255 || p4 > 255) {
	return -1;
    }
    return 0;
}

int
pp_check_mac_syntax(const char * mac)
{
    unsigned int p1, p2, p3, p4, p5, p6;
    char dummy[2];

    if (mac == NULL ||
	sscanf(mac, "%x:%x:%x:%x:%x:%x%1[^\n]",
	       &p1, &p2, &p3, &p4, &p5, &p6, dummy) != 6 ||
	p1 > 255 || p2 > 255 || p3 > 255 || p4 > 255 || p5 > 255 || p6 > 255) {
	return -1;
    }
    return 0;
}

int
pp_get_mac(const char * if_name, char mac[6])
{
    struct ifreq ifr;
    int fd, ret;

    if (!if_name) return -1;

    if ((fd = socket(AF_INET, SOCK_RAW, htons(ETH_P_ALL))) == -1 ||
	strcpy(ifr.ifr_name, if_name) == NULL ||
	ioctl(fd, SIOCGIFHWADDR, &ifr) == -1) {
	ret = -1;
    } else {
	memcpy(mac, ifr.ifr_hwaddr.sa_data, 6);
	ret = 0;
    }

    if (fd != -1) close(fd);

    return ret;
}

int
pp_get_linklocal_ipv6_address(const char* if_name, char * buf, size_t len)
{
#define IPV6_ADDR_LINKLOCAL 0x20
    char addr6[40], devname[21];
    char addr6p[8][5];
    struct sockaddr_in6 sin6;
    FILE * f;
    int if_idx, plen, scope, dad_status, ret = -1;

    if ((f = fopen("/proc/net/if_inet6", "r")) != NULL) {
        while (fscanf(f, "%4s%4s%4s%4s%4s%4s%4s%4s %02x %02x %02x %02x %20s\n",
                      addr6p[0], addr6p[1], addr6p[2], addr6p[3],
                      addr6p[4], addr6p[5], addr6p[6], addr6p[7],
		      &if_idx, &plen, &scope, &dad_status, devname) != EOF) {
            if (!strcmp(devname, if_name) && scope == IPV6_ADDR_LINKLOCAL) {
                snprintf(addr6, sizeof(addr6), "%s:%s:%s:%s:%s:%s:%s:%s",
			 addr6p[0], addr6p[1], addr6p[2], addr6p[3],
			 addr6p[4], addr6p[5], addr6p[6], addr6p[7]);

		sin6.sin6_family = AF_INET6;
		sin6.sin6_port = 0;
		if (inet_pton(AF_INET6, addr6, sin6.sin6_addr.s6_addr) <= 0) {
		    break;
		}
		if (!inet_ntop(AF_INET6, &sin6.sin6_addr.s6_addr, buf, len)) {
		    break;
		}
		ret = 0;
	    }
	}
    }

    if (f) fclose(f);

    return ret;
}

int
pp_get_linklocal_ipv4_address(const char* if_name, char* buf, size_t len)
{
    int sock = -1, ret = -1;
    size_t i;
    struct ifconf ifc;
    struct ifreq  ifr[10];

    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) >= 0) {
	ifc.ifc_len = 10 * sizeof(struct ifreq);
	ifc.ifc_req = ifr;
	if (ioctl(sock, SIOCGIFCONF, &ifc) == 0) {
	    for (i = 0; i < (ifc.ifc_len / sizeof(struct ifreq)); ++i) {
		if (!strcmp(ifr[i].ifr_name, if_name)) {
		    struct sockaddr_in * sa = (struct sockaddr_in*)&(ifr[i].ifr_addr);
		    if (!inet_ntop(AF_INET, &sa->sin_addr, buf, len)) break;
		    ret = 0;
		    break;
		}
	    }
	}
	close(sock);
    }
    return ret;
}

static int
get_eui64(const char* if_name, pp_eui64_t* ui, char* strbuf, size_t buflen)
{   
    int sock;
    struct ifreq ifr;
    unsigned char* a, *u;

    if (0 > (sock = socket(AF_INET, SOCK_STREAM, 0)))
	return -1;

    strcpy(ifr.ifr_name, if_name);
    if (0 > ioctl(sock, SIOCGIFHWADDR, &ifr))
	return -1;

    a = (char*)ifr.ifr_hwaddr.sa_data;
    u = ui->ui;
    u[0] = a[0]^0x02; u[1] = a[1]; u[2] = a[2]; u[3] = 0xff;
    u[4] = 0xfe     ; u[5] = a[3]; u[6] = a[4]; u[7] = a[5];

    pp_eui64_to_str(ui, strbuf, buflen);
    return 0;
}

void
pp_eui64_to_str(pp_eui64_t* ui, char* strbuf, size_t buflen)
{
    unsigned char* u;
    u = ui->ui;
    snprintf(strbuf, buflen,
	     "%2.2hhx:%2.2hhx:%2.2hhx:%2.2hhx:%2.2hhx:%2.2hhx:%2.2hhx:%2.2hhx",
             u[0], u[1], u[2], u[3], u[4], u[5], u[6], u[7]);
}

unsigned long
pp_strtoul(const char * str, unsigned long deflt, int base, int * error)
{
    unsigned long ret = deflt;
    char * endptr;

    if (str) ret = strtoul(str, &endptr, base);

    if (str == NULL
	|| *str == '\0'
	|| *endptr != '\0') {
	if (error) *error = 1;
	ret = deflt;
    } else {
	if (error) *error = 0;
    }
    return ret;
}

long
pp_strtol(const char * str, long deflt, int base, int * error)
{
    long ret = deflt;
    char * endptr;

    if (str) ret = strtol(str, &endptr, base);

    if (str == NULL
	|| *str == '\0'
	|| *endptr != '\0') {
	if (error) *error = 1;
	ret = deflt;
    } else {
	if (error) *error = 0;
    }
    return ret;
}

unsigned long long
pp_strtoull(const char * str, unsigned long long deflt, int base, int * error)
{
    unsigned long long ret = deflt;
    char * endptr;

    if (str) ret = strtoull(str, &endptr, base);

    if (str == NULL
	|| *str == '\0'
	|| *endptr != '\0') {
	if (error) *error = 1;
	ret = deflt;
    } else {
	if (error) *error = 0;
    }
    return ret;
}

long long
pp_strtoll(const char * str, long long deflt, int base, int * error)
{
    long long ret = deflt;
    char * endptr;

    if (str) ret = strtoll(str, &endptr, base);

    if (str == NULL
	|| *str == '\0'
	|| *endptr != '\0') {
	if (error) *error = 1;
	ret = deflt;
    } else {
	if (error) *error = 0;
    }
    return ret;
}

float
pp_strtof(const char * str, float deflt, int * error)
{
    float ret = deflt;
    char * endptr;

    if (str) ret = strtof(str, &endptr);

    if (str == NULL
	|| *str == '\0'
	|| *endptr != '\0') {
	if (error) *error = 1;
	ret = deflt;
    } else {
	if (error) *error = 0;
    }
    return ret;
}

int
pp_strcmp_safe(const char * s1, const char * s2)
{
    if (s1 != NULL && s2 != NULL) return strcmp(s1, s2);
    return s1 != s2; /* equal if both are NULL */
}

char
pp_bin_to_hex(u_char bin)
{
    const char * lookup_table = "012456789ABCDEF";
    return lookup_table[bin & 0xF];
}

u_char
pp_hex_to_bin(char hex)
{
    return (hex >= 'a' && hex <= 'f' ? hex - 'a' + 10 : hex >= 'A' && hex <= 'F' ? hex - 'A' + 10 : hex - '0');
}

char*
pp_bin_to_hex_string(char *s, const unsigned char *bin, size_t len)
{
    unsigned int i;
    for (i = 0; i < len; i++) snprintf(s + i * 2, 3, "%02X", bin[i]);
    return s;
}

unsigned int pp_bin_to_bcd(unsigned int bin) {
    unsigned int i, d, bcd = 0;

    for (i = 1000000; i > 0; i = i / 10) {
        d = bin / i;
        bcd = (bcd << 4) | d;
        bin = bin - (d * i);
    }
    return bcd;
}

int
pp_sem_get(key_t key)
{
    int sem_id;

    if ((sem_id = semget(key, 1, IPC_CREAT | 0666)) == -1) {
	pp_log_err("%s(): semget()", ___F);
	return -1;
    }

    return sem_id;
}

int 
pp_sem_lock(int sem_id)
{
 again:
    if (semop(sem_id, semop_lock, 2) == -1) {
	if (errno == EINTR) goto again;
	pp_log_err("%s(): semop()", ___F);
	return -1;
    }

    return 0;
}

int 
pp_sem_lock_nb(int sem_id)
{
 again:
    if (semop(sem_id, semop_lock_nb, 2) == -1) {
	if (errno == EINTR) goto again;
	if (errno != EAGAIN) pp_log_err("%s(): semop()", ___F);
	return -1;
    }

    return 0;
}

/**
 * do not unlock semaphore on process exit
 * this is used e.g. in flash_config.sh
 */
int 
pp_sem_lock_nu(int sem_id)
{
 again:
    if (semop(sem_id, semop_lock_nu, 2) == -1) {
	if (errno == EINTR) goto again;
	if (errno != EAGAIN) pp_log_err("%s(): semop()", ___F);
	return -1;
    }

    return 0;
}

/**
 * neither block nor unlock semaphore on process exit
 * this is used e.g. for external changes magic (get_config)
 */
int 
pp_sem_lock_nb_nu(int sem_id)
{
 again:
    if (semop(sem_id, semop_lock_nb_nu, 2) == -1) {
	if (errno == EINTR) goto again;
	if (errno != EAGAIN) pp_log_err("%s(): semop()", ___F);
	return -1;
    }

    return 0;
}

int
pp_sem_unlock(int sem_id)
{
 again:
    if (semop(sem_id, semop_unlock, 1) == -1) {
	if (errno == EINTR) goto again;
        if (errno != EAGAIN) {
            /* if semaphore was not set, this is no error! */
            pp_log_err("%s(): semop(flush_unlock)", ___F);
	    return -1;
        }
    }

    return 0;
}

int
pp_sem_unlock_nu(int sem_id)
{
 again:
    if (semop(sem_id, semop_unlock_nu, 1) == -1) {
	if (errno == EINTR) goto again;
        if (errno != EAGAIN) {
            /* if semaphore was not set, this is no error! */
            pp_log_err("%s(): semop() (%d)", ___F, errno);
	    return -1;
        }
    }

    return 0;
}

int
pp_sem_unlock_bl(int sem_id)
{
 again:
    if (semop(sem_id, semop_unlock_bl, 1) == -1) {
	if (errno == EINTR) goto again;
        if (errno != EAGAIN) {
            /* if semaphore was not set, this is no error! */
            pp_log_err("%s(): semop() (%d)", ___F, errno);
	    return -1;
        }
    }

    return 0;
}
