#include <stdio.h>
#include <unistd.h>
#include <net/if.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <pp/base.h>

#if !defined(PP_FW_TYPE_PRODUCTION)
# include <pp/intl.h>
#endif

typedef unsigned long long u64;	/* hack, so we may include kernel's ethtool.h */
typedef u_int32_t u32;		/* ditto */
typedef u_int16_t u16;		/* ditto */
typedef u_int8_t u8;		/* ditto */

#include <linux/ethtool.h>
#include <linux/sockios.h>


const char * media[] = {
    "10 Mbps, half duplex",
    "10 Mbps, full duplex",
    "100 Mbps, half duplex",
    "100 Mbps, full duplex"
};

char *
pp_eth_get_link_state(const char * ifname)
{
    struct ethtool_cmd ecmd;
    struct ethtool_value edata;
    const char * autoneg;
    const char * speed;
    const char * duplex;
    const char * lnk;
    char buf[128];

    if (PP_FAILED(pp_eth_get_link_state_core(ifname, &ecmd, &edata, NULL))) {
        return NULL;
    }

    autoneg =  ecmd.autoneg == AUTONEG_DISABLE ? _("autonegotiation off") : _("autonegotiation on");
    speed = ecmd.speed == SPEED_10 ? _("10 Mbps") : (ecmd.speed == SPEED_100 ? _("100 Mbps") : _("unknown speed"));
    duplex = ecmd.duplex == DUPLEX_HALF ? _("half duplex") : (ecmd.duplex == DUPLEX_FULL ? _("full duplex") : _("unknown duplex mode"));
    lnk = edata.data ? _("link ok") : _("no link");
    snprintf(buf, sizeof(buf), "%s, %s, %s, %s", autoneg, speed, duplex, lnk);

    return strdup(buf);
}

int pp_eth_get_link_state_core(const char *ifname, ethtool_cmd_t *ecmd, 
                               ethtool_value_t *edata, char **ret_str) {
    static struct ifreq ifr;
    int fd = -1;
    int ret = PP_ERR;

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
	pp_log_err("%s(): socket()", ___F);
	if (ret_str) *ret_str = strdup(_("Error: Cannot open socket."));
	goto bail;
    }

    if(ecmd) {
        ecmd->cmd = ETHTOOL_GSET;
        snprintf(ifr.ifr_name, IFNAMSIZ, ifname);
        ifr.ifr_data = (caddr_t)ecmd;
        if (ioctl(fd, SIOCETHTOOL, &ifr) == -1) {
            pp_log_err("%s(): ioctl(SIOCETHTOOL)", ___F);
            if (ret_str) *ret_str = strdup(_("Error: Cannot get device settings."));
            goto bail;
        }
    }

    if(edata) {
        edata->cmd = ETHTOOL_GLINK;
        ifr.ifr_data = (caddr_t)edata;
        if (ioctl(fd, SIOCETHTOOL, &ifr) == -1) {
            if (ret_str) *ret_str = strdup(_("Error: Cannot get link state."));
            goto bail;
        }
    }

    ret = PP_SUC;

bail:
    if (fd != -1) close(fd);
    return ret;
}

int
pp_eth_have_phy(const char * ifname)
{
    const char * fn = ___F;
    static struct ifreq ifr;
    struct ethtool_cmd ecmd;
    int fd = -1;
    int ret = 0;

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
	pp_log_err("%s(): socket()", fn);
	goto bail;
    }      
    
    ecmd.cmd = ETHTOOL_GSET;
    snprintf(ifr.ifr_name, IFNAMSIZ, ifname);
    ifr.ifr_data = (caddr_t)&ecmd;
    if (ioctl(fd, SIOCETHTOOL, &ifr) == -1) {
	pp_log_err("%s(): ioctl(SIOCETHTOOL)", fn);
	goto bail;
    }

    ret = (ecmd.transceiver == XCVR_EXTERNAL);

 bail:
    if (fd != -1) close(fd);
    return ret;
}

int
pp_eth_set_parameters(const char * ifname, pp_eth_speed_t speed,
		      pp_eth_duplex_t duplex)
{
    const char * fn = ___F;
    static struct ifreq ifr;
    struct ethtool_cmd ecmd;
    int fd = -1;
    int ret = -1;

    if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
	pp_log_err("%s(): socket()", fn);
	goto bail;
    }

    ecmd.cmd = ETHTOOL_GSET;
    snprintf(ifr.ifr_name, IFNAMSIZ, ifname);
    ifr.ifr_data = (caddr_t)&ecmd;
    if (ioctl(fd, SIOCETHTOOL, &ifr) == -1) {
	pp_log_err("%s(): ioctl(SIOCETHTOOL)", fn);
	goto bail;
    }
    
   
    if ((speed != PP_ETH_SPEED_AUTO) && (duplex != PP_ETH_DUPLEX_AUTO)) {
	/* neither speed nor duplex mode are set to auto -> NO autonegotiation */
	ecmd.autoneg = AUTONEG_DISABLE;
	ecmd.advertising = 0;
	ecmd.speed = (speed == PP_ETH_SPEED_10) ? SPEED_10 : SPEED_100;
	ecmd.duplex = (duplex == PP_ETH_DUPLEX_HALF) ? DUPLEX_HALF : DUPLEX_FULL;
	
    } else {
	/* speed or duplex mode are set to auto -> run autonegotiation */
	ecmd.autoneg = AUTONEG_ENABLE;
	ecmd.advertising = ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full
	                 | ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full
	                 | ADVERTISED_Autoneg;

	if (speed == PP_ETH_SPEED_10) {
	    ecmd.advertising &= ~(ADVERTISED_100baseT_Half | ADVERTISED_100baseT_Full);
	}
	if (speed == PP_ETH_SPEED_100) {
	    ecmd.advertising &= ~(ADVERTISED_10baseT_Half | ADVERTISED_10baseT_Full);
	}
	if (duplex == PP_ETH_DUPLEX_HALF) {
	    ecmd.advertising &= ~(ADVERTISED_10baseT_Full | ADVERTISED_100baseT_Full);
	}
	if (duplex == PP_ETH_DUPLEX_FULL) {
	    ecmd.advertising &= ~(ADVERTISED_10baseT_Half | ADVERTISED_100baseT_Half);
	}
    }

    ecmd.cmd = ETHTOOL_SSET;
    snprintf(ifr.ifr_name, IFNAMSIZ, ifname);
    
    ifr.ifr_data = (caddr_t)&ecmd;
    if (ioctl(fd, SIOCETHTOOL, &ifr) == -1) {
	pp_log_err("%s(): ioctl(SIOCETHTOOL)", fn);
	goto bail;
    }
    ret = 0;
 bail:
    if (fd != -1) close(fd);
    return ret;
}
