#include <sys/types.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <pp/hash.h>
#include <pp/log.h>
#include <pp/win32.h>


#define BITSPERBYTE 8
#define BITS(type)  (BITSPERBYTE * (int)sizeof(type))

struct _pp_hash_entry_t {
    unsigned int intname;
    char * name;
    void * content;
    pp_hash_elem_free_func_t free_content_fn;
    struct _pp_hash_entry_t * next;
};

struct _pp_hash_t {
    size_t size;
    unsigned int num_entries;
    pp_hash_entry_t ** table;
    unsigned int next_index;
    pp_hash_entry_t * next_entry;
};

static pp_hash_entry_t * do_hash(pp_hash_t * hash, const char * name);
static int hash_index(pp_hash_t * hash, const char * name);
static unsigned int is_prime(unsigned int n);
static int calc_prime(unsigned int size);
static pp_hash_entry_t * do_hash_i(pp_hash_i_t * hash, const unsigned int name);
static int hash_index_i(pp_hash_i_t * hash, const unsigned int name);

pp_hash_i_t *
pp_hash_create_i(size_t size)
{
    // cast, because pp_hash_t and pp_hash_i_t is the same
    return (pp_hash_i_t *) pp_hash_create(size);
}

void pp_hash_delete_i(pp_hash_i_t * h)
{
    pp_hash_delete((pp_hash_t *) h);
}

void
pp_hash_clear_i(pp_hash_i_t * h)
{
    pp_hash_clear((pp_hash_t *) h);
}

void *
pp_hash_get_entry_i(pp_hash_i_t * h, unsigned int name)
{
    pp_hash_entry_t * entry;

    assert(h);

    for (entry = do_hash_i(h, name); entry != NULL; entry = entry->next) {
      if (name == entry->intname) {
          return entry->content;
      }
    }
    return NULL;
}

int
pp_hash_set_entry_i(pp_hash_i_t * h, unsigned int name,
		    void * content, pp_hash_elem_free_func_t free_content_fn)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;

    assert(h);
    
    hindex = hash_index_i(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (name == entry->intname) {
              if (entry->free_content_fn) {
                  entry->free_content_fn(entry->content);
              }
              entry->content = content;
              entry->free_content_fn = free_content_fn;
              return 0;
	    }
	    prev_entry_next_p = &entry->next;
	}
    }
    
    if ((entry = (pp_hash_entry_t *)malloc(sizeof(pp_hash_entry_t))) == NULL) {
	return -1;
    }
    
    entry->name = NULL;
    entry->intname = name;
    
    entry->content = content;
    entry->free_content_fn = free_content_fn;
    entry->next = NULL;
    
    *prev_entry_next_p = entry;
    
    h->num_entries++;
    
    return 0;
}

void 
pp_hash_rehash_entry_i(pp_hash_i_t * h, unsigned int name,
		       const unsigned int new_name)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;

    assert(h);
    
    hindex = hash_index_i(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (name == entry->intname) {
		if (entry == h->next_entry) {
		    h->next_entry = entry->next;
		}
		*prev_entry_next_p = entry->next;
		pp_hash_set_entry_i(h, new_name, entry->content,
                                           entry->free_content_fn);
		free(entry);
		return;
	    }
	    prev_entry_next_p = &entry->next;
	}
    }
}

void
pp_hash_delete_entry_i(pp_hash_i_t * h, unsigned int name)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;
    
    assert(h);

    hindex = hash_index_i(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (name == entry->intname) {
		if (entry == h->next_entry) {
		    h->next_entry = entry->next;
		}
		*prev_entry_next_p = entry->next;
		if (entry->free_content_fn) {
		    entry->free_content_fn(entry->content);
		}
		free(entry);
		if (h->num_entries > 0) { h->num_entries--; }
		return;
	    }
	    prev_entry_next_p = &entry->next;
	}
    }
}


unsigned int
pp_hash_get_first_key_i(pp_hash_i_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;
    
    assert(h);
    
    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->intname;
	}
    }
    
    h->next_index = h->size;
    h->next_entry = NULL;
    
    return 0;
}

unsigned int
pp_hash_get_next_key_i(pp_hash_i_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;
    
    assert(h);
    
    for (i = h->next_index; i < h->size; i++) {
	if ((entry = h->next_entry) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->intname;
	}
	h->next_index = i + 1;
	h->next_entry = h->table[h->next_index];
    }
    
    return 0;
}

void *
pp_hash_get_first_entry_i(pp_hash_i_t * h)
{
    return pp_hash_get_first_entry((pp_hash_t *)h);
}

void *
pp_hash_get_next_entry_i(pp_hash_i_t * h)
{
    return pp_hash_get_next_entry((pp_hash_t *) h);
}

unsigned int
pp_hash_get_entry_count_i(pp_hash_i_t * h)
{
    assert(h);
    return h->num_entries;
}

static pp_hash_entry_t *
do_hash_i(pp_hash_i_t * h, const unsigned int name)
{
    return h->table[hash_index_i(h, name)];
}

static int
hash_index_i(pp_hash_i_t * h, const unsigned int name)
{
    return name % h->size;
}

pp_hash_t *
pp_hash_create(size_t size)
{
    pp_hash_t * h;

    assert(size > 2);

    if ((h = (pp_hash_t *)malloc(sizeof(pp_hash_t))) == NULL) {
	return NULL;
    }

    memset(h, 0, sizeof(pp_hash_t));

    h->size = calc_prime(size);
    /* we allocate one element more than size - makes life easier */
    h->table = (pp_hash_entry_t**)malloc((h->size+1) * sizeof(pp_hash_entry_t*));
    if (h->table == NULL) {
	free(h);
	return NULL;
    }

    memset(h->table, 0, (h->size + 1) * sizeof(pp_hash_entry_t *));

    h->next_index = h->size;
    h->next_entry = NULL;

    return h;
}

void
pp_hash_delete(pp_hash_t * h)
{
    if (h != NULL) {
	pp_hash_clear(h);
	free(h->table);
	free(h);
    }
}

void
pp_hash_clear(pp_hash_t * h)
{
    unsigned int i;
    pp_hash_entry_t * entry;
    pp_hash_entry_t * next;

    assert(h);

    for (i = 0; i < h->size; i++) {
	for (entry = h->table[i]; entry != NULL; entry = next) {
	    next = entry->next;
	    free(entry->name);
	    if (entry->free_content_fn) {
		entry->free_content_fn(entry->content);
	    }
	    free(entry);
	}
    }

    memset(h->table, 0, (h->size + 1) * sizeof(pp_hash_entry_t *));

    h->num_entries = 0;
    h->next_index = h->size;
    h->next_entry = NULL;
}

void *
pp_hash_get_entry(pp_hash_t * h, const char * name)
{
    pp_hash_entry_t * entry;

    assert(h);
    assert(name);

    for (entry = do_hash(h, name); entry != NULL; entry = entry->next) {
	if (!strcmp(name, entry->name)) {
	    return entry->content;
	}
    }

    return NULL;
}


/*
 * extended hash get method allows the distinction
 * between the following two semantics:
 * 'key is not in hash'
 *     and
 * 'NULL value was stored under key'
 */
int pp_hash_get_entry2(pp_hash_t* h, const char* name, void** value)
{
    pp_hash_entry_t * entry;

    assert(h);
    assert(name);

    for (entry = do_hash(h, name); entry != NULL; entry = entry->next) {
	if (!strcmp(name, entry->name)) {
	    *value = entry->content;
	    return 1;
	}
    }
    return 0;
}

int
pp_hash_set_entry_gently(pp_hash_t * h, int overwrite, const char * name,
			 void * content, pp_hash_elem_free_func_t free_content_fn)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;

    assert(h);
    assert(name);

    hindex = hash_index(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (!strcmp(name, entry->name)) {
		if (overwrite) {
		    if (entry->free_content_fn) {
			entry->free_content_fn(entry->content);
		    }
		    entry->content = content;
		    entry->free_content_fn = free_content_fn;
		    return 0;
		} else {
		    errno = EEXIST;
		    return -1;
		}
	    }
	    prev_entry_next_p = &entry->next;
	}
    }

    if ((entry = (pp_hash_entry_t *)malloc(sizeof(pp_hash_entry_t))) == NULL) {
	return -1;
    }

    if ((entry->name = strdup(name)) == NULL) {
	free(entry);
	return -1;
    }

    entry->content = content;
    entry->free_content_fn = free_content_fn;
    entry->next = NULL;

    *prev_entry_next_p = entry;

    h->num_entries++;

    return 0;
}

void
pp_hash_rehash_entry(pp_hash_t * h, const char * name, const char * new_name)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;

    assert(h);
    assert(name);
    assert(new_name);

    hindex = hash_index(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (!strcmp(name, entry->name)) {
		if (entry == h->next_entry) {
		    h->next_entry = entry->next;
		}
		*prev_entry_next_p = entry->next;
		pp_hash_set_entry(h, new_name, entry->content,
				  entry->free_content_fn);
		free(entry->name);
		free(entry);
		return;
	    }
	    prev_entry_next_p = &entry->next;
	}
    }
}

void
pp_hash_delete_entry(pp_hash_t * h, const char * name)
{
    pp_hash_entry_t ** prev_entry_next_p;
    pp_hash_entry_t * entry;
    unsigned int hindex;

    assert(h);
    assert(name);

    hindex = hash_index(h, name);
    prev_entry_next_p = &h->table[hindex];
    if ((entry = h->table[hindex]) != NULL) {
	for (; entry != NULL; entry = entry->next) {
	    if (!strcmp(name, entry->name)) {
		if (entry == h->next_entry) {
		    h->next_entry = entry->next;
		}
		*prev_entry_next_p = entry->next;
		free(entry->name);
		if (entry->free_content_fn) {
		    entry->free_content_fn(entry->content);
		}
		free(entry);
		if (h->num_entries > 0) { h->num_entries--; }
		return;
	    }
	    prev_entry_next_p = &entry->next;
	}
    }
}

const char *
pp_hash_get_first_key(pp_hash_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->name;
	}
    }

    h->next_index = h->size;
    h->next_entry = NULL;

    return NULL;
}

const char *
pp_hash_get_next_key(pp_hash_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = h->next_index; i < h->size; i++) {
	if ((entry = h->next_entry) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->name;
	}
	h->next_index = i + 1;
	h->next_entry = h->table[h->next_index];
    }

    return NULL;
}

const char *
pp_hash_get_first_key_r(pp_hash_t * h, pp_hash_iterator_t * it)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);
    assert(it);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
	    return entry->name;
	}
    }

    it->next_index = h->size;
    it->next_entry = NULL;

    return NULL;
}

const char *
pp_hash_get_next_key_r(pp_hash_t * h, pp_hash_iterator_t * it)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);
    assert(it);

    for (i = it->next_index; i < h->size; i++) {
	if ((entry = it->next_entry) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
	    return entry->name;
	}
	it->next_index = i + 1;
	it->next_entry = h->table[it->next_index];
    }

    return NULL;
}

void *
pp_hash_get_first_entry(pp_hash_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->content;
	}
    }

    h->next_index = h->size;
    h->next_entry = NULL;

    return NULL;
}

void *
pp_hash_get_next_entry(pp_hash_t * h)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = h->next_index; i < h->size; i++) {
	if ((entry = h->next_entry) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
	    return entry->content;
	}
	h->next_index = i + 1;
	h->next_entry = h->table[h->next_index];
    }

    return NULL;
}

void *
pp_hash_get_first_entry_r(pp_hash_t * h, pp_hash_iterator_t * it)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);
    assert(it);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
	    return entry->content;
	}
    }

    it->next_index = h->size;
    it->next_entry = NULL;

    return NULL;
}

void *
pp_hash_get_next_entry_r(pp_hash_t * h, pp_hash_iterator_t * it)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);
    assert(it);

    for (i = it->next_index; i < h->size; i++) {
	if ((entry = it->next_entry) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
	    return entry->content;
	}
	it->next_index = i + 1;
	it->next_entry = h->table[it->next_index];
    }

    return NULL;
}

unsigned int
pp_hash_get_entry_count(pp_hash_t * h)
{
    assert(h);

    return h->num_entries;
}

void
pp_hash_get_first_key_and_content(pp_hash_t * h, char **key, void **content)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
            *key = entry->name;
            *content = entry->content;
	    return;
	}
    }

    h->next_index = h->size;
    h->next_entry = NULL;

    *key = NULL;
    *content = NULL;
    return;
}

void
pp_hash_get_next_key_and_content(pp_hash_t * h, char **key, void **content)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = h->next_index; i < h->size; i++) {
	if ((entry = h->next_entry) != NULL) {
	    if (entry->next == NULL) {
		h->next_index = i + 1;
		h->next_entry = h->table[h->next_index];
	    } else {
		h->next_index = i;
		h->next_entry = entry->next;
	    }
            *key = entry->name;
            *content = entry->content;
	    return;
	}
	h->next_index = i + 1;
	h->next_entry = h->table[h->next_index];
    }

    *key = NULL;
    *content = NULL;
    return;
}

void
pp_hash_get_first_key_and_content_r(pp_hash_t * h, pp_hash_iterator_t * it,
				    char **key, void **content)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);
    assert(it);

    for (i = 0; i < h->size; i++) {
	if ((entry = h->table[i]) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
            *key = entry->name;
            *content = entry->content;
	    return;
	}
    }

    it->next_index = h->size;
    it->next_entry = NULL;

    *key = NULL;
    *content = NULL;
    return;
}

void
pp_hash_get_next_key_and_content_r(pp_hash_t * h, pp_hash_iterator_t * it,
				   char **key, void **content)
{
    pp_hash_entry_t * entry;
    unsigned int i;

    assert(h);

    for (i = it->next_index; i < h->size; i++) {
	if ((entry = it->next_entry) != NULL) {
	    if (entry->next == NULL) {
		it->next_index = i + 1;
		it->next_entry = h->table[it->next_index];
	    } else {
		it->next_index = i;
		it->next_entry = entry->next;
	    }
            *key = entry->name;
            *content = entry->content;
	    return;
	}
	it->next_index = i + 1;
	it->next_entry = h->table[it->next_index];
    }

    *key = NULL;
    *content = NULL;
    return;
}

/**
 * serializes the given hashtable (containing strings as keys and values!)
 * into the buffer buf, not writing more than buf_size bytes including
 * the terminating 0
 *
 *   - returns number of bytes written to buffer
 *     or -1 if buffer was too small
 */
int
pp_hash_serialize(pp_hash_t * h, char * buf, size_t buf_size)
{
    const char* key;
    ssize_t buf_free = buf_size;
    size_t buf_off = 0;

    assert(buf);

    *buf = '\0';
    for (key = pp_hash_get_first_key(h); key != NULL; key = pp_hash_get_next_key(h)){
	char * value = pp_hash_get_entry(h, key);
	int n = snprintf(buf + buf_off, buf_free, "%s=%s\n", key, value ? value : "");
	if (n >= buf_free) {	    
	    pp_log("%s(): WARNING: not enough space in buffer of size %zd\n", ___F, buf_size);
	    return -1;
	}
	buf_off += n;
	buf_free -= n;
    }
    return buf_off + 1;
}

#ifndef _WIN32
/**
 * deserializes the buffer upto buf_size-1 bytes into the hash table
 *   - all values will be newly allocated
 *   - returns  0 if OK
 *             -1 if some parse error occured
 */
int
pp_hash_deserialize(pp_hash_t * h, char * buf, size_t buf_size)
{
    char * strtok_state;
    char * line = buf;
    u_int line_num;

    assert(buf);

    if (buf_size > 0) buf[buf_size - 1] = '\0'; /* make sure that it's 0-terminated */

    for (line = strtok_r(line, "\n", &strtok_state), line_num = 1;
	 line != NULL && *line != '\0';
	 line = strtok_r(NULL, "\n", &strtok_state), ++line_num) {
	char key[64];
	char value[2048];
	int n = sscanf(line, "%63[^= \t\n]=%2047[^\n]", key, value);
	if (n == 2) {
	    pp_hash_set_entry(h, key, strdup(value), free);
	} else if (n == 1) {
	    pp_hash_set_entry(h, key, NULL, free);
	} else if (n == 0) {
	    pp_log("%s(): parse error in line %u\n", ___F, line_num);
	    return -1;
	}
    }

    return 0;
}
#endif

static pp_hash_entry_t *
do_hash(pp_hash_t * h, const char * name)
{
    return h->table[hash_index(h, name)];
}

static int
hash_index(pp_hash_t * h, const char * name)
{
    unsigned int sum;
    int i;

    /*
     * Add in each character shifted up progressively by 7 bits. The shift
     * amount is rounded so as to not shift too far. It thus cycles with
     * each new cycle placing character shifted up by one bit.
     */
    i = 0;
    sum = 0;
    while (*name) {
	sum += (((int)*name++) << i);
	i = (i + 7) % (BITS(int) - BITSPERBYTE);
    }

    return sum % h->size;
}

static unsigned int
is_prime(unsigned int n)
{
    unsigned int i, max;

    /* assert(n > 0); */

    max = n / 2;
    for (i = 2; i <= max; i++) {
	if (n % i == 0) {
	    return 0;
	}
    }

    return 1;
}

static int
calc_prime(unsigned int size)
{
    unsigned int count;

    /* assert(size > 0); */

    for (count = size; count > 0; count--) {
	if (is_prime(count)) {
	    return count;
	}
    }

    return 1;
}
