/**
 * Copyright 2002 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: access PPC high resolution Timer
 */
#include <pp/base.h>
#include <sys/ioctl.h>

#if defined(__powerpc__) || defined(__arm__)
#include <lara.h>
#endif /* __powerpc__ || __arm__ */

/* nano seconds per clock tick */
static u_int32_t ppc_ns_per_tick;

#ifdef __powerpc__

# define PPC_TBR(u, l) {                          \
    u_int32_t u2;                                \
    asm volatile("1: mftbu %0\n"                 \
                 "   mftb  %1\n"                 \
                 "   mftbu %2\n"                 \
                 "   cmpw  %0, %2\n"             \
                 "   bne   1b\n"                 \
                 : "=r" (u), "=r" (l), "=r" (u2) \
                 );                              \
}

int
pp_hrtime_init()
{
    // get clock speed information
    clock_info_t clock_info;
    if (ioctl(eric_fd, PPIOCGETCLOCKINFO, &clock_info) == -1) {
	pp_log_err("%s(): Could not determine clock speed for PPC", ___F);
	return -1;
    }

    ppc_ns_per_tick = 1000000000 / clock_info.cpu;

    return 0;
}

u_int64_t pp_hrtime() {
    u_int64_t t;
    u_int32_t upper, lower;
    PPC_TBR(upper, lower);
    t = ((u_int64_t)upper << 32) | lower;
    return t;
}

#elif defined(__arm__)

int
pp_hrtime_init()
{
    // get clock speed information
    clock_info_t clock_info;
    if (ioctl(eric_fd, PPIOCGETCLOCKINFO, &clock_info) == -1) {
	pp_log_err("%s(): Could not determine clock speed", ___F);
	return -1;
    }
    ppc_ns_per_tick = 1000000000 / (clock_info.arch.arm.apb); // the timer runs with APB clock
    return 0;
}

u_int64_t pp_hrtime() {
    u_int64_t t;
    if (ioctl(eric_fd, PPIOCGETCLOCKVALUE, &t) == -1) {
	pp_log_err("%s(): Could not determine clock speed", ___F);
    }
    return t;
}

#elif defined(__i386__) || defined(__x86__) || defined(__x86_64__)

int
pp_hrtime_init()
{
    // get clock speed information
    FILE* cpuinfo;
    float mhz = -1.0;
    char tmp[200];
    
    if(NULL == (cpuinfo = fopen("/proc/cpuinfo", "r"))) {
	pp_log_err("%s(): Could not read CPU info", ___F);
	return -1;
    } 
    
    while(!ferror(cpuinfo) && !feof(cpuinfo) && fgets(tmp, 200, cpuinfo)) {
        if(sscanf(tmp, "cpu MHz         : %f", &mhz)) {
            break;
        }
    }
    fclose(cpuinfo);
    
    if(mhz < 0) {
	pp_log_err("%s(): CPU info invalid", ___F);
        return -1;
    }

    ppc_ns_per_tick = 1000 / mhz;

    return 0;
}

u_int64_t pp_hrtime() {
    u_int64_t t;
    
    asm volatile("rdtsc" : "=A" (t)); // IA32
//    asm volatile("mov %0=ar.itc" : "=r"(t) :: "memory"); // IA64

    return t;
}

#else /* !__powerpc__ && !__arm__ && !__i386__ && !__x86__ && !__x86_64__ */

#eror no high resolution timer implementation

#endif


u_int64_t pp_hrtime_ns() {
    return pp_hrtime() * ppc_ns_per_tick;
}

u_int64_t pp_hrtime_us() {
    return (pp_hrtime() / 1000) * ppc_ns_per_tick;
}

u_int64_t pp_hrtick_to_us(u_int64_t t1) {
    return (t1 / 1000) * ppc_ns_per_tick;
}

u_int64_t pp_hrtimediff_ns(u_int64_t t1, u_int64_t t2) {
    u_int64_t tdiff;
    if(t1 > t2) tdiff = t1 - t2;
    else tdiff = t2 - t1;
    return tdiff * ppc_ns_per_tick;
}

void
pp_hrdelay_us (u_int64_t delay)
{
    u_int64_t wut = pp_hrtime_us() + delay;
    while (pp_hrtime_us() < wut);
}


void pp_hrtime_start(pp_stopwatch_t* watch) {
    watch->t_start = pp_hrtime();
}

void pp_hrtime_stop(pp_stopwatch_t* watch) {
    watch->t_stop = pp_hrtime();
}

u_int64_t pp_hrtime_read(pp_stopwatch_t* watch) {
    return pp_hrtimediff_ns(watch->t_stop, watch->t_start);
}

