#ifndef _PP_BASE_H
#define _PP_BASE_H

#include <assert.h>

#include <byteswap.h>
#include <endian.h>
#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <netinet/in.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>

#include <pp/features.h>
#include <pp/base64.h>
#include <pp/log.h>
#include <pp/hash.h>
#include <pp/syms.h>
#include <pp/strstream.h>
#include <pp/error.h>

#ifdef PP_FW_TYPE_ERLA
/* defined in lara_kernel_modules  */
# include <pp_byte_order.h>
#endif

#if defined(PP_FEAT_DEVICE) && defined(PP_FEAT_PROPCHANGE)
# include <pp/propchange.h>
#endif /* PP_FEAT_DEVICE && PP_FEAT_PROPCHANGE */

#ifdef DMALLOC
# include <dmalloc.h>
#endif

/* production hack: avoid libpp_intl dependency */
#if defined(PP_FW_TYPE_PRODUCTION)
#define N_(str) (str)
#define _(str) (str)
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if defined(PP_FEAT_DEVICE)
# define ERIC_DEV_PATH		"/dev/eric0"
# define ERIC_PROGRAM_PATH	"/bin/eric"
# define ERIC_PIDFILE		"/var/run/eric.pid"
#endif /* PP_FEAT_DEVICE */

#if defined(PP_FEAT_LOCAL_VIDEO_X_CONSOLE)
/* FIXME: need to change evilwm ;-) */
# define PP_WM_PROGRAM_PATH	"/bin/evilwm"
# define PP_WM_PIDFILE  "/var/run/wm.pid"
# define SIGSWITCHPORT 42
#endif /* PP_FEAT_LOCAL_VIDEO_X_CONSOLE */

#define min(x,y) ({ \
        const typeof(x) _x = (x);       \
        const typeof(y) _y = (y);       \
        (void) (&_x == &_y);            \
        _x < _y ? _x : _y; })
  
#define max(x,y) ({ \
        const typeof(x) _x = (x);       \
        const typeof(y) _y = (y);       \
        (void) (&_x == &_y);            \
        _x > _y ? _x : _y; })


	/* PP_ASSERT: use PP_ASSERT(variable, condition), this prevents that the variable is unused when compiled without debug */
#define PP_ASSERT(x,y) assert(y); (void)x

	/*
 * endianess/swapping macros
 */

#define bswap_constant_16(x) \
     ((((x) >> 8) & 0xff) | (((x) & 0xff) << 8))

#define bswap_constant_32(x) \
     ((((x) & 0xff000000) >> 24) | (((x) & 0x00ff0000) >>  8) |               \
      (((x) & 0x0000ff00) <<  8) | (((x) & 0x000000ff) << 24))

#define bswap_constant_24(x) \
    ((((x) & 0xff0000) >> 16) | ((x) & 0x00ff00) | (((x) & 0x0000ff) << 16))
static __inline__ unsigned int bswap_24(unsigned int x)
    { return bswap_constant_24(x); }

#if __BYTE_ORDER == __BIG_ENDIAN
#  define le16_to_cpu(x) bswap_16(x)
#  define le32_to_cpu(x) bswap_32(x)
#  define be16_to_cpu(x) (x)
#  define be32_to_cpu(x) (x)
#  define cpu_to_le16(x) bswap_16(x)
#  define cpu_to_le32(x) bswap_32(x)
#  define cpu_to_be16(x) (x)
#  define cpu_to_be32(x) (x)
#  define le16_to_cpu_const(x) bswap_constant_16(x)
#  define le32_to_cpu_const(x) bswap_constant_32(x)
#  define be16_to_cpu_const(x) (x)
#  define be32_to_cpu_const(x) (x)
#  define cpu_to_le16_const(x) bswap_constant_16(x)
#  define cpu_to_le32_const(x) bswap_constant_32(x)
#  define cpu_to_be16_const(x) (x)
#  define cpu_to_be32_const(x) (x)
#  define le24_to_cpu(x) bswap_24(x)
#  define be24_to_cpu(x) (x)
#  define cpu_to_le24(x) bswap_24(x)
#  define cpu_to_be24(x) (x)
#  define le24_to_cpu_const(x) bswap_constant_24(x)
#  define be24_to_cpu_const(x) (x)
#  define cpu_to_le24_const(x) bswap_constant_24(x)
#  define cpu_to_be24_const(x) (x)
#elif __BYTE_ORDER == __LITTLE_ENDIAN
#  define le16_to_cpu(x) (x)
#  define le32_to_cpu(x) (x)
#  define be16_to_cpu(x) bswap_16(x)
#  define be32_to_cpu(x) bswap_32(x)
#  define cpu_to_le16(x) (x)
#  define cpu_to_le32(x) (x)
#  define cpu_to_be16(x) bswap_16(x)
#  define cpu_to_be32(x) bswap_32(x)
#  define le16_to_cpu_const(x) (x)
#  define le32_to_cpu_const(x) (x)
#  define be16_to_cpu_const(x) bswap_constant_16(x)
#  define be32_to_cpu_const(x) bswap_constant_32(x)
#  define cpu_to_le16_const(x) (x)
#  define cpu_to_le32_const(x) (x)
#  define cpu_to_be16_const(x) bswap_constant_16(x)
#  define cpu_to_be32_const(x) bswap_constant_32(x)
#  define le24_to_cpu(x) (x)
#  define be24_to_cpu(x) bswap_24(x)
#  define cpu_to_le24(x) (x)
#  define cpu_to_be24(x) bswap_24(x)
#  define le24_to_cpu_const(x) (x)
#  define be24_to_cpu_const(x) bswap_constant_24(x)
#  define cpu_to_le24_const(x) (x)
#  define cpu_to_be24_const(x) bswap_constant_24(x)
#endif

   
/*
 * MUTEX_LOCK is a wrapper around pthread_mutex_lock().
 */

#define MUTEX_LOCK(mutex)                                        \
    {                                                            \
        register int mutex_error;                                \
        mutex_error = pthread_mutex_lock(mutex);		 \
	if (mutex_error) { assert(mutex_error == 0); exit(1); }	 \
    }

/*
 * MUTEX_UNLOCK is a wrapper around pthread_mutex_unlock().
 */

#define MUTEX_UNLOCK(mutex)                                      \
    {                                                            \
        register int mutex_error;                                \
        mutex_error = pthread_mutex_unlock(mutex);		 \
	if (mutex_error) { assert(mutex_error == 0); exit(1); }	 \
    }

#define RWLOCK_RDLOCK(rwlock)					  \
    {								  \
        register int rwlock_error;                                \
        rwlock_error = pthread_rwlock_rdlock(rwlock);		  \
	if (rwlock_error) { assert(rwlock_error == 0); exit(1); } \
    }

#define RWLOCK_WRLOCK(rwlock)					  \
    {								  \
        register int rwlock_error;                                \
        rwlock_error = pthread_rwlock_wrlock(rwlock);		  \
	if (rwlock_error) { assert(rwlock_error == 0); exit(1); } \
    }

#define RWLOCK_UNLOCK(rwlock)					  \
    {								  \
        register int rwlock_error;                                \
        rwlock_error = pthread_rwlock_unlock(rwlock);		  \
	if (rwlock_error) { assert(rwlock_error == 0); exit(1); } \
    }


#define MUTEX_CREATE_ERRORCHECKING(mutex)                            \
    {                                                                \
        int mutex_error;                                             \
        pthread_mutexattr_t err_check_mtx_attr;			     \
        mutex_error = pthread_mutexattr_init(&err_check_mtx_attr);   \
        assert(mutex_error == 0);                                    \
        mutex_error = pthread_mutexattr_settype(&err_check_mtx_attr, \
					PTHREAD_MUTEX_ERRORCHECK_NP);\
        assert(mutex_error == 0);                                    \
        pthread_mutex_init((mutex), &err_check_mtx_attr);	     \
    }

#define MUTEX_CREATE_RECURSIVE(mutex)                            \
    {                                                                \
        int mutex_error;                                             \
        pthread_mutexattr_t err_check_mtx_attr;			     \
        mutex_error = pthread_mutexattr_init(&err_check_mtx_attr);   \
        assert(mutex_error == 0);                                    \
        mutex_error = pthread_mutexattr_settype(&err_check_mtx_attr, \
					 PTHREAD_MUTEX_RECURSIVE_NP);\
        assert(mutex_error == 0);                                    \
        pthread_mutex_init((mutex), &err_check_mtx_attr);	     \
    }
    
/*
 * macro parameter stringification
 * (with or without macro expansion first).
 */

#define PP_STRINGIFY_EXPANDED(param) PP_STRINGIFY_UNEXPANDED(param)
#define PP_STRINGIFY_UNEXPANDED(param) #param

/**
 * pp_buf_ptr_t can be used to iterate over memory and storing
 * various scalar data of different types. This avoids casted
 * lvalues that are not allowed anymore with gcc4.
 *
 * Example:	char * buf = ...;
 *		pp_buf_ptr_t buf_ptr = buf;
 *		buf_ptr.bp_int++ = 6;
 *		buf_ptr.bp_char++ = 'k';
 */

typedef union {
    void * bp_void;
    char * bp_char;
    short * bp_short;
    int * bp_int;
    long * bp_long;
    size_t * bp_size_t;
} pp_buf_ptr_t;

typedef union {
    const void * bp_void;
    const char * bp_char;
    const short * bp_short;
    const int * bp_int;
    const long * bp_long;
    const size_t * bp_size_t;
} pp_buf_ptr_const_t;

/* ---- base.c ------------------------------------------------------------- */

/*
 * Extended unique identfier
 */
typedef struct {
    u_int8_t ui[8];
} pp_eui64_t;

typedef enum {
    PP_FALSE       = 0,
    PP_TRUE        = 1
} pp_bool_t;

#if defined(PP_FEAT_DEVICE)
extern int eric_fd;
#endif

#define pp_strtoul_10(str, deflt, error) pp_strtoul(str, deflt, 10, error)
#define pp_strtol_10(str, deflt, error) pp_strtol(str, deflt, 10, error)
#define pp_strtoull_10(str, deflt, error) pp_strtoull(str, deflt, 10, error)
#define pp_strtoll_10(str, deflt, error) pp_strtoll(str, deflt, 10, error)

extern int pp_i_am_eric;
extern char pp_eui64_str[24];
extern pp_eui64_t pp_eui64;

extern int pp_base_init(const char * prog_name, log_silent_t log_silent);
extern void pp_base_cleanup(void);
    
extern int pp_check_ip_syntax(const char * ip);
extern int pp_check_mac_syntax(const char * mac);
extern int pp_get_mac(const char * if_name, char mac[18]);
extern int pp_get_linklocal_ipv6_address(const char* if_name, char * buf, size_t len);
extern int pp_get_linklocal_ipv4_address(const char* if_name, char* buf, size_t len);
extern void pp_eui64_to_str(pp_eui64_t* ui, char* strbuf, size_t buflen);
extern unsigned long pp_strtoul(const char * str, unsigned long deflt, int base, int * error);
extern long pp_strtol(const char * str, long deflt, int base, int * error);
extern unsigned long long pp_strtoull(const char * str, unsigned long long deflt, int base, int * error);
extern long long pp_strtoll(const char * str, long long deflt, int base, int * error);
extern float pp_strtof(const char * str, float deflt, int * error);
extern int pp_strcmp_safe(const char * s1, const char * s2);
static inline char * pp_strdup_safe(const char *s) {
    return s ? strdup(s) : NULL;
}
#ifdef _GNU_SOURCE
static inline char * pp_strndup_safe(const char *s, size_t n) {
    return s ? strndup(s, n) : NULL;
}
#endif /* _GNU_SOURCE */
extern char pp_bin_to_hex(u_char bin);
extern u_char pp_hex_to_bin(char hex);
extern char* pp_bin_to_hex_string(char *s, const unsigned char *bin, size_t len);
extern unsigned int pp_bin_to_bcd(unsigned int bin);
int pp_sem_get(key_t key);
int pp_sem_lock(int sem_id);
int pp_sem_lock_nu(int sem_id);
int pp_sem_lock_nb(int sem_id);
int pp_sem_lock_nb_nu(int sem_id);
int pp_sem_unlock(int sem_id);
int pp_sem_unlock_nu(int sem_id);
int pp_sem_unlock_bl(int sem_id);

/* ---- queue.c ------------------------------------------------------------ */

typedef enum {
    QUEUE_TYPE_STANDARD,
    QUEUE_TYPE_PRIORITY,
    QUEUE_TYPE_PRIORITY_REVERSE /* less priority first */
} pp_queue_type_t;

typedef struct _pp_queue_entry_t {
    void * data;
    size_t len;
    unsigned int priority;
    struct _pp_queue_entry_t * prev;
} pp_queue_entry_t;

typedef struct _pp_queue_t pp_queue_t;

extern pp_queue_t * pp_queue_alloc(pp_queue_type_t queue_type);
extern void pp_queue_free(pp_queue_t * queue);
extern int pp_queue_enqueue(pp_queue_t * queue, void * data, size_t data_len,
			    unsigned int priority);
extern pp_queue_entry_t * pp_queue_dequeue(pp_queue_t * queue);
/* not yet implemented?!
extern pp_queue_entry_t * pp_queue_next(pp_queue_t * queue,
					pp_queue_entry_t * entry);
*/
extern pp_queue_entry_t * pp_queue_first(pp_queue_t * queue);
/**
 * dequeues first entry in queue with assigned priority
 * returns NULL if no entry with that priority can be found
 */
pp_queue_entry_t * pp_queue_get(pp_queue_t * queue, unsigned int priority);

/* ---- lists -------------------------------------------------------------- */

/*
 * Simple doubly linked list implementation.
 *
 * Some of the internal functions ("__xxx") are useful when
 * manipulating whole lists rather than single entries, as
 * sometimes we already know the next/prev entries and we can
 * generate better code by using them directly rather than
 * using the generic single-entry routines.
 */

struct list_head {
    struct list_head *next, *prev;
};

#define LIST_HEAD_INIT(name) { &(name), &(name) }

#define LIST_HEAD(name) \
	struct list_head name = LIST_HEAD_INIT(name)

#define INIT_LIST_HEAD(ptr) do { \
	(ptr)->next = (ptr); (ptr)->prev = (ptr); \
} while (0)

/*
 * Insert a new entry between two known consecutive entries. 
 *
 * This is only for internal list manipulation where we know
 * the prev/next entries already!
 */
static __inline__ void __list_add(struct list_head * _new,
				  struct list_head * prev,
				  struct list_head * next)
{
    next->prev = _new;
    _new->next = next;
    _new->prev = prev;
    prev->next = _new;
}

/**
 * list_add - add a new entry
 * @new: new entry to be added
 * @head: list head to add it after
 *
 * Insert a new entry after the specified head.
 * This is good for implementing stacks.
 */
static __inline__ void list_add(struct list_head *_new, struct list_head *head)
{
    __list_add(_new, head, head->next);
}

/**
 * list_add_tail - add a new entry
 * @new: new entry to be added
 * @head: list head to add it before
 *
 * Insert a new entry before the specified head.
 * This is useful for implementing queues.
 */
static __inline__ void list_add_tail(struct list_head *_new,
				     struct list_head *head)
{
    __list_add(_new, head->prev, head);
}

/*
 * Delete a list entry by making the prev/next entries
 * point to each other.
 *
 * This is only for internal list manipulation where we know
 * the prev/next entries already!
 */
static __inline__ void __list_del(struct list_head * prev,
				  struct list_head * next)
{
    next->prev = prev;
    prev->next = next;
}

/**
 * list_del - deletes entry from list.
 * @entry: the element to delete from the list.
 * Note: list_empty on entry does not return true after this, the entry is in
 *       an undefined state.
 */
static __inline__ void list_del(struct list_head *entry)
{
    __list_del(entry->prev, entry->next);
}

/**
 * list_del_init - deletes entry from list and reinitialize it.
 * @entry: the element to delete from the list.
 */
static __inline__ void list_del_init(struct list_head *entry)
{
    __list_del(entry->prev, entry->next);
    INIT_LIST_HEAD(entry); 
}

/**
 * list_empty - tests whether a list is empty
 * @head: the list to test.
 */
static __inline__ int list_empty(struct list_head *head)
{
    return head->next == head;
}

/**
 * list_splice - join two lists
 * @list: the new list to add.
 * @head: the place to add it in the first list.
 */
static __inline__ void list_splice(struct list_head *list,
				   struct list_head *head)
{
    struct list_head *first = list->next;

    if (first != list) {
	struct list_head *last = list->prev;
	struct list_head *at = head->next;

	first->prev = head;
	head->next = first;

	last->next = at;
	at->prev = last;
    }
}

/**
 * list_swap - swaps the place of two list entries.
 *             can also be used to swaps the content of entrie lists by using on list heads.
 * @list1: first list (head).
 * @list2: second list (head).
 */
static __inline__ void list_swap(struct list_head *item1,
				 struct list_head *item2)
{
    struct list_head *prev2, *next2;

    if (list_empty(item1)) next2 = prev2 = item2;
    else {
        item1->prev->next = item1->next->prev = item2;
        prev2 = item1->prev;
        next2 = item1->next;
    }

    if (list_empty(item2)) item1->next = item1->prev = item1;
    else {
        item2->prev->next = item2->next->prev = item1;
        item1->prev = item2->prev;
        item1->next = item2->next;
    }

    item2->prev = prev2;
    item2->next = next2;
}

/**
 * list_entry - get the struct for this entry
 * @ptr:	the &struct list_head pointer.
 * @type:	the type of the struct this is embedded in.
 * @member:	the name of the list_struct within the struct.
 */
#define list_entry(ptr, type, member) \
	((type *)((char *)(ptr)-(unsigned long)(&((type *)0)->member)))

/**
 * list_for_each	-	iterate over a list
 * @pos:	the &struct list_head to use as a loop counter.
 * @head:	the head for your list.
 */
#define list_for_each(pos, head) \
	for (pos = (head)->next; pos != (head); pos = pos->next)

/**
 * list_for_each_safe   -       iterate over a list safe against removal of list entry
 * @pos:        the &struct list_head to use as a loop counter.
 * @n:          another &struct list_head to use as temporary storage
 * @head:       the head for your list.
 */
#define list_for_each_safe(pos, n, head) \
        for (pos = (head)->next, n = pos->next; pos != (head); \
                pos = n, n = pos->next)

/* ---- properties --------------------------------------------------------- */

/* Properties are like config entries, i.e name value pairs
 * however they are not stored in a file, but in a hashtable
 * only. Properties may be serialized into a flat buffer and
 * deserialized from such a buffer.
 */

typedef struct {
    pp_hash_t * nvs;
    struct list_head listeners;
    int         myown;
} pp_prop_t;

typedef struct {
    void (*added)(pp_prop_t* props, const char* name);
    void (*changed)(pp_prop_t* props, const char* name);
    void (*removed)(pp_prop_t* props, const char* name);
} pp_prop_listener_t;

#define PP_PROP_DELI '='
#define PP_PROP_COMMENT '#'
extern pp_prop_t * pp_prop_new(pp_prop_t* props, size_t initsize);
extern void pp_prop_delete(pp_prop_t* props);
extern int pp_prop_read(pp_prop_t* props, void* buf, size_t n);
extern int pp_prop_write(pp_prop_t* props, void* buf, size_t n);
extern int pp_prop_set(pp_prop_t* props, const char* name, char* value);
extern char* pp_prop_get(pp_prop_t* props, const char* name);
extern int pp_prop_add_listener(pp_prop_t* props,
                                pp_prop_listener_t* listener, const char* prop);
extern int pp_prop_remove_listener(pp_prop_t* props,
				   pp_prop_listener_t* listener);

/* ---- misc.c ------------------------------------------------------------- */

typedef struct {
    int pid;
    FILE* stdout; /* parent view */
    FILE* stdin;  /* parent view */
} pp_popen_rw_t;

extern int pp_popen_rw(const char* command, pp_popen_rw_t * data);
extern int pp_pclose_rw(pp_popen_rw_t * data);

extern void pp_free_dynamic_array(char ** array, ssize_t num_entries,
				  void(*free_elem_func)(void *));
extern int pp_check_and_alloc_mem(void **mem_p, size_t size,
				  unsigned int *count, unsigned int new_count);
extern void * pp_mem_search(const void * data, size_t data_size,
			    const void * pattern, size_t pattern_size);
extern char * pp_trim_string(char * s);
char * pp_strtolower(char * s);
char * pp_strtoupper(char * s);
extern void pp_md5_hash_to_str(const char * hash, char * hash_str);
extern int pp_system(const char *command);
int pp_systeme(const char* command, char *const envp[]);
int pp_systemf(const char* command, ...);
extern void pp_set_hardware_clock_exact(const time_t settime, const struct timeval ref_time);
extern int64_t timeval_diff_usec(struct timeval * newt, struct timeval * old);
struct timespec pp_get_wakeup_time(unsigned int mstime /*ms*/);

static inline int64_t timeval_diff_msec(struct timeval* newtv,
					struct timeval* oldtv) {
    return timeval_diff_usec(newtv, oldtv) / 1000;
}
static inline int timeval_now(struct timeval* now) {
    return gettimeofday(now, NULL);
}


/* ---- alloc.c ------------------------------------------------------------ */

extern void * pp_mmap_malloc(size_t size);
extern void   pp_mmap_free(void * ptr);
extern void * pp_mmap_realloc(void * ptr, size_t size);

/* ---- ethtool.c ---------------------------------------------------------- */

typedef enum {
    PP_ETH_SPEED_AUTO,
    PP_ETH_SPEED_10,
    PP_ETH_SPEED_100
} pp_eth_speed_t;

typedef enum {
    PP_ETH_DUPLEX_AUTO,
    PP_ETH_DUPLEX_HALF,
    PP_ETH_DUPLEX_FULL
} pp_eth_duplex_t;

typedef struct ethtool_cmd ethtool_cmd_t;
typedef struct ethtool_value ethtool_value_t;
int pp_eth_get_link_state_core(const char *ifname, ethtool_cmd_t *ecmd, 
                               ethtool_value_t *edata, char **ret_str);
extern char * pp_eth_get_link_state(const char * ifname);
extern int pp_eth_have_phy(const char * ifname);
extern int pp_eth_set_parameters(const char * ifname,
				 pp_eth_speed_t speed,
				 pp_eth_duplex_t duplex);

/* ---- access high resolution timer ---------------------------------------- */
int pp_hrtime_init(void); // init timer (get clockspeed), done in base init
u_int64_t pp_hrtime(void);    // count clockticks
u_int64_t pp_hrtime_ns(void); // count nano seconds
u_int64_t pp_hrtime_us(void); // count micro seconds
u_int64_t pp_hrtimediff_ns(u_int64_t t1, u_int64_t t2); // for ticks
u_int64_t pp_hrtick_to_us(u_int64_t t1);

typedef struct pp_stopwatch_s {
    u_int64_t t_start;
    u_int64_t t_stop;
} pp_stopwatch_t;

/* Busy waiting for n microsecs -- keeps timeslice active, hence finer
 * granularity than usleep. */
void pp_hrdelay_us (u_int64_t delay);

void pp_hrtime_start(pp_stopwatch_t* watch);
void pp_hrtime_stop(pp_stopwatch_t* watch);
u_int64_t pp_hrtime_read(pp_stopwatch_t* watch);

/* ---- stringtool.c ----------------------------------------------------- */
char* pp_str_escape(char esc_char, const char* plain_list, const char* esc_list, const char* s);
char* pp_str_unescape(char esc_char, const char* plain_list, const char* esc_list, const char* s);

/* -------------- change notification ---------------------------------- */

int pp_cfgmsg_update(void);

/* -------------- singleton  ------------------------------------------- */

typedef enum {
    SINGLETON_OK,
    SINGLETON_ALREADY_RUNNING,
    SINGLETON_UNCRITICAL_EXCEPTION,
    SINGLETON_CRITICAL_EXCEPTION,
} singleton_ret_code_t;

singleton_ret_code_t pp_base_enforce_singleton(const char * pidfile);
singleton_ret_code_t pp_base_release_singleton(const char * pidfile);
int pp_base_get_pid_of_program_by_pidfile(const char * pidfile, const char * program_path);

/* ------------ gpio, for dev numbers see pp_gpio.h --------------------- */
/*
 * Note 1:
 * existing GPIO devices can be looked up in pp_gpio.h, currently there are
 * PP_GPIO_DEV_IBM
 * PP_GPIO_DEV_FARADAY
 * PP_GPIO_DEV_OPENCORES_0
 * PP_GPIO_DEV_OPENCORES_1
 * PP_GPIO_DEV_XR17_0
 * PP_GPIO_DEV_XR17_1
 * 
 * Note 2:
 * GPIO bit positions and GPIO masks can be calculated
 * with the following convenience macros in pp_gpio.h:
 * PP_GPIO_BIT(num)
 * PP_IBM_GPIO_BIT(num)
 * PP_GPIO_MASK(num)
 * PP_IBM_GPIO_MASK(num)
 */

/**
 * Get the value of the gpio device with the specified number.
 * Single gpio signals are mapped to bits in the return value.
 */
u_long pp_gpio_get(u_char dev_no);

/**
 * Change the value of the specified gpio device. Single bits can
 * either be set, cleared or tri-stated by adding them to the
 * corresponding bitmask 'set', 'clr' or 'tri'.
 */
void   pp_gpio_set(u_char dev_no, u_long set, u_long clr, u_long tri);


/**
 * Get the value of one signal in the specified gpio device.
 */
u_char pp_gpio_bit_get(u_char dev_no, u_char bit);

/**
 * Set the value of one signal in the specified gpio device.
 */
void   pp_gpio_bit_set(u_char dev_no, u_char bit, u_char value);

/**
 * Configure one GPIO signal as output (value=1) or input (value=0).
 */
void   pp_gpio_bit_set_enable(u_char dev_no, u_char bit, u_char value);


/**
 * Configure the GPIO alternate usage. Applies to some devices
 * only. Don't use these calls until you know what you do.
 */
u_long pp_gpio_get_alternate(u_char dev_no);

/**
 * Configure the GPIO alternate usage. Applies to some devices
 * only. Don't use these calls until you know what you do.
 */
void   pp_gpio_set_alternate(u_char dev_no, u_long set, u_long clr);

/**
 *  open and return the fd of a gpio device, for advanced ioctl usage
 */
int pp_gpio_get_fd(u_char dev_no);

/* ---- features.c ----------------------------------------------------- */
int pp_feature_supported(const char *feature);

#ifdef __cplusplus
}
#endif

#endif /* _PP_BASE_H */
