/*
 * mutex abstration
 * that may work process local (a simple pthread mutex)
 * or may work shared among many processe (a System V semaphore)
 *
 * Currently this is NOT a universally usable mutex since
 * it does need to know the memory allocator that is used to
 * allocate the memory which eventually gets protected by the mutex.
 * This is rather unimportant for heap memory but not for shared memory.
 *
 * So if you are looking for a mutex that can be shared between different
 * processes but you don't have an associated shared memory, this class
 * won't help, unless you extend it, what would be a great idea in this
 * case!!
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#ifndef _PP_MUTEX_H
#define _PP_MUTEX_H

#include <sys/types.h>

/* class of MUTEX */
typedef enum {
    PP_MUTEX_PROC_LOCAL = 0,             /* used for normal and recursive */
    PP_MUTEX_PROC_SHARED = 1,
    PP_MUTEX_PROC_SHARED_RECURSIVE = 2
} pp_mutex_type_t;

typedef enum {
    PP_MUTEX_NORMAL,
    PP_MUTEX_RECURSIVE
} pp_mutex_kind_t;

typedef struct {
    key_t shmkey; 
    int   semid;
} pp_sem_t;

typedef struct {
    key_t     shmkey; 
    int       semid;
    pid_t     pid;
    pthread_t tid;
    int       count;
} pp_sem_recursive_t;

typedef union {
    pthread_mutex_t    mtx;
    pp_sem_t           sem;
    pp_sem_recursive_t sem_recsv;
} pp_mutex_var_t;

typedef struct pp_mutex_s pp_mutex_t;

struct pp_mutex_s {
    pp_mutex_type_t type;
    pp_mutex_var_t  mvar;
};

void pp_mutex_lock(pp_mutex_t* m);

void pp_mutex_unlock(pp_mutex_t* m);

void pp_mutex_destroy(pp_mutex_t* m);

void pp_mutex_init_proclocal(pp_mutex_t* mutex, pp_mutex_kind_t kind);

void pp_mutex_init_procshared(pp_mutex_t* mutex, key_t shmkey,
			      pp_mutex_kind_t kind);

int pp_sema_init(key_t key);

void pp_sema_lock(int semid);

void pp_sema_unlock(int semid);

void pp_sema_destroy(int semid);

#endif /* _PP_MUTEX_H */
