/**
 * rbuf.h
 *
 * ring buffer interface
 * 
 * generic ring buffer that is capable of tracking 
 * overflow conditions, if initialized to do so.
 *
 * (c) 2004 Peppercon AG, 2005/4/29, tbr@peppecon.de
 */

#ifndef __PP_RBUF_H__
#define __PP_RBUF_H__

#include <sys/types.h>

/* ringbuffer */
typedef struct pp_rbuf_s pp_rbuf_t;

/*
 * initialices a ring buffer of the given size
 * the track_ovfl bits indicates whether overflows
 * will be tracked or not
 */
pp_rbuf_t* pp_rbuf_create(size_t capacity, int track_ovfl);

/*
 * cleans up rbuf
 */
void pp_rbuf_destroy(pp_rbuf_t* rbuf);

/*
 * returns the free space in the ring buffer
 */
size_t pp_rbuf_space(pp_rbuf_t* rbuf);

/*
 * returns the number of bytes in the ring buffer
 */
size_t pp_rbuf_size(pp_rbuf_t* rbuf);

/*
 * Appends the buffer of given len to the ringbuf.
 * The buffer will not overflow but discard bytes that
 * don't fit in.
 *
 * returns the number of bytes that were actually
 * appended. this number may be smaller as requested
 * because of overflow.
 */
size_t pp_rbuf_append(pp_rbuf_t* rbuf, const char* buf, const size_t len);

/*
 * Reads copies the requested number of bytes or all what is available
 * from the head of the ringbuf to the target buffer. In case a
 * overflow condition is inbetween the requested bytes, the continous
 * area will be copied only
 *
 * returns the number of bytes copied, i.e. 0 if empty
 *         the ovfl out parameter indicates whether there
 *         was a overflow condition infront of the copied buffer
 */
size_t pp_rbuf_remove(pp_rbuf_t* rbuf, int* ovfl, char* buf,
		      const size_t len);

/*
 * Discards all data in rbuf
 * not implemented, yet
void pp_rbuf_clear(pp_rbuf_t* rbuf);
 */

/*
 * checks whether next remove starts with a overflow condition,
 * i.e. would set the ovfl param of pp_rbuf_remove to TRUE
 */
int pp_rbuf_next_is_ovfl(pp_rbuf_t* rbuf);

#endif /* __PP_RBUF_H__ */
