/**
 * selector.h
 *
 * Selector interface used to implement single threaded event
 * based frameworks. There are two possible events: I/O and Timer.
 * User code will be notified about the activity using the supplied
 * callback
 *
 * ATTENTION: currently this magic is entirely single threaded, i.e.
 * we do not assume that some other thread is calling any of these 
 * functions except that single thread that will finally run the loop.
 * If you want to terminate the magic, one of the event handlers has
 * to call pp_select_interrupt what will cause the infinite loop to 
 * break. 
 * 
 * (c) 2004 Peppercon AG, 11/9/2004, tbr@peppecon.de
 */

#ifndef  _PP_SELECTOR_H
#define _PP_SELECTOR_H

#include <sys/poll.h>

/**
 * init selector
 */
int pp_select_init(void);

/**
 * frees all data
 * should be called after stopping
 * the whole magic using pp_select_interrupt
 */
void pp_select_cleanup(void);


/**
 * I/O selector callback function type
 */
typedef int (*pp_select_fd_hndl_t)(int item_id, int fd, short event,
				   void* ctx);

/**
 * Timeout selector function type
 */
typedef int (*pp_select_to_hndl_t)(int item_id, void* ctx);

/**
 * Scheduled function selector function type
 */
typedef int (*pp_select_sf_hndl_t)(void* ctx);

				      
/**
 * add a file descriptor item to the selector.
 * @param fd the fd to add
 * @param events the events causing the selector to call the callback
 *               available values are defined in sys/poll.h
 * @param io_handl callback to call
 * @param ctx arbitrary data hook
 * @return PP_ERR (-1) in case of error, errno will be set
 *         selector item id (will be > 0) in case of success,
 *         this id needs to be supplied to the remove call
 */
int pp_select_add_fd(int fd, short events, pp_select_fd_hndl_t io_hndl,
		     void* ctx);

/**
 * add timer item to the selector.
 * @param timeout the timeout in milliseconds
 * @param repeat boolean defining whether the to should fire
 *        periodically or only once
 * @param to_hndl callback to call
 * @param ctx arbitrary data hook
 * @return PP_ERR (-1) in case of error, errno will be set
 *         selector item id (will be > 0) in case of success,
 *         this id needs to be supplied to the remove call
 */
int pp_select_add_to(unsigned long timeout, int repeat,
		     pp_select_to_hndl_t to_hndl, void* ctx);

/**
 * add scheduled function to selector.
 * This function is thread safe. It is meant to pass activity
 * from another thread to the selector thread.
 * A scheduled function will be called as soon as possible and
 * exactly one time.
 * This function should not be called from the selector thread!
 *
 * @param pp_select_sf_hndl_t pointer to function to call
 * @param ctx arbitrary data hook passed along into the function
 * @return PP_ERR(-1) in case of error, errno will be set
 *         selector item id (will be > 0) in case of success,
 *         this id needs to be supplied to the remove call
 */
int pp_select_add_sf(pp_select_sf_hndl_t sf_hndl, void* ctx);

/**
 * removes an I/O selector item, from the selector
 * @param item_id the selector item id returned by one of the add calls
 * @return PP_SUC if OK
 *         PP_ERR in case the item id is unknown
 */
int pp_select_remove_fd(int item_id);

/**
 * removes a TO selector item the selector
 * @param item_id the selector item id returned by one of the add calls
 * @return PP_SUC if OK
 *         PP_ERR in case the item id is unknown
 */
int pp_select_remove_to(int item_id);

/**
 * removes an SF selector item from the selector.
 * In contrast to pp_select_add_sf this function is not thread safe.
 * It must be called from within the selector thread only
 *
 * @param item_id the selector item id returned by one of the add calls
 * @return PP_SUC if OK
 *         PP_ERR in case the item_id is unknown
 */
int pp_select_remove_sf(int item_id);

/**
 * get the remaining time till the specified TO times out (in milliseconds)
 * @param item_id the selector item id returned by the add call
 * @return remaining time to next to in milliseconds
 *         PP_ERR if the specified item is unknown
 */
int pp_select_remaining_time(int item_id);

/**
 * blocks until an error occures or there are no more item
 * in the selector. Might be terminated by calling pp_select_interrupt
 * @return PP_SUC if interupted
 *         PP_ERR if some error occured, errno ist set
 */
int pp_select_loop(void);


/**
 * interrupts the blocked select, this may be used by another thread
 * to terminate the whole magic
 */
void pp_select_interrupt(void);

#endif /* _PP_SELECTOR_H */
