#include <stdlib.h>
#include <ctype.h>
#include <stdio.h>
#include <stddef.h>
#include <signal.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/syscall.h>
#include <openssl/md5.h>
#include <pp/base.h>
#include <pp/strstream.h>

void *
pp_mem_search(const void * data, size_t data_size,
	      const void * pattern, size_t pattern_size)
{
    const void * data_end = data + data_size - pattern_size + 1;
    void * found = NULL;

    if (data    == NULL || data_size    == 0 ||
	pattern == NULL || pattern_size == 0) {

	return NULL;
    }

    while (data < data_end) {
	if ((found = memchr(data, *(const char*)pattern, data_end - data)) == NULL ||
	    !memcmp(found, pattern, pattern_size)) {
	    return found;
	}
	data = found + 1;
    }

    return NULL;
}

char *
pp_trim_string(char * s)
{
    size_t s_len;

    if (s && (s_len = strlen(s)) > 0) {
	char * slp = s;
	char * srp = &s[s_len-1];
	while (*slp && isspace(*slp)) { slp++; s_len--; }
	while (srp > slp && *srp && isspace(*srp)) { srp--; s_len--; }
	memmove(s, slp, s_len);
	s[s_len] = '\0';
    }

    return s;
}

/* convert a string to lower case */
char *
pp_strtolower(char * s)
{
    char *sp;

    if (s == NULL) return NULL;

    for (sp = s; *sp != '\0'; ++sp) {
	if (isupper(*sp)) {
	    *sp = (char) tolower(*sp);
	}
    }
    return s;
}

/* convert a string to upper case */
char *
pp_strtoupper(char * s)
{
    char *sp;

    if (s == NULL) return NULL;

    for (sp = s; *sp != '\0'; ++sp) {
	if (islower(*sp)) {
	    *sp = (char) toupper(*sp);
	}
    }
    return s;
}

/*
 * md5_hash_to_str()
 *
 * This routine converts a MD5 hash into its string representation.
 */
void
pp_md5_hash_to_str(const char * hash, char * hash_str)
{
    char * hs = hash_str;
    int i;

    for (i = 0; i < MD5_DIGEST_LENGTH; i++, hs += 2) {
	sprintf(hs, "%02x", hash[i]);
    }
}

/*
 * free_dynamic_array() - frees a dynamic array
 *
 * NOTE: If "num_entries" is lower than 0 the array is assumed
 *       to be NULL terminated.
 */
void
pp_free_dynamic_array(char ** array, ssize_t num_entries,
		      void(*free_elem_func)(void *))
{
    ssize_t i;

    if (!array) { return; }

    if (free_elem_func) {
	if (num_entries < 0) {
	    for (i = 0; array[i]; i++) { free_elem_func(array[i]); }
	} else {
	    for (i = 0; i < num_entries; i++) { free_elem_func(array[i]); }
	}
    }
    free(array);
}

int
pp_check_and_alloc_mem(void **mem_p, size_t size,
		       unsigned int * count, unsigned int new_count)
{
#define MEM_SPARE_COUNT 20
    void * new_mem;

    if (count == NULL) { return -1; }
    
    if (new_count > *count) {
	*count = new_count + MEM_SPARE_COUNT;
	if ((new_mem = realloc(*mem_p, (*count) * size)) == NULL) {
	    return -1;
	}
	*mem_p = new_mem;
    }
    return 0;
#undef MEM_SPARE_COUNT
}

/*
 * int pp_system(char *command)
 *
 * This routing works just like the normal system with the
 * exception that it closes all file descriptors > 2
 * after the fork.
 */
int pp_system(const char *command) {
    return pp_systeme(command, NULL);
}

int
pp_systeme(const char *command, char *const envp[])
{
    int wait_val, pid;
    __sighandler_t save_quit, save_int, save_chld;
    struct rlimit rlim;
    int i, max_fd;
    
    if (command == 0)
	return 1;    
    
    save_quit = signal(SIGQUIT, SIG_IGN);
    save_int = signal(SIGINT, SIG_IGN);
    save_chld = signal(SIGCHLD, SIG_DFL);
    
    if ((pid = fork()) < 0) {
	signal(SIGQUIT, save_quit);
	signal(SIGINT, save_int);
	signal(SIGCHLD, save_chld);
	return -1;
    }
    if (pid == 0) {
	/* get max fd nr */
	if (!getrlimit(RLIMIT_NOFILE, &rlim)) {
	    max_fd = rlim.rlim_max;
	} else {
	    max_fd = 256;
	}
	for (i = 3; i < max_fd; i++) {
	    close(i);
	}
	
	signal(SIGQUIT, SIG_DFL);
	signal(SIGINT, SIG_DFL);
	signal(SIGCHLD, SIG_DFL);
	
	execle("/bin/sh", "sh", "-c", command, (char *) 0, envp);
	_exit(127);
    }
    /* Signals are not absolutly guarenteed with vfork */
    signal(SIGQUIT, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    
#if 0
    printf("Waiting for child %d\n", pid);
#endif
    
    if (TEMP_FAILURE_RETRY (waitpid (pid, &wait_val, 0)) != pid)
	wait_val = -1;
    
    signal(SIGQUIT, save_quit);
    signal(SIGINT, save_int);
    signal(SIGCHLD, save_chld);
    return wait_val;
}

int pp_systemf(const char* command, ...) {
    int ret;
    va_list ap;
    pp_strstream_t strbuf;
    const char* p = command;
    char c;
    char *s;
    
    assert(command && *command);
    
    /* get the commandline... */
    pp_strstream_init(&strbuf);
    va_start(ap, command);
    while((c = *p++) != '\0') {
        switch (c) {
            case '%':
                switch (*p++) {
                    case 'd':
                        pp_strappendf(&strbuf, " \\\"%d\\\"", va_arg(ap, int));
                        break;
                    case 's':
                        s = va_arg(ap, char*);
                        pp_strappend(&strbuf, " \\\"");
                        if(s && *s)
                           pp_strappend(&strbuf, s);
                        pp_strappend(&strbuf, "\\\"");
                        break;
                    default: assert(0);
                }
                break;
	    default:
	        pp_strappendchr(&strbuf, c);
	}
    }
    va_end(ap);
    
    ret = pp_system(strbuf.buf);
    
    pp_strstream_free(&strbuf);
    
    return ret;
}

/**
 * pp_popen_rw/pp_pclose_rw
 *
 * Just like popen/pclose, but uses two pipes
 * to allow bidirectional comunication with the child
 */
int
pp_popen_rw(const char* command, pp_popen_rw_t * data)
{
    int fd_out[2] = { -1, -1 };
    int fd_in[2] = { -1, -1 };
    int ret = -1;

    assert(command);
    assert(data);

    data->pid = -1;
    data->stdout = data->stdin = NULL;

    if (pipe(fd_out) != 0) {
	pp_log_err("%s(): '%s' on pipe out\n", ___F, command);
	goto bail;
    }
    if (pipe(fd_in) != 0) {
	pp_log_err("%s(): '%s' on pipe in\n", ___F, command);
	goto bail;
    }

    if ((data->stdin  = fdopen(fd_in[0], "r")) == NULL) {
	pp_log_err("%s(): '%s' on fdopen in\n", ___F, command);
	goto bail;
    }
    if ((data->stdout = fdopen(fd_out[1], "w")) == NULL) {
	pp_log_err("%s(): '%s' on fdopen out\n", ___F, command);
	goto bail;
    }
    
    if ((data->pid = fork()) < 0) {
	pp_log_err("%s(): '%s' on fork\n", ___F, command);
	goto bail;
    }
    if (data->pid == 0) { /* child */
	close(fd_in[0]); close(fd_out[1]);
	close(1); dup2(fd_in[1],  1); close(fd_in[1]);
	close(0); dup2(fd_out[0], 0); close(fd_in[0]);

	execl("/bin/sh", "sh", "-c", command, (char *) 0);
	_exit(255);
    }
    
    close(fd_in[1]);
    close(fd_out[0]);

    ret = 0;
    
 bail:
    if (ret != 0) {
	if (data->stdin) {
	    fclose(data->stdin);
	    fd_in[0] = -1;
	}
	if (data->stdout) {
	    fclose(data->stdout);
	    fd_out[1] = -1;
	}
	if (fd_in[0] != -1) close(fd_in[0]);
	if (fd_in[1] != -1) close(fd_in[1]);
	if (fd_out[0] != -1) close(fd_out[0]);
	if (fd_out[1] != -1) close(fd_out[1]);
    }
    return ret;
}

int
pp_pclose_rw(pp_popen_rw_t * data)
{
    int ret;

    assert(data);

    fclose(data->stdin);
    fclose(data->stdout);

    if (TEMP_FAILURE_RETRY (waitpid (data->pid, &ret, 0)) != data->pid) {
	ret = -1;
    }

    return ret;
}

int64_t
timeval_diff_usec(struct timeval * new, struct timeval * old)
{
    int64_t old_sec, old_usec, new_sec, new_usec;

    assert(new);
    assert(old);

    old_sec = (uint32_t)old->tv_sec;
    old_usec = (uint32_t)old->tv_usec;
    new_sec = (uint32_t)new->tv_sec;
    new_usec = (uint32_t)new->tv_usec;

    return (new_sec - old_sec) * 1000000LL + (new_usec - old_usec);
}

struct timespec
pp_get_wakeup_time(unsigned int mstime /*ms*/) {
    struct timeval now;
    struct timespec to;

    gettimeofday(&now, NULL);
    now.tv_usec += (mstime % 1000) * 1000;
    if (now.tv_usec >= 1000000) {
        now.tv_usec -= 1000000;
        now.tv_sec++;
    }
    now.tv_sec += mstime / 1000;

    to.tv_sec = now.tv_sec;
    to.tv_nsec = now.tv_usec * 1000;
    return to;
}
