/**
 * selector.c
 *
 * Selector implementation used to implement single threaded event
 * based frameworks. There are two possible events: I/O and Timer.
 * User code will be notified about the activity using the supplied
 * callback
 * 
 * ATTENTION: currently this magic is entirely single threaded, i.e.
 * we do not assume that some other thread is calling any of these 
 * functions except that single thread that will finally run the loop.
 * If you want to terminate the magic, one of the event handlers has
 * to call pp_select_interrupt what will cause the infinite loop to 
 * break. 
 * 
 * (c) 2004 Peppercon AG, 11/9/2004, tbr@peppecon.de
 */


#include <assert.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/time.h>
#include <sys/poll.h>
#include <pp/base.h>
#include <pp/selector.h>
#include <limits.h>

/*
 * private declarations
 * ----------------------
 */
typedef struct sel_item_s {
    struct list_head node;
    int item_id;
    int marked_for_rm;    // remove marker
} sel_item_t;

typedef struct fd_sel_item_s {
    sel_item_t base;
    int fd;
    short events;
    pp_select_fd_hndl_t fd_hndl;
    void* ctx;
} fd_sel_item_t;

typedef struct to_sel_item_s {
    sel_item_t base;
    struct timeval wakeup;
    unsigned long repeat_to;
    pp_select_to_hndl_t to_hndl;
    void* ctx;
} to_sel_item_t;

typedef struct sf_sel_item_s {
    sel_item_t base;
    pp_select_sf_hndl_t sf_hndl;
    void* ctx;
} sf_sel_item_t;

typedef struct sf_sel_notify_fd_s {
    int read;
    int write;
} sf_sel_notify_fd_t;

static fd_sel_item_t* fd_sel_item_create(int fd, short events,
					 pp_select_fd_hndl_t fd_hndl,
					 void* ctx);
static to_sel_item_t* to_sel_item_create(const struct timeval* to, 
					 int repeat,
					 pp_select_to_hndl_t to_hndl,
					 void* ctx);
static sf_sel_item_t* sf_sel_item_create(pp_select_sf_hndl_t sf_hndl,
					 void* ctx);
static void to_sel_list_add_sorted(to_sel_item_t* toi);
static int sel_list_remove(struct list_head* list, int item_id, int rm_pend);
static void sel_list_cleanup(struct list_head* list);
static inline unsigned int get_unique_item_id(unsigned long* id);
static void timeval_add(struct timeval* tv_dst, const struct timeval* tv_src,
			unsigned long tms);
static long timeval_diff(const struct timeval* now,
			 const struct timeval* then);
static void timeval_now_add(struct timeval* tv, unsigned long tms);
static long timeval_now_diff(const struct timeval* then);
static inline int timeval_greater(const struct timeval* this,
				  const struct timeval* that);
static void sel_list_remove_marked(struct list_head* list);
static int sf_sel_notified(int item_id, int fd, short event, void* ctx);
static void sf_sel_notify(void);

/*
 * Private data
 * ---------------
 */
static LIST_HEAD(fd_sel_list);
static int fd_sel_list_size = 0;
static struct list_head fd_add_list; // will be initialized in loop
static int fd_add_list_size;         // will be initialized in loop

static LIST_HEAD(to_sel_list);

static LIST_HEAD(sf_sel_list);
static LIST_HEAD(sf_notify_list);
static pthread_mutex_t sf_sel_list_mtx =
    PTHREAD_ERRORCHECK_MUTEX_INITIALIZER_NP;
static sf_sel_notify_fd_t sf_sel_notify_fd = { -1, -1 };

static unsigned long unique_id = 1;
static unsigned long sf_unique_id = 1;
static int select_loop_state = 0;
static int removes_pending = 0;
static int interrupted;

#if !defined(NDEBUG)
pthread_t sel_thread_id = -1;
#endif /* !NDEBUG */

#define SELECT_LOOP_RUN		1
#define SELECT_LOOP_FD_CHECK	2

/*
 * Public function implementations
 * ---------------------------------
 */

int pp_select_init() {
    if (0 > pipe(&sf_sel_notify_fd.read) ||
	0 > fcntl(sf_sel_notify_fd.read, F_SETFL, O_NONBLOCK) ||
	0 > fcntl(sf_sel_notify_fd.write, F_SETFL, O_NONBLOCK)) {
	return PP_ERR;
    }
    pp_select_add_fd(sf_sel_notify_fd.read, POLLIN, sf_sel_notified, NULL);
    return PP_SUC;
}

void pp_select_cleanup() {
    sel_list_cleanup(&fd_sel_list);
    sel_list_cleanup(&to_sel_list);
    sel_list_cleanup(&sf_sel_list);
}

int pp_select_add_fd(int fd, short events, 
		     pp_select_fd_hndl_t fd_hndl, void* ctx) {
    struct list_head* i;
    fd_sel_item_t* item;
    
    // check all non removed entries,
    // duplicates of fds are only allowed if they request different events
    list_for_each(i, &fd_sel_list) {
	fd_sel_item_t* fdi = list_entry(i, fd_sel_item_t, base.node);
	if (!fdi->base.marked_for_rm && fd == fdi->fd &&
	    (fdi->events & events) != 0) {
	    errno = EINVAL;
            pp_log_err("pp_select_add_fd: duplicate fd + events");
	    return PP_ERR;
	}
    }

    item = fd_sel_item_create(fd, events, fd_hndl, ctx);
    if ((select_loop_state & SELECT_LOOP_RUN) &&
	(select_loop_state & SELECT_LOOP_FD_CHECK)) {
	list_add(&item->base.node, &fd_add_list);
	++fd_add_list_size;
    } else {
	list_add(&item->base.node, &fd_sel_list);
	++fd_sel_list_size;
    }
    return item->base.item_id;
}
    

int pp_select_add_to(unsigned long timeout, int repeat,
		     pp_select_to_hndl_t to_hndl, void* ctx) {
    to_sel_item_t* item;
    struct timeval wakeup;

    timeval_now_add(&wakeup, timeout);

    item = to_sel_item_create(&wakeup, repeat ? timeout : 0, to_hndl, ctx);
    to_sel_list_add_sorted(item);
    
    return item->base.item_id;
}

int pp_select_add_sf(pp_select_sf_hndl_t sf_hndl, void* ctx) {
    sf_sel_item_t* item;

    MUTEX_LOCK(&sf_sel_list_mtx);
    item = sf_sel_item_create(sf_hndl, ctx);
    list_add_tail(&item->base.node, &sf_sel_list);
    sf_sel_notify();
    MUTEX_UNLOCK(&sf_sel_list_mtx);
    
    return item->base.item_id;
}    

int pp_select_remove_fd(int item_id) {
    --fd_sel_list_size;
    return sel_list_remove(&fd_sel_list, item_id, 1);
}

int pp_select_remove_to(int item_id) {
    return sel_list_remove(&to_sel_list, item_id, 1);
}

/* supposed to be called from selector thread only!! *
 * sf_notify_list handling is not thread safe        */
int pp_select_remove_sf(int item_id) {
    int ret;
    /* make sure we are selector thread */
    assert(sel_thread_id == pthread_self());
    MUTEX_LOCK(&sf_sel_list_mtx);
    if (PP_SUC != (ret = sel_list_remove(&sf_sel_list, item_id, 0))) {
	ret = sel_list_remove(&sf_notify_list, item_id, 0);
    }
    MUTEX_UNLOCK(&sf_sel_list_mtx);
    return ret;
}

int pp_select_remaining_time(int item_id) {
    int rv = PP_ERR;
    int diff;
    struct list_head* i;

    i = to_sel_list.next;
    while (i != &to_sel_list) {
        to_sel_item_t *toi;
        toi = list_entry(i, to_sel_item_t, base.node);
        if (toi->base.item_id == item_id) {
            // This is the timer we are looking for
            diff = timeval_now_diff(&(toi->wakeup));
	    if (diff < 0) {
	        diff = 0;
	    }
	    rv = diff;
            i = &to_sel_list;  // this will exit the loop
        } else {
            i = i->next;
        }
    }

    return rv;
}

int pp_select_loop() {
    static struct pollfd* fds = NULL;
    static int fds_size = 0;
    struct list_head* i;
    int r, c, timeout, ret = PP_SUC;
    select_loop_state |= SELECT_LOOP_RUN;
#if !defined(NDEBUG)
    sel_thread_id = pthread_self();
#endif /* !NDEBUG */

    for (interrupted = 0; !interrupted; ) {
	// check for size of poll array
	if (fd_sel_list_size > fds_size) {
	    // spare some reallocs
	    fds_size = (fd_sel_list_size + 8) & 0xfffffff8; 
	    fds = realloc(fds,  fds_size * sizeof(struct pollfd));
	}

	// and create the poll array
	for (c = 0, i = fd_sel_list.next; c < fd_sel_list_size; 
	     ++c, i = i->next) {
	    fds[c].fd = list_entry(i, fd_sel_item_t, base.node)->fd;
	    fds[c].events = list_entry(i, fd_sel_item_t, base.node)->events;
	    fds[c].revents = 0;
	}

	// get timediff to next alarm
	if (!list_empty(&to_sel_list)) {
	    timeout = timeval_now_diff(&list_entry(to_sel_list.next,
				       to_sel_item_t, base.node)->wakeup);
	    if (timeout < 0) timeout = 0;
	} else if (fd_sel_list_size == 0) {
	    // nothing to select, is this really an error???
	    errno = EINVAL;
	    ret = PP_ERR;
	    break;
	} else {
	    timeout = -1;
	}
	// go and sleep until activity or timeout occures
	if (0 > (r = poll(fds, fd_sel_list_size, timeout))) {
	    if (errno == EINTR) continue;
	    ret = PP_ERR;
	    break;
	} else if (r > 0) {
	    // check for signaled fds
	    INIT_LIST_HEAD(&fd_add_list); // empty add list
	    fd_add_list_size = 0;
	    select_loop_state |= SELECT_LOOP_FD_CHECK;
	    for (c = 0, i = fd_sel_list.next; c < fd_sel_list_size; 
		 ++c, i = i->next) {
		fd_sel_item_t* fdi = list_entry(i, fd_sel_item_t, base.node);
		if (fds[c].revents != 0 && !fdi->base.marked_for_rm) {
		    fdi->fd_hndl(fdi->base.item_id, fdi->fd, fds[c].revents,
				 fdi->ctx);
		}
	    }
	    select_loop_state &= ~SELECT_LOOP_FD_CHECK;
	    // add newly added fds to fd_sel_list
	    list_splice(&fd_add_list, &fd_sel_list);
	    fd_sel_list_size += fd_add_list_size;
	}

	// check for timeouts
	i = to_sel_list.next;
	while (i != &to_sel_list &&
	       timeval_now_diff(&list_entry(i, to_sel_item_t, 
					    base.node)->wakeup) <= 0) {
	    to_sel_item_t* toi = list_entry(i, to_sel_item_t, base.node);
	    i = i->next;
	    if (toi->base.marked_for_rm) continue; // skip removed entries
	    list_del(&toi->base.node);
	    if (toi->repeat_to > 0) {
		timeval_now_add(&toi->wakeup, toi->repeat_to);
		to_sel_list_add_sorted(toi);
		toi->to_hndl(toi->base.item_id, toi->ctx);
	    } else {
		toi->to_hndl(toi->base.item_id, toi->ctx);
		free(toi);
	    }
	}

	// really delete all entries that have been marked in the handlers
	if (removes_pending) {
	    sel_list_remove_marked(&fd_sel_list);
	    sel_list_remove_marked(&to_sel_list);
	    removes_pending = 0;
	}
    }
    free(fds);
    select_loop_state &= ~SELECT_LOOP_RUN;
    return ret;
}

void pp_select_interrupt() {
    interrupted = 1;
}

/*
 * Private function implementations
 * ----------------------------------
 */
static void sf_sel_notify() {
    unsigned char d = 1;
    /* if EAGAIN is returned, the pipe is full of notifications        *
     * so no need to worry here, though it seems strange if it happens */
    if (0 > write(sf_sel_notify_fd.write, &d, 1)) {
	pp_log_err("sf_sel_notify failed");
    }
}

static int sf_sel_notified(int item_id UNUSED, int fd, 
			   short event UNUSED, void* ctx UNUSED) {
    char buf[10];
    struct list_head* i, *next;
    ssize_t n;

    MUTEX_LOCK(&sf_sel_list_mtx);
    
    /* empty pipe, so notify won't block */
    do {
	n = read(fd, buf, sizeof(buf));
    } while(n == sizeof(buf));
#if !defined(NDEBUG)
    if (n < 0 && errno != EAGAIN) {
	pp_log_err("sf_sel_notified: read failed");
    }
#endif

    /* move elements over to notify list, so we can unlock the list during callback */
    list_splice(&sf_sel_list, &sf_notify_list);
    INIT_LIST_HEAD(&sf_sel_list); // fix up selector list

    MUTEX_UNLOCK(&sf_sel_list_mtx);
    
    list_for_each_safe(i, next, &sf_notify_list) {
	sf_sel_item_t* sfi = list_entry(i, sf_sel_item_t, base.node);
	if (!sfi->base.marked_for_rm) {
	    sfi->sf_hndl(sfi->ctx);
	}
	list_del(i);
	free(sfi);
    }
    
    return PP_SUC;
}

static fd_sel_item_t* fd_sel_item_create(int fd, short events,
					 pp_select_fd_hndl_t fd_hndl,
					 void* ctx) {
    fd_sel_item_t* i;

    i = malloc(sizeof(fd_sel_item_t));
    i->base.item_id = get_unique_item_id(&unique_id);
    i->base.marked_for_rm = 0;
    i->fd = fd;
    i->events = events;
    i->fd_hndl = fd_hndl;
    i->ctx = ctx;

    return i;
}

static to_sel_item_t* to_sel_item_create(const struct timeval* to, 
					 int repeat_to,
					 pp_select_to_hndl_t to_hndl,
					 void* ctx) {
    to_sel_item_t* i;

    i = malloc(sizeof(to_sel_item_t));
    i->base.item_id = get_unique_item_id(&unique_id);
    i->base.marked_for_rm = 0;
    i->wakeup = *to;
    i->repeat_to = repeat_to;
    i->to_hndl = to_hndl;
    i->ctx = ctx;

    return i;
}

static sf_sel_item_t* sf_sel_item_create(pp_select_sf_hndl_t sf_hndl,
					 void* ctx) {
    sf_sel_item_t* i;

    i = malloc(sizeof(sf_sel_item_t));
    i->base.item_id = get_unique_item_id(&sf_unique_id);
    i->base.marked_for_rm = 0;
    i->sf_hndl = sf_hndl;
    i->ctx = ctx;

    return i;
}

static void to_sel_list_add_sorted(to_sel_item_t* toi) {
    struct list_head* i;
    
    list_for_each(i, &to_sel_list) {
	if (!timeval_greater(&toi->wakeup, &list_entry(i, to_sel_item_t, 
						       base.node)->wakeup))
	    break;
    }
    list_add_tail(&toi->base.node, i);
}

static int sel_list_remove(struct list_head* list, int item_id, int rm_pend) {
    struct list_head* i;
    sel_item_t* e;

    list_for_each(i, list) {
	e = list_entry(i, sel_item_t, node);
	if (e->item_id == item_id) {
	    if (select_loop_state & SELECT_LOOP_RUN) {
		e->marked_for_rm = 1;
		removes_pending = rm_pend;
	    } else {
		list_del(i);
		free(e);
	    }
	    return PP_SUC;
	}
    }
    return PP_ERR;
}

static void sel_list_remove_marked(struct list_head* list) {
    struct list_head *i, *n;
    list_for_each_safe(i, n, list) {
	if (list_entry(i, sel_item_t, node)->marked_for_rm) {
	    list_del(i);
	    free(list_entry(i, sel_item_t, node));
	}
    }
}


static void sel_list_cleanup(struct list_head* sel_list) {
    struct list_head *i, *n;
    list_for_each_safe(i, n, sel_list) {
	list_del(i);
	free(list_entry(i, sel_item_t, node));
    }
}

 
static inline unsigned int get_unique_item_id(unsigned long* id) {
    // WARNING: maybe we need todo something here to make it
    // really unique. however, currently I think the id space
    // is big enough to avoid conflicts
    unsigned long r = *id;
    if (r == INT_MAX) *id = 1; // don't use 0, can be used as 'no-hndl-value'
    else ++*id;
    return r;
}

static void timeval_add(struct timeval* tv_dst, const struct timeval* tv_src,
			unsigned long tms) {
    time_t as = tms / 1000;
    suseconds_t ams = tms % 1000;
    suseconds_t cms = tv_src->tv_usec / 1000;
    suseconds_t nms = ams + cms;
    assert(cms >= 0);

    tv_dst->tv_sec = tv_src->tv_sec + as + (nms / 1000);
    tv_dst->tv_usec = (nms % 1000) * 1000;
}

static long timeval_diff(const struct timeval* now,
			 const struct timeval* then) {
    long long t1, t2; // in ms
    
    t1 = now->tv_sec * 1000 + now->tv_usec / 1000;
    t2 = then->tv_sec * 1000 + then->tv_usec / 1000;
    
    return (long)(t2 - t1);
}

static void get_processor_time(struct timeval* now, struct timezone *tz UNUSED) {
    u_int64_t nowproc;
    nowproc = pp_hrtime_us();
    now->tv_usec = (suseconds_t)(nowproc % 1000000);
    now->tv_sec = (time_t)(nowproc / 1000000);
}

static void timeval_now_add(struct timeval * tv, unsigned long tms) {
    struct timeval now;
    // NOTE: do not use gettimeofday() here because the timebase may change
    get_processor_time(&now, NULL);
    timeval_add(tv, &now, tms);
}

static long timeval_now_diff(const struct timeval* then) {
    struct timeval now;
    // NOTE: do not use gettimeofday() here because the timebase may change
    get_processor_time(&now, NULL);
    return timeval_diff(&now, then);
}

static inline int timeval_greater(const struct timeval* this,
				  const struct timeval* that) {
    return this->tv_sec > that->tv_sec ||
	(this->tv_sec == that->tv_sec && this->tv_usec > that->tv_usec);
}
