/******************************************************************************
 * This routines are used to enforce that only one instance of a process
 * is running. This is achieved by checking and creating a pidfile.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <pp/base.h>

/* max retries to aquire lock on pidfile - 500ms between each retry */
#define MAX_LOCK_RETRIES 6

singleton_ret_code_t
pp_base_enforce_singleton(const char * pidfile)
{
    int fd = -1;
    singleton_ret_code_t ret;
    while (1) {
	ret = SINGLETON_CRITICAL_EXCEPTION;

        fd = open(pidfile, O_CREAT | O_EXCL | O_WRONLY, S_IRUSR | S_IWUSR);
        if (fd != -1) {
            if (flock(fd, LOCK_EX) == -1) goto out_close;
            /* the pid file was created - all ok */
            if (dprintf(fd, "%d", getpid()) > 0) {
		/* wrote our pidfile */
		ret = SINGLETON_OK;
	    }
	    goto out_unlock_close;
        }

        /*
	 * There is a problem creating the pidfile, e.g. another process using
	 * the same pidfile. Check if the old process is running or if it's a
	 * stale file.
	 */
        fd = open(pidfile, O_RDONLY);
        if (fd != -1) {
	    char proc_pid_exe_path[128];
	    char pidstr[12];
	    char * ep;
	    unsigned long pid;
	    struct stat own_sbuf, other_sbuf;
	    int n, lock_retry_cnt;

	    for (lock_retry_cnt = 0; lock_retry_cnt < MAX_LOCK_RETRIES; lock_retry_cnt++) {
		if (flock(fd, LOCK_EX | LOCK_NB) == -1) {
		    if (errno == EWOULDBLOCK) {
			usleep(500000);
			continue;
		    }
		    goto out_close;
		} else break;
	    }
	    if (lock_retry_cnt == MAX_LOCK_RETRIES) {
		ret = SINGLETON_ALREADY_RUNNING;
		goto out_close;
	    }

	    if ((n = read(fd, pidstr, sizeof(pidstr))) < 0)
		goto out_unlock_close;
	    pidstr[n] = '\0';

	    pid = strtoul(pidstr, &ep, 10);
	    if (*ep != '\0' || pid > 65535) {
		/* pidfile contains invalid data - delete it */
		if (unlink(pidfile) == -1) goto out_unlock_close;
		flock(fd, LOCK_UN);
		close(fd);
		fd = -1;
                continue;
	    }

	    flock(fd, LOCK_UN);
	    close(fd);
	    fd = -1;

	    snprintf(proc_pid_exe_path, sizeof(proc_pid_exe_path), "/proc/%lu/exe", pid);
	    if (stat(proc_pid_exe_path, &other_sbuf) == -1) {
		/*
		 * We cannot stat the exe link of the (possible) other process.
		 * Try to unlink the pidfile and retry.
		 */
		if (unlink(pidfile) == -1) goto out;
                continue;
            }

	    snprintf(proc_pid_exe_path, sizeof(proc_pid_exe_path), "/proc/%u/exe", getpid());
	    if (stat(proc_pid_exe_path, &own_sbuf) == -1) goto out;
            if ((pid_t)pid != (pid_t)getpid()
		&& other_sbuf.st_dev == own_sbuf.st_dev
		&& other_sbuf.st_ino == own_sbuf.st_ino) {

		/* there is really another instance of the program - just exit */
		ret = SINGLETON_ALREADY_RUNNING;
		goto out;
            } else {
                /* there runs another process with the old saved pid, just delete the pidfile */
		if (unlink(pidfile) == -1) goto out;
                continue;
            }
        }
        // we cannot read the pidfile and cannot creat it
        // 1. it is there, but not readable to us
        // 2. we have no rights to create it
	goto out;
    }

 out_unlock_close:
    if (fd != -1) flock(fd, LOCK_UN);
 out_close:
    if (fd != -1) close(fd);
 out:
    return ret;
}

singleton_ret_code_t
pp_base_release_singleton(const char * pidfile)
{
    if (unlink(pidfile) == -1)
	return SINGLETON_UNCRITICAL_EXCEPTION;
    return SINGLETON_OK;
};

int
pp_base_get_pid_of_program_by_pidfile(const char * pidfile, const char * program_path)
{
    int ret = -1;
    int fd = open(pidfile, O_RDONLY);
    if (fd == -1) {
	if (errno == ENOENT) ret = 0;
    } else {
	char proc_pid_exe_path[128];
	char pidstr[12];
	char * ep;
	unsigned long pid;
	struct stat sbuf1, sbuf2;
	int n, lock_retry_cnt;

	for (lock_retry_cnt = 0; lock_retry_cnt < MAX_LOCK_RETRIES; lock_retry_cnt++) {
	    if (flock(fd, LOCK_EX | LOCK_NB) == -1) {
		if (errno == EWOULDBLOCK) {
		    usleep(500000);
		    continue;
		}
		goto out_close;
	    } else break;
	}
	if (lock_retry_cnt == MAX_LOCK_RETRIES) goto out_close;

	if ((n = read(fd, pidstr, sizeof(pidstr))) < 0)
	    goto out_unlock_close;
	pidstr[n] = '\0';

	pid = strtoul(pidstr, &ep, 10);
	if (*ep != '\0' || pid > 65535) {
	    /* pidfile contains invalid data */
	    goto out_unlock_close;
	}

	flock(fd, LOCK_UN);
	close(fd);
	fd = -1;

	snprintf(proc_pid_exe_path, sizeof(proc_pid_exe_path), "/proc/%lu/exe", pid);
	if (stat(proc_pid_exe_path, &sbuf1) == -1) goto out;
	if (stat(program_path, &sbuf2) == -1) goto out;
	if (sbuf1.st_dev == sbuf2.st_dev && sbuf1.st_ino == sbuf2.st_ino) ret = pid;
    }
 out_unlock_close:
    if (fd != -1) flock(fd, LOCK_UN);
 out_close:
    if (fd != -1) close(fd);
 out:
    return ret;
}
