/**
 * Copyright 2005 Peppercon AG
 * Author: Thomas Breitfeld <thomas@peppercon.de>
 *
 * Description: contains routines to control a serial
 *              line terminal more easily
 *              mainly intended for kme code
 */

#include <string.h>
#include <errno.h>
#include <termios.h>
#include <pp/base.h>
#include <pp/termios.h>

/*
 * Set baudrate, parity and number of bits.
 * @param fd     the tty file descriptor
 * @param baudr  the rate in baud pro second
 * @param par    the parity "E" - even
 *                          "O" - odd
 *                          ""  - no parity switched on 
 * @param bits   number of bits, directly given decimal
 * @param stob2  if true, 2 stop bits will be sent, otherwise 1
 * @param hwf    if true, hardware handshake will be enabled
 * @param swf    if true, software handshake will be enabled
 * @return -1 in case of error, errno will be set
 *         > 0 otherwise
 */
int pp_base_set_tty_params(int fd, int baudr, const char* par, 
			   int bits, int stop2, int hwf, int swf) {
    return pp_base_set_tty_params_ex(fd, baudr, par, bits, stop2, hwf, swf, 1, TTY_BREAK_IGNORE);
}


/*
 * Set baudrate, parity and number of bits.
 * @param fd     the tty file descriptor
 * @param baudr  the rate in baud pro second
 * @param par    the parity "E" - even
 *                          "O" - odd
 *                          ""  - no parity switched on 
 * @param bits   number of bits, directly given decimal
 * @param stob2  if true, 2 stop bits will be sent, otherwise 1
 * @param hwf    if true, hardware handshake will be enabled
 * @param swf    if true, software handshake will be enabled
 * @param local  if true, CLOCAL control flag will be set
 * @param ttybrk set BREAK behavior (see TTY_BREAK_xxx)
 * @return -1 in case of error, errno will be set
 *         > 0 otherwise
 */
int pp_base_set_tty_params_ex(int fd, int baudr, const char* par, 
			      int bits, int stop2, int hwf, int swf,
			      int local, int ttybrk)
{
    int spd = -1;
    int ret = PP_SUC;
    struct termios tty;
    // get current attributes
    errno = 0;  
  
    if (tcgetattr(fd, &tty) < 0) {
	pp_log_err("%s(): tcgetattr() failed", ___F);
	return PP_ERR;
    }
  
    // determine baud rate constant...
    switch (baudr) {
      case 0:		spd = B0;	break;
      case 300:		spd = B300;	break;
      case 600:		spd = B600;	break;
      case 1200:	spd = B1200;	break;
      case 2400:	spd = B2400;	break;
      case 4800:	spd = B4800;	break;
      case 9600:	spd = B9600;	break;
      case 19200:	spd = B19200;	break;
      case 38400:	spd = B38400;	break;
      case 57600:	spd = B57600;	break;
      case 115200:	spd = B115200;	break;
      case 230400:	spd = B230400;	break;
      default: errno = EINVAL; return PP_ERR;
    }
    // ...and set speed for in and out
    if (spd != -1) {
	cfsetospeed(&tty, (speed_t)spd);
	cfsetispeed(&tty, (speed_t)spd);
    }
    // determine number of bit constant
    switch (bits) {
      case '5': tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS5; break;
      case '6': tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS6; break;
      case '7': tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS7; break;
      case '8':
      default: tty.c_cflag = (tty.c_cflag & ~CSIZE) | CS8; break;
    }	
    // Set into raw, no echo mode 
    tty.c_iflag = 0;
    tty.c_lflag = 0;
    tty.c_oflag = 0;
    tty.c_cflag |= CREAD;
    tty.c_cc[VMIN] = 1;
    tty.c_cc[VTIME] = 5;

    // set CLOCAL flag (observer modem signals or not)
    if (local) tty.c_cflag |= CLOCAL;
    else       tty.c_cflag &= ~CLOCAL;

    if (ttybrk == TTY_BREAK_AS_SIGNAL) { // send SIGINT
        tty.c_iflag |= BRKINT;
    } else if (ttybrk == TTY_BREAK_AS_NULL_CHAR) { // read BREAK as '\0'
        // no flags to set
    } else if (ttybrk == TTY_BREAK_AS_SEQUENCE) { // read BREAK as '\337''\0''\0'
        tty.c_iflag |= PARMRK;
    } else { // TTY_BREAK_IGNORE or else: ignore incoming BREAKs
        tty.c_iflag |= IGNBRK;
    }
    
    // set hardware flow control CRTSCTS
    if (hwf) tty.c_cflag |= CRTSCTS;
    else	tty.c_cflag &= ~CRTSCTS;

    // set software flow control XON/XOFF
    if (swf) tty.c_iflag |= IXON | IXOFF;
    else tty.c_iflag &= ~(IXON|IXOFF|IXANY);
  
    // reset and set parity
    tty.c_cflag &= ~(PARENB | PARODD);
    if (par[0] == 'E') tty.c_cflag |= PARENB;
    else if (par[0] == 'O') tty.c_cflag |= (PARENB | PARODD);

    // set number of stop bits
    if (stop2) tty.c_cflag |= CSTOPB;
    else tty.c_cflag &= ~CSTOPB;

    // write back termio structure and set changes 
    // immediately into effect
    ret = tcsetattr(fd, TCSANOW, &tty);
 
    return ret == 0 ? PP_SUC : PP_ERR;
}

