/*******************************************************************************
 * Generic Timer Queue
 ******************************************************************************/

#ifdef PP_FEAT_DEVICE
 
#include <stdlib.h>
#include <sys/time.h>
#include <pp/base.h>
#include <liberic_pthread.h>
#include <pp/timerq.h>
 
static void* timerq_thread(void *arg);
static void timerq_destroy(pp_timerq_t *tq);
static void update_basetime(pp_timerq_t *tq);
 
typedef struct {
    pp_timerq_cb        cb;
    void*               ctx;
    struct timespec     timeout;
} timerq_entry_t;    

pp_timerq_t* pp_timerq_create(void) {
    pp_timerq_t *tq;
    
    tq = (pp_timerq_t*)malloc(sizeof(pp_timerq_t));
    pthread_mutex_init(&tq->mtx, NULL);
    pthread_cond_init(&tq->cond, NULL);
    tq->state = NEW;
    tq->queue = pp_queue_alloc(QUEUE_TYPE_PRIORITY_REVERSE);
    update_basetime(tq);
    
    return tq;
}

int pp_timerq_start(pp_timerq_t *tq) {
    int ret = PP_SUC;
    pthread_t thr;
    
    assert(tq);
    
    switch(tq->state) {
        case NEW:
            if (eric_pthread_create(&thr, 1, 65536, timerq_thread, (void*)tq)) {
                ret = PP_ERR;
                pp_log("pp_timerq_start: can't create thread\n");
                abort();
            }
            // no break!
        case SLEEPING:
            tq->state = RUNNING;

            /* wake up timer thread */
            if (pthread_cond_signal(&tq->cond) != 0) {
                pp_log("%s(): pthread_cond_signal() failed\n", ___F);
            }
            break;
        case RUNNING: // already running, do nothing
            break;
        default:
            assert(0); // something is wrong here
            break;
    }
    return ret;
}

int pp_timerq_stop(pp_timerq_t *tq) {
    assert(tq);
    
    if(tq->state == RUNNING)
        tq->state = SLEEPING;
    
    return PP_SUC;
}

void pp_timerq_destroy(pp_timerq_t *tq) {
    assert(tq);
    
    // destroy thread!
    tq->state = DESTROY;
}

/* API */
int pp_timerq_add(pp_timerq_t *tq, unsigned int timeout,
                  pp_timerq_cb cb, void *ctx) {
    timerq_entry_t *tqe;
    unsigned int wakeup, tsec;
    struct timeval now;
    int ret;
    
    assert(tq);
    
//    if(pp_timerq_empty(tq))
//        update_basetime(tq);
    tqe = (timerq_entry_t*)malloc(sizeof(timerq_entry_t));
    tqe->cb = cb;
    tqe->ctx = ctx;
    MUTEX_LOCK(&tq->mtx);
    /* calculate wake up time */
    gettimeofday(&now, NULL);
    wakeup = (now.tv_sec - tq->basetime) * 1000 + /* msec since basetime */ 
             (now.tv_usec / 1000) +               /* msec since now.tv_sec */
             timeout;                             /* timeout in msec */
             
    /* generate timeout timestamp */
    tsec = timeout / 1000;
    assert(tsec * 1000 <= timeout);
    timeout -= tsec * 1000;
    tqe->timeout.tv_sec = now.tv_sec + tsec;
    tqe->timeout.tv_nsec = ((timeout * 1000) + now.tv_usec) * 1000;
    if(tqe->timeout.tv_nsec > 1000000000) {
        ++tqe->timeout.tv_sec;
        tqe->timeout.tv_nsec -= 1000000000;
    }
    
    /* enqueue timer */
    pp_queue_enqueue(tq->queue, (void*)tqe, 0 /* do not copy */, wakeup); 
    MUTEX_UNLOCK(&tq->mtx);
    
    ret = (int)wakeup;
    
    /* wake up timer thread */
    if (pthread_cond_signal(&tq->cond) != 0) {
	pp_log("%s(): pthread_cond_signal() failed\n", ___F);
        ret = PP_ERR;
    }
    
    return ret;
}

int pp_timerq_rem(pp_timerq_t *tq, unsigned int id) {
    int ret = PP_ERR;
    pp_queue_entry_t *qe;
    
    assert(tq);
    
    MUTEX_LOCK(&tq->mtx);
    if((qe = pp_queue_get(tq->queue, id)) != NULL) {
        ret = PP_SUC;
        if (qe->len > 0) {
	    free(qe->data);
	}
	free(qe);
    }
    MUTEX_UNLOCK(&tq->mtx);
    
    return ret;
}

int pp_timerq_empty(pp_timerq_t *tq) {
    assert(tq);
    return(pp_queue_first(tq->queue) ? 0 : 1);
}

/****************************** static functions ******************************/

static void* timerq_thread(void *arg) {
    pp_timerq_t *tq;
    pp_queue_entry_t *qe;
    timerq_entry_t *tqe = NULL;
    static struct timespec infinite = {LONG_MAX, LONG_MAX};
    struct timespec *timeout;
    struct timeval now;
    
    assert(arg);
    
    tq = (pp_timerq_t*)arg;
    
    while(tq->state != DESTROY) {
        if(tq->state == SLEEPING) {
            MUTEX_LOCK(&tq->mtx);
            pthread_cond_timedwait(&tq->cond, &tq->mtx, &infinite);
            MUTEX_UNLOCK(&tq->mtx);
        }

        gettimeofday(&now, NULL);
        MUTEX_LOCK(&tq->mtx);
        while((qe = pp_queue_first(tq->queue)) != NULL) {
            tqe = (timerq_entry_t*)qe->data;
            if(tqe->timeout.tv_sec > now.tv_sec ||
               tqe->timeout.tv_nsec > now.tv_usec * 1000)
                break; /* timeout of first entry is in future */
                
            /* timer timed out, dequeue */
            qe = pp_queue_dequeue(tq->queue); 
            assert(qe);
            tqe = (timerq_entry_t*)qe->data;

            /* run callback */
            assert(tqe->cb);
            tqe->cb(tqe->ctx);

            /* and free timer queue and queue data structures */
            if (qe->len > 0) {
                free(qe->data);
            }
            free(qe);
        }
        if(qe == NULL) /* no timer enqueued */
            timeout = &infinite;
        else
            timeout = &tqe->timeout;
        pthread_cond_timedwait(&tq->cond, &tq->mtx, timeout);
        MUTEX_UNLOCK(&tq->mtx);
    }
    
    // destroy timer queue
    timerq_destroy(tq);
    
    return NULL;
}

static void timerq_destroy(pp_timerq_t *tq) {
    assert(tq);
    
    pp_queue_free(tq->queue);
    free(tq);
}

static void update_basetime(pp_timerq_t *tq) {
    struct timeval now;
    
    assert(tq);
    
    /* do not change base time while entries are enqueued */
    MUTEX_LOCK(&tq->mtx);
    if(pp_queue_first(tq->queue))
        return;
    
    gettimeofday(&now, NULL);
    tq->basetime = now.tv_sec;
    MUTEX_UNLOCK(&tq->mtx);
}

#endif /* PP_FEAT_DEVICE */
