/* crypto/bio/bss_conn.c */
/* Copyright (C) 1995-1998 Eric Young (eay@cryptsoft.com)
 * All rights reserved.
 *
 * This package is an SSL implementation written
 * by Eric Young (eay@cryptsoft.com).
 * The implementation was written so as to conform with Netscapes SSL.
 * 
 * This library is free for commercial and non-commercial use as long as
 * the following conditions are aheared to.  The following conditions
 * apply to all code found in this distribution, be it the RC4, RSA,
 * lhash, DES, etc., code; not just the SSL code.  The SSL documentation
 * included with this distribution is covered by the same copyright terms
 * except that the holder is Tim Hudson (tjh@cryptsoft.com).
 * 
 * Copyright remains Eric Young's, and as such any Copyright notices in
 * the code are not to be removed.
 * If this package is used in a product, Eric Young should be given attribution
 * as the author of the parts of the library used.
 * This can be in the form of a textual message at program startup or
 * in documentation (online or textual) provided with the package.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *    "This product includes cryptographic software written by
 *     Eric Young (eay@cryptsoft.com)"
 *    The word 'cryptographic' can be left out if the rouines from the library
 *    being used are not cryptographic related :-).
 * 4. If you include any Windows specific code (or a derivative thereof) from 
 *    the apps directory (application code) you must include an acknowledgement:
 *    "This product includes software written by Tim Hudson (tjh@cryptsoft.com)"
 * 
 * THIS SOFTWARE IS PROVIDED BY ERIC YOUNG ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * 
 * The licence and distribution terms for any publically available version or
 * derivative of this code cannot be changed.  i.e. this code cannot simply be
 * copied and put under another distribution licence
 * [including the GNU Public Licence.]
 */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <net/if.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <openssl/ssl.h>
#include <pp/base.h>
#include <pp/bio.h>
#include <pp/bio_internal.h>

typedef struct {
    int state;
    char * param_hostname;
    char * param_port;
    int nbio;
    int addr_family;
    long io_timeout;
    union {
	struct sockaddr_in ipv4;
	struct sockaddr_in6 ipv6;
    } them;
} pp_connect_ctx_t;

static int connect_new(BIO * b);
static int connect_free(BIO * b);
static int connect_state(BIO * b, pp_connect_ctx_t * ctx);
static int connect_read(BIO * b, char * out, int outl);
static int connect_write(BIO * b, const char * in, int inl);
static long connect_ctrl(BIO * b, int cmd, long num, void * ptr);
static int connect_puts(BIO * b, const char * str);
static void connect_close_socket(BIO * b);

static BIO_METHOD methods_pp_connect = {
    BIO_TYPE_PP_CONNECT,
    "pp socket connect",
    connect_write,
    connect_read,
    connect_puts,
    NULL, /* connect_gets, */
    connect_ctrl,
    connect_new,
    connect_free,
    NULL
};

BIO_METHOD * BIO_pp_s_connect(void)
{
    return &methods_pp_connect;
}

static int connect_new(BIO * b)
{
    pp_connect_ctx_t * ctx;

    b->init = 0;
    b->num = -1;
    b->flags = 0;

    if (!(ctx = (pp_connect_ctx_t *)OPENSSL_malloc(sizeof(*ctx)))) {
	return 0;
    }

    memset(ctx, 0, sizeof(*ctx));
    ctx->state = BIO_CONN_S_BEFORE;

    b->ptr = (char *)ctx;
    b->init = 1;
    b->shutdown = 1;

    return 1;
}

static int connect_free(BIO * b)
{
    pp_connect_ctx_t * ctx;

    if (b == NULL || !b->init) return 0;

    if (b->shutdown) {
	connect_close_socket(b);
	ctx = (pp_connect_ctx_t *)b->ptr;
	if (ctx->param_hostname != NULL) OPENSSL_free(ctx->param_hostname);
	if (ctx->param_port != NULL) OPENSSL_free(ctx->param_port);
	OPENSSL_free(ctx);
	b->ptr = NULL;
	b->init = 0;
	b->flags = 0;
    }

    return 1;
}

static int connect_state(BIO * b, pp_connect_ctx_t * ctx)
{
    const char * fn = ___F;
    int i, n;
    unsigned short port;
    char port_str[6];
    char host[256];

    for (;;) {
	switch (ctx->state) {
	  case BIO_CONN_S_BEFORE:
	      if (ctx->param_hostname == NULL) {
		  pp_log("%s(): No hostname specified.\n", fn);
		  goto err;
	      } else if ((n = sscanf(ctx->param_hostname, "[%255[^]]]:%5[0-9]", host, port_str)) == 2) {
		  if (ctx->addr_family != AF_INET6) return -1;
	      } else if ((n = sscanf(ctx->param_hostname, "%255[^:]:%5[0-9]", host, port_str)) != 2 &&
			 (n = sscanf(ctx->param_hostname, "%255[^\n]", host) != 1)) {
		  goto err;
	      }

	      if (n == 2) {
		  if (ctx->param_hostname != NULL)
		      OPENSSL_free(ctx->param_hostname);
		  ctx->param_hostname = BUF_strdup(host);
		  if (ctx->param_port != NULL)
		      OPENSSL_free(ctx->param_port);
		  ctx->param_port = BUF_strdup(port_str);
	      }

	      memset(&ctx->them, 0, sizeof(ctx->them));

	      if (ctx->param_port == NULL || sscanf(ctx->param_port, "%hu", &port) != 1) {
		  pp_log("%s(): No or invalid port specified.\n", fn);
		  goto err;
	      }

	      if (ctx->addr_family == AF_INET6) {
		  ctx->them.ipv6.sin6_family = AF_INET6;
		  ctx->them.ipv6.sin6_port = htons(port);
		  /* HACK ALERT */
		  ctx->them.ipv6.sin6_scope_id = if_nametoindex("eth0");
		  if (!_pp_bio_get_host_ip(ctx->param_hostname,
					   (char *)&ctx->them.ipv6.sin6_addr,
					   ctx->addr_family))
		      goto err;
	      } else {
		  ctx->them.ipv4.sin_family = AF_INET;
		  ctx->them.ipv4.sin_port = htons(port);
		  if (!_pp_bio_get_host_ip(ctx->param_hostname,
					   (char *)&ctx->them.ipv4.sin_addr,
					   ctx->addr_family))
		      goto err;
	      }

	      if ((b->num = socket(ctx->addr_family, SOCK_STREAM, IPPROTO_TCP)) == -1) {
		  pp_log_err("%s(): Couldn't create connect socket", fn);
		  goto err;
	      }

	      if (ctx->nbio && !BIO_socket_nbio(b->num, 1)) {
		  pp_log("%s(): Couldn't set non-blocking mode on connect socket.\n", fn);
		  goto err;
	      }

	      i = 1;
	      if (setsockopt(b->num, SOL_SOCKET, SO_KEEPALIVE, &i, sizeof(i)) < 0) {
		  pp_log_err("%s(): Couldn't set keep-alive mode on connect socket", fn);
		  goto err;
	      }

	      /* set I/O timeout if requested */
	      if (ctx->io_timeout > 0) {
		  struct timeval tv;

		  tv.tv_sec = ctx->io_timeout;
		  tv.tv_usec = 0;

		  /* set send timeout */
		  if (setsockopt(b->num, SOL_SOCKET, SO_SNDTIMEO, (void *)&tv, sizeof(tv)) < 0) {
		      pp_log_err("%s(): Couldn't set send timeout", fn);
		  }
		  /* set receive timeout */
		  if (setsockopt(b->num, SOL_SOCKET, SO_RCVTIMEO, (void *)&tv, sizeof(tv)) < 0) {
		      pp_log_err("%s(): Couldn't set receive timeout", fn);
		  }
	      }

	      ctx->state = BIO_CONN_S_CONNECT;;
	      break;
	  case BIO_CONN_S_CONNECT:
	      BIO_clear_retry_flags(b);
	      b->retry_reason = 0;
	      if ((i = connect(b->num, (struct sockaddr *)&ctx->them, sizeof(ctx->them))) < 0) {
		  if (!BIO_sock_should_retry(i)) {
		      pp_log_err("%s(): Couldn't connect", fn);
		      goto err;
		  }
		  BIO_set_retry_special(b);
		  ctx->state = BIO_CONN_S_BLOCKED_CONNECT;
		  b->retry_reason = BIO_RR_CONNECT;
	      } else {
		  ctx->state = BIO_CONN_S_OK;
	      }
	      break;
	  case BIO_CONN_S_BLOCKED_CONNECT:
	      if (BIO_sock_error(b->num)) {
		  BIO_clear_retry_flags(b);
		  pp_log("%s(): Couldn't connect.\n", fn);
		  goto err;
	      }
	      ctx->state = BIO_CONN_S_OK;
	      break;
	  case BIO_CONN_S_OK:
	      return 1;
	  default:
	      goto err;
	}
    }
 err:
    if (b->num != -1) close(b->num);
    b->num = -1;
    return -1;
}

static int connect_read(BIO * b, char * out, int outl)
{
    pp_connect_ctx_t * ctx;
    int ret = 0;

    if (b == NULL || !b->init) return -1;

    ctx = (pp_connect_ctx_t *)b->ptr;
    if (ctx->state != BIO_CONN_S_OK) {
	ret = connect_state(b, ctx);
	if (ret <= 0) return ret;
    }

    if (out != NULL) {
	ret = read(b->num, out, outl);
	BIO_clear_retry_flags(b);
	if (ret <= 0) {
	    if (BIO_sock_should_retry(ret))
		BIO_set_retry_read(b);
	}
    }

    return ret;
}

static int connect_write(BIO * b, const char * in, int inl)
{
    pp_connect_ctx_t * ctx;
    int ret;

    ctx = (pp_connect_ctx_t *)b->ptr;
    if (ctx->state != BIO_CONN_S_OK) {
	ret = connect_state(b, ctx);
	if (ret <= 0) return ret;
    }

    ret = write(b->num, in, inl);
    BIO_clear_retry_flags(b);
    if (ret <= 0) {
	if (BIO_sock_should_retry(ret))
	    BIO_set_retry_write(b);
    }
    return ret;
}

static long connect_ctrl(BIO * b, int cmd, long num, void * ptr)
{
    pp_connect_ctx_t * ctx;
    BIO *dbio;
    const char **pptr;
    long ret=1;

    if (b == NULL || !b->init) return -1;

    ctx = (pp_connect_ctx_t *)b->ptr;

    switch (cmd) {
      case BIO_CTRL_RESET:
	  ret = 0;
	  ctx->state = BIO_CONN_S_BEFORE;
	  connect_close_socket(b);
	  b->flags = 0;
	  break;
      case BIO_C_DO_STATE_MACHINE:
	  /* use this one to start the connection */
	  if (ctx->state != BIO_CONN_S_OK)
	      ret = (long)connect_state(b, ctx);
	  else
	      ret = 1;
	  break;
      case BIO_C_GET_CONNECT:
	  if (ptr != NULL) {
	      pptr = (const char **)ptr;
	      if (num == 0) {
		  *pptr = ctx->param_hostname;
	      } else if (num == 1) {
		  *pptr = ctx->param_port;
	      }
	      ret = 1;
	  }
	  break;
      case BIO_C_SET_CONNECT:
	  if (ptr != NULL) {
	      if (num == 0) {
		  if (ctx->param_hostname != NULL)
		      OPENSSL_free(ctx->param_hostname);
		  ctx->param_hostname = BUF_strdup(ptr);
	      } else if (num == 1) {
		  if (ctx->param_port != NULL)
		      OPENSSL_free(ctx->param_port);
		  ctx->param_port = BUF_strdup(ptr);
	      } else if (num == 4) {
		  ctx->addr_family = (int)ptr;
	      } else if (num == 5) {
		  ctx->io_timeout = (long)ptr;
	      } else {
		  return -1;
	      }
	  } else {
	      return -1;
	  }
	  break;
      case BIO_C_SET_NBIO:
	  ctx->nbio=(int)num;
	  break;
      case BIO_C_GET_FD:
	  if (ptr != NULL)
	      *((int *)ptr) = b->num;
	  ret = b->num;
	  break;
      case BIO_CTRL_GET_CLOSE:
	  ret = b->shutdown;
	  break;
      case BIO_CTRL_SET_CLOSE:
	  b->shutdown=(int)num;
	  break;
      case BIO_CTRL_PENDING:
      case BIO_CTRL_WPENDING:
	  ret=0;
	  break;
      case BIO_CTRL_FLUSH:
	  break;
      case BIO_CTRL_DUP:
	  dbio = (BIO *)ptr;
	  if (dbio) {
	      if (ctx->param_port)
		  BIO_set_conn_port(dbio, ctx->param_port);
	      if (ctx->param_hostname)
		  BIO_set_conn_hostname(dbio,ctx->param_hostname);
	      BIO_set_nbio(dbio, ctx->nbio);
	  }
	  break;
      default:
	  ret = 0;
	  break;
    }
    return ret;
}

static int connect_puts(BIO * b, const char * str)
{
    return connect_write(b, str, strlen(str));
}

static void connect_close_socket(BIO * b)
{
    if (b != NULL && b->init && b->num != -1) {
	close(b->num);
	b->num = -1;
    }
}

BIO * pp_bio_new_connect(char * host_port, int addr_family, long io_timeout)
{
    BIO * b = NULL;

    if ((b = BIO_new(BIO_pp_s_connect())) != NULL &&
	BIO_set_conn_hostname(b, host_port) == 1 &&
	BIO_ctrl(b, BIO_C_SET_CONNECT, 4, (void *)addr_family) == 1 &&
	BIO_ctrl(b, BIO_C_SET_CONNECT, 5, (void *)io_timeout) == 1) {

	return b;
    }

    if (b != NULL) BIO_free(b);
    return NULL;
}
