#include <errno.h>
#include <unistd.h>
#include <string.h> // for memcpy
#include <sys/poll.h>
#include <pp/bio.h>
#include <pp/features.h>

typedef struct {
    char buf[11];
    off_t buf_pos;
    size_t buf_size;
    int do_check;
    pp_bio_rfb_check_callback_t check_cb;
} rfb_check_ctx_t;

static int rfb_check_new(BIO * b);
static int rfb_check_free(BIO * b);
static int rfb_check_read(BIO * b, char * out, int outl);
static int rfb_check_write(BIO * b, const char * in, int inl);
static long rfb_check_ctrl(BIO * b, int cmd, long num, void * ptr);
static int rfb_check_gets(BIO * b, char * buf, int size);
static int rfb_check_puts(BIO * b, const char * str);

static BIO_METHOD methods_pp_rfb_check = {
    BIO_TYPE_PP_RFB_CHECK,
    "pp rfb check",
    rfb_check_write,
    rfb_check_read,
    rfb_check_puts,
    rfb_check_gets,
    rfb_check_ctrl,
    rfb_check_new,
    rfb_check_free,
    NULL
};

int
pp_rfb_check_cb(void * data, size_t data_size)
{
#ifdef PP_FEAT_DRIVE_REDIRECTION
    if (!memcmp(data, "e-RIC MSP P", data_size)) {
    	return BIO_PP_PROTO_MSP;
    }
#endif /* PP_FEAT_DRIVE_REDIRECTION */
    if (!memcmp(data, "e-RIC AUTH=", data_size)) {
    	return BIO_PP_PROTO_RFB;
    }
    return BIO_PP_PROTO_HTTPS;
}

BIO_METHOD *
BIO_pp_f_rfb_check(void)
{
    return (&methods_pp_rfb_check);
}

static int
rfb_check_new(BIO * b)
{
    rfb_check_ctx_t * ctx;

    b->init = 0;
    b->flags = 0;

    ctx = (rfb_check_ctx_t *)OPENSSL_malloc(sizeof(rfb_check_ctx_t));
    if (ctx == NULL) {
	return 0;
    }

    ctx->buf_pos = 0;
    ctx->buf_size = 0;
    ctx->do_check = 1;
    ctx->check_cb = NULL;

    b->ptr = (char *)ctx;
    b->init = 1;

    return 1;
}

static int
rfb_check_free(BIO * b)
{
    if (b == NULL || !b->init) {
	return 0;
    }

    if (b->ptr != NULL) {
	OPENSSL_free(b->ptr);
	b->ptr = NULL;
    }
    b->init = 0;
    b->flags = 0;

    return 1;
}
        
static int
rfb_check_read(BIO * b, char * out, int outl)
{
    rfb_check_ctx_t * ctx = (rfb_check_ctx_t *)b->ptr;
    int ret;

    if (b == NULL || !b->init) {
	return -1;
    }

    if (b->next_bio == NULL || out == NULL || outl <= 0 || ctx == NULL) {
	return 0;
    }

    BIO_clear_retry_flags(b);

    if (ctx->buf_size > 0) {
	if (ctx->buf_size >= (u_int)outl) {
	    memcpy(out, &ctx->buf[ctx->buf_pos], outl);
	    ctx->buf_pos  += outl;
	    ctx->buf_size -= outl;
	    return outl;
	} else {
	    memcpy(out, &ctx->buf[ctx->buf_pos], ctx->buf_size);
	    if ((ret = BIO_read(b->next_bio, &out[ctx->buf_size],
				outl - ctx->buf_size)) > 0) {
		ret += ctx->buf_size;
	    } else {
		ret = ctx->buf_size;
	    }
	    ctx->buf_pos += ctx->buf_size;
	    ctx->buf_size = 0;
	}
    } else {
	ret = BIO_read(b->next_bio, out, outl);
	BIO_copy_next_retry(b);
    }

    return ret;
}

static int
rfb_check_write(BIO * b, const char * in, int inl)
{
    int ret;

    if (b == NULL || !b->init) {
	return -1;
    }

    if (b->next_bio == NULL || in == NULL || inl <= 0) {
	return 0;
    }

    ret = BIO_write(b->next_bio, in, inl);
    BIO_copy_next_retry(b);
    return ret;
}

static long
rfb_check_ctrl(BIO * b, int cmd, long num, void * ptr)
{
    rfb_check_ctx_t * ctx;
    BIO * dbio;
    int ret = 0;

    if (b == NULL || !b->init) return -1;

    ctx = (rfb_check_ctx_t *)b->ptr;

    BIO_clear_retry_flags(b);

    switch (cmd) {
      case BIO_C_PP_CHECK_RFB:
	  if (ctx) {
	      while (ctx->do_check && ctx->buf_size < sizeof(ctx->buf)) {
		  ret = BIO_read(b->next_bio, &ctx->buf[ctx->buf_pos],
				 sizeof(ctx->buf)-ctx->buf_pos);
		  BIO_copy_next_retry(b);

		  if (ret == 0) {
		      return 0;
		  }

		  if (ret == -1) {
		      if (errno == EINTR) continue;
		      return 0;
		  }

		  ctx->buf_pos  += ret;
		  ctx->buf_size += ret;
	      }

	      if (ctx->check_cb) {
		  ret = ctx->check_cb(ctx->buf, sizeof(ctx->buf));
	      } else {
		  ret = 0;
	      }

	      ctx->buf_pos = 0;
	      ctx->do_check = 0;
	  }
	  break;
      case BIO_C_PP_SET_CHECK_CALLBACK:
	  ctx->check_cb = (pp_bio_rfb_check_callback_t)ptr;
	  ret = 1;
	  break;
      case BIO_CTRL_DUP:
	  dbio = (BIO *)ptr;
	  if (dbio) {
	      rfb_check_ctx_t * dbio_ctx = (rfb_check_ctx_t *)dbio->ptr;
	      dbio_ctx->check_cb = ctx->check_cb;
	  }
	  ret = 1;
	  break;
      default:
	  if (b->next_bio == NULL) {
	      ret = 0;
	  } else {
	      ret = BIO_ctrl(b->next_bio, cmd, num, ptr);
	      BIO_copy_next_retry(b);
	  }
    }

    return ret;
}

static int
rfb_check_gets(BIO * b, char * buf, int size)
{
    int ret;

    if (b == NULL || !b->init) {
	return -1;
    }

    if (b->next_bio == NULL) {
	return 0;
    }

    ret = BIO_gets(b->next_bio, buf, size);
    BIO_copy_next_retry(b);

    return ret;
}

static int
rfb_check_puts(BIO * b, const char * str)
{
    int ret;

    if (b == NULL || !b->init) {
	return -1;
    }

    if (b->next_bio == NULL) {
	return 0;
    }

    ret = BIO_puts(b->next_bio, str);
    BIO_copy_next_retry(b);

    return ret;
}

BIO *
pp_bio_new_rfb_check(pp_bio_rfb_check_callback_t check_cb)
{
    BIO * b;

    if ((b = BIO_new(BIO_pp_f_rfb_check())) != NULL &&
	BIO_ctrl(b, BIO_C_PP_SET_CHECK_CALLBACK, 0, check_cb)) {

	return b;
    }

    if (b != NULL) BIO_free(b);
    return NULL;
}
