/*
 * operation to dump a configuration description
 * (mainly intended for debugging purposes)
 *
 * tbr@peppercon.de
 * (c) 2004 Peppercon AG
 */

#include <stdio.h>
#include <stdlib.h>
#include <pp/cd.h>

static char* spaces(int no);

static int
print_for_cd(pp_cd_as_cd_t* cd,  pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    printf("%sconfig description ->\n", spaces(in));
    pp_cd_op_cd(cd, op, (void*)(in+3));
    return 0;
}
    
static int
print_for_section(pp_cd_as_section_t* s, pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    const char* n = NULL;
    switch (s->base.type & AS_SECTION) {
      case AS_SECTION_TYPES: n = "types"; break;
      case AS_SECTION_CONFIGS: n = "configs"; break;
      case AS_SECTION_VALUES:n = "values"; break;
      default: assert(0);
    }
    printf("%s%s section ->\n", spaces(in), n);
    pp_cd_op_list(s->decls, op, (void*)(in+3));
    return 0;
}

static int
print_for_type(pp_cd_as_type_t* type, pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    const char* t = NULL;
    switch (type->base.type & AS_TYPE) {
      case AS_TYPE_BOOL: t = "boolean"; break;
      case AS_TYPE_FKT: t = "function-declaration"; break;
      default: assert(0);
    }
    printf("%s%s: %s\n", spaces(in), type->name, t);
    if (type->properties)
	pp_cd_op_list(type->properties, op, (void*)(in+3));
    return 0;
}

static int
print_for_type_alias(pp_cd_as_type_alias_t* a, pp_cd_as_op_t* op,
			 void* ctx) {
    long in = (long)ctx;
    printf("%s%s: alias -> %s\n", spaces(in), a->base.name, a->ref_type_name);
    if (a->base.properties)
	pp_cd_op_list(a->base.properties, op, (void*)(in+3));
    return 0;
}

static int
print_for_type_struct(pp_cd_as_type_struct_t* s, pp_cd_as_op_t* op,
			  void* ctx) {
    long in = (long)ctx;
    printf("%s%s: struct ->\n", spaces(in), s->base.name);
    pp_cd_op_list(s->elements, op, (void*)(in+3));
    if (s->base.properties)
	pp_cd_op_list(s->base.properties, op, (void*)(in+6));
    return 0;
}
    
static int
print_for_type_enum(pp_cd_as_type_enum_t* e, pp_cd_as_op_t* op,
			void* ctx) {
    long in = (long)ctx;
    printf("%s%s: enum ->\n", spaces(in), e->base.name);
    pp_cd_op_list(e->elements, op, (void*)(in+3));
    if (e->base.properties)
	pp_cd_op_list(e->base.properties, op, (void*)(in+6));
    return 0;
}

static int
print_for_type_string_tmpl(pp_cd_as_type_string_tmpl_t* st,
			       pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    printf("%s%s: string<'%s', %d, %d>\n",
	   spaces(in), st->base.name, st->regexp, st->min, st->max);
    if (st->base.properties)
	pp_cd_op_list(st->base.properties, op, (void*)(in+3));
    return 0;
}

static int
print_for_type_int_tmpl(pp_cd_as_type_int_tmpl_t* it, pp_cd_as_op_t* op,
			    void* ctx) {
    long in = (long)ctx;
    printf("%s%s: int<%lld, %lld>\n",
	   spaces(in), it->base.name, it->min, it->max);
    if (it->base.properties)
	pp_cd_op_list(it->base.properties, op, (void*)(in+3));
    return 0;
}

static int
print_for_type_choice_tmpl(pp_cd_as_type_choice_tmpl_t* ct,
			       pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    printf("%s%s: choice<%s> ->\n", spaces(in), ct->base.name,
	   ct->tmpl_type_name);
    pp_cd_op_list(ct->elements, op, (void*)(in+3));
    if (ct->base.properties)
	pp_cd_op_list(ct->base.properties, op, (void*)(in+6));
    return 0;
}
    
static int
print_for_type_vector_tmpl(pp_cd_as_type_vector_tmpl_t* v,
			       pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    printf("%s%s: vector<%s, %s, %d>\n", spaces(in), v->base.name,
	   v->elem_type_name, pp_cd_fund_type_name(v->idx_type), v->bound);
    if (v->base.properties)
	pp_cd_op_list(v->base.properties, op, (void*)(in+3));
    return 0;
}
    
static int
print_for_type_vector_tmpl_spec(pp_cd_as_type_vector_tmpl_spec_t* v, 
				    pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    // TODO
    printf("%s%s: vector_spec<%s> -> TODO\n",
	   spaces(in), v->base.name, v->place_holder);
    //print_for_type_vector_tmpl(v->vector_type, op, (void*)(in+3));
    if (v->base.properties)
	pp_cd_op_list(v->base.properties, op, (void*)(in+6));
    return 0;
}

static int
print_for_type_vector_tmpl_inst(pp_cd_as_type_vector_tmpl_inst_t* v,
				    pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    // TODO
    printf("%s%s: vector_inst -> TODO\n",
	   spaces(in), v->base.name);
    if (v->base.properties)
	pp_cd_op_list(v->base.properties, op, (void*)(in+3));
    return 0;
}
    
static int
print_for_enumerator(pp_cd_as_enumerator_t* er, pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    printf("%s%s\n", spaces(in), er->value);
    if (er->properties)
	pp_cd_op_list(er->properties, op, (void*)(in+3));
    return 0; 
}
 
static int
print_for_property(pp_cd_as_property_t* p, pp_cd_as_op_t* op UNUSED, 
		   void* ctx) {
    long in = (long)ctx;
    unsigned long i;
    printf("%s%s: ", spaces(in), p->key);
    switch(p->base.type & AS_MISC) {
      case AS_PROPERTY_TAG:
	  for(i = 0; i < vector_size(p->value.tags); ++i)
	      printf(" %s", (char*)vector_get(p->value.tags, i));
	  break;
      case AS_PROPERTY_STRING:
	  printf("'%s'", p->value.text);
	  break;
      default: assert(0);
    }
    printf("\n");
    return 0; 
}

static int
print_for_fkt_decl(pp_cd_as_fkt_decl_t* fd, pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    pp_cd_as_type_t* fd_base;

    printf("%sname: %s\n", spaces(in), fd->base.name);
    if (fd->params) {
        printf("%sparams ->\n", spaces(in));
        pp_cd_op_list(fd->params, op, (void *)(in+3));
    }
    fd_base = (pp_cd_as_type_t*)fd;
    if (fd_base->properties) {
        printf("%sproperties ->\n", spaces(in));
        pp_cd_op_list(fd_base->properties, op, (void*)(in+3));
    }
    return 0;
}

static int
print_for_value_expr(pp_cd_as_value_expr_t* ve, pp_cd_as_op_t* op, void* ctx) {
    long in = (long)ctx;
    char* qkey = pp_cd_qualikey_to_string(NULL, ve->qkey);
    printf("%s%s: value-expression -> ", spaces(in), qkey);
    free(qkey);
    switch (ve->base.type & AS_EXPR) {
    case AS_EXPR_INT_LITERAL:
      printf("%d\n", ve->value.intval);
      break;
    case AS_EXPR_BOOL_LITERAL:
      printf("%s\n", ve->value.boolval ? "true" : "false");
      break;
    case AS_EXPR_ENUM_LITERAL:
    case AS_EXPR_STRING_LITERAL:
      printf("%s\n", ve->value.stringval);
      break;
    case AS_EXPR_CHOICE_LITERAL:
      printf("%s ->\n", ve->value.choiceval.choice);
      pp_cd_op_list(ve->value.choiceval.choicelist, op, (void*)(in+3));
      break;
    case AS_EXPR_FKT_DECL:
      printf("ftk_decl ->\n");
      print_for_fkt_decl( ve->value.fktval, op, (void*)(in+3) );
      break;
    case AS_EXPR_VALUE_LIST:
      printf("\n");
      pp_cd_op_list(ve->value.listval, op, (void*)(in+3));
      break;
    default: assert(0);
    }
    return 0;
}

pp_cd_as_op_t pp_cd_as_op_print = {
    print_for_cd,
    print_for_section,
    print_for_type,
    print_for_type_alias,
    print_for_type_struct,
    print_for_type_enum,
    print_for_type_string_tmpl,
    print_for_type_int_tmpl,
    print_for_type_choice_tmpl,
    print_for_type_vector_tmpl,
    print_for_type_vector_tmpl_spec,
    print_for_type_vector_tmpl_inst,
    print_for_enumerator,
    print_for_property,
    print_for_value_expr,
    NULL
};

static char* spaces(int no) {
    static char empty[] = "                                 ";
    int len = sizeof(empty) - 1;
    if(no > len) no = len;
    return &empty[len - no];
}
