/*
 * validation routines checking values against a type definition
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <sys/types.h>
#include <string.h>
#include <regex.h>
#include <pp/cd.h>
#include <pp/log.h>
#include <pp/intl.h>

#ifndef _WIN32
#define DEBUG
#ifdef DEBUG
# define D(fmt, args...) pp_log(fmt, ##args);
#else
# define D(fmt, args...)
#endif
#endif

const char* val_err_msg[] = {
    /* PP_CD_VALIDATION_ERROR_INTERNAL  = 0 */
    N_("Internal error, validation rule invalid!"),
    /* PP_CD_VALIDATION_ERROR_INVALID   = 1 */
    N_("The '%s' is invalid."),
    /* PP_CD_VALIDATION_ERROR_TOO_SHORT = 2 */
    N_("The '%s' is too short. Minimum length is %ld characters."),
    /* PP_CD_VALIDATION_ERROR_TOO_LONG  = 3 */
    N_("The '%s' is too long. Maximum length is %ld characters."),
    /* PP_CD_VALIDATION_ERROR_TOO_SMALL = 3 */
    N_("The '%s' is too small. Minimum value is %ld."),
    /* PP_CD_VALIDATION_ERROR_TOO_LARGE = 5 */
    N_("The '%s' is too large. Maximum value is %ld.")
};

static int validate_bool(pp_cd_as_type_t* as, pp_cd_as_op_t* op, void* ctx);
static int validate_enum(pp_cd_as_type_enum_t* as, pp_cd_as_op_t* op,
			 void* ctx);
static int validate_string(pp_cd_as_type_string_tmpl_t* as, pp_cd_as_op_t* op,
			   void* ctx);
static int validate_int(pp_cd_as_type_int_tmpl_t* as, pp_cd_as_op_t* op,
			void* ctx);
static int validate_choice(pp_cd_as_type_choice_tmpl_t* as, pp_cd_as_op_t* op,
			   void* ctx);
static int validate_vector(pp_cd_as_type_vector_tmpl_t* as, pp_cd_as_op_t* op,
			   void* ctx);
static int validate_int_intern(void* ctx, int min, int max);

static pp_cd_as_op_t pp_cd_as_op_validate = {
    NULL,
    NULL,
    validate_bool,
    NULL,
    NULL,
    validate_enum,
    validate_string,
    validate_int,
    validate_choice,
    validate_vector,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL,
    NULL
};

typedef struct {
    const char* value;
    pp_cd_validation_error_t err;
    int err_param;
} validation_ctx;

int pp_cd_type_validate_value(const pp_cd_as_type_t* type, const char* value,
			      pp_cd_validation_error_t* err, int* err_param) {
    /* FIXME: API misdesign!
     *        Hack to avoid casting "const" away! Never use this kludge without
     *        first discussing its use with SW engineering! Otherwise bad
     *        APIs will remain undetected or creep in uninvited.
     */
    union {
	const pp_cd_as_type_t* type_const;
	pp_cd_as_type_t* type;
    } type_kludge = { .type_const = type };
    validation_ctx ctx;
    int valid;
    
    ctx.value = value;
    
    valid = pp_cd_execute_op((pp_cd_as_t*)type_kludge.type, &pp_cd_as_op_validate, &ctx);
    *err = ctx.err;
    *err_param = ctx.err_param;
    
    return valid;
}

static int validate_bool(pp_cd_as_type_t* as UNUSED, pp_cd_as_op_t* op UNUSED,
			 void* ctx) {
    validation_ctx* c = (validation_ctx*)ctx;
    const char* v = c->value;
    if (v == NULL || *v == '\0' || !strcmp(v, "yes") || !strcmp(v, "no"))
	return 1;
    c->err = PP_CD_VALIDATION_ERROR_INVALID;
    return 0;
}

static int validate_enum(pp_cd_as_type_enum_t* as, pp_cd_as_op_t* op UNUSED,
			 void* ctx) {
    validation_ctx* c = (validation_ctx*)ctx;
    const char* v = c->value;
    char* en;
    size_t i, s = vector_size(as->elements);
    
    for (i = 0; i < s; ++i) {
	en = ((pp_cd_as_enumerator_t*)vector_get(as->elements, i))->value;
	if (!strcmp(v, en))
	    return 1;
    }

    D("validate_enum: %s : '%s'\n", as->base.name, v);
    
    c->err = PP_CD_VALIDATION_ERROR_INVALID;
    return 0;
}

static int validate_string(pp_cd_as_type_string_tmpl_t* as,
			   pp_cd_as_op_t* op UNUSED, void* ctx) {
    validation_ctx* c = (validation_ctx*)ctx;
    const char* val = c->value;
    ssize_t val_length;
    regex_t compiled_regex;
    int ret = 0;
    
    /* empty regex is always invalid */
    if (as->regexp[0] == '\0') {
	c->err = PP_CD_VALIDATION_ERROR_INVALID;
	goto error;
    }

    /* check emptiness */
    if (val == NULL) {
	c->err = PP_CD_VALIDATION_ERROR_INVALID;
	goto error;
    }
     
    val_length = strlen(val);

    /* check length constraint */
    if(!(as->min == 0 && as->max == 0)) {
        if (val_length < as->min) {
	    c->err = PP_CD_VALIDATION_ERROR_TOO_SHORT;
	    c->err_param = as->min;
	    goto error;
        } else if (val_length > as->max) {
	    c->err = PP_CD_VALIDATION_ERROR_TOO_LONG;
	    c->err_param = as->max;
	    goto error;
        }
    }

    /* compile the regex */
    if (regcomp(&compiled_regex, as->regexp, REG_EXTENDED|REG_NOSUB) != 0) {
	c->err = PP_CD_VALIDATION_ERROR_INTERNAL;
	goto error;
    }

    /* try to match the value against the regex */
    switch (regexec(&compiled_regex, val, 0, NULL, 0)) {
      case 0: /* match */
	  ret = 1;
	  break;
      case REG_NOMATCH:
	  c->err = PP_CD_VALIDATION_ERROR_INVALID;
	  break;
      default:
	  c->err = PP_CD_VALIDATION_ERROR_INTERNAL;
	  break;
    }

    /* free regex memory */
    regfree(&compiled_regex);

    if(!ret) goto error;
    
    return ret;
    
 error:
 
    D("validate_string: '%s'<%d,%d> : '%s'\n",
	   as->regexp, as->min, as->max, val);
	   
    return ret;
}

static int validate_int(pp_cd_as_type_int_tmpl_t* as, pp_cd_as_op_t* op UNUSED,
			void* ctx) {
    return validate_int_intern(ctx, as->min, as->max);
}

static int validate_choice(pp_cd_as_type_choice_tmpl_t* as, pp_cd_as_op_t* op,
			   void* ctx) {
    // check for the enum only
    D("validate_choice: -->\n");
    return pp_cd_execute_op((pp_cd_as_t*)as->tmpl_type, op, ctx);
}

static int validate_vector(pp_cd_as_type_vector_tmpl_t* as,
			   pp_cd_as_op_t* op UNUSED, void* ctx) {
    // check for the vector index
    D("validate_vector: -->\n");
    return validate_int_intern(ctx, 0, as->bound);
}

static int validate_int_intern(void* ctx, int min, int max) {
    validation_ctx* c = (validation_ctx*)ctx;
    char* endptr;
    long v;

    v = strtol(c->value, &endptr, 10);
    if (c->value == endptr) {
	c->err = PP_CD_VALIDATION_ERROR_INVALID;
	goto error;
    }
    if(!(max == 0 && min == 0)) {
        if (v < min) {
	    c->err = PP_CD_VALIDATION_ERROR_TOO_SMALL;
	    c->err_param = min;
	    goto error;
        }
        if (v > max) {
	    c->err = PP_CD_VALIDATION_ERROR_TOO_LARGE;
	    c->err_param = max;
	    goto error;
        }
    }
    return 1;
    
 error:
    D("validate_int_intern: <%d,%d> : '%s'\n", min, max, c->value);
    
    return 0;
}

const char* pp_cd_val_err_msg(int validation_error) {
    if(validation_error >= 0 && 
       validation_error < (int)(sizeof(val_err_msg) / sizeof(char*))) {
        return val_err_msg[validation_error];
    }
    abort(); // should not happen
    return NULL;
}

