/*
 * generel implementation to create, represent and browse
 * a concrete configuration description
 *
 * (c) 2004 Peppercon AG
 * tbr@peppercon.de
 */

#include <stdlib.h>
#include <string.h>
#include <pp/cd.h>
#include <pp/strstream.h>
#include "cd_intern.h"

static void* malloc0(pp_mallocator_t* a, size_t size);
static void init_as(pp_cd_as_t* as, pp_cd_as_id_t type, int pos,
		    int(*e)(struct pp_cd_as_s*, pp_cd_as_op_t*, void*),
		    void(*d)(pp_mallocator_t* a, struct pp_cd_as_s*));
static void init_type(pp_cd_as_type_t* t, pp_cd_as_id_t type, int pos,
		      int(*e)(struct pp_cd_as_s*, pp_cd_as_op_t*, void*),
		      void(*d)(pp_mallocator_t* a, struct pp_cd_as_s*),
		      char* name);
static void clean_type(pp_mallocator_t* a, pp_cd_as_type_t* t);
static void destroy_cd(pp_mallocator_t* a, pp_cd_as_t* cd);
static void destroy_section(pp_mallocator_t* a, pp_cd_as_t* cd);
static void destroy_type(pp_mallocator_t* a, pp_cd_as_t* cd);
static void destroy_alias(pp_mallocator_t* a, pp_cd_as_t* cd);
static void destroy_struct(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_enum(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_string_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_int_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_choice_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_vector_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_vector_tmpl_spec(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_vector_tmpl_inst(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_enumerator(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_property(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_value_expr(pp_mallocator_t* a, pp_cd_as_t* _o);
static void destroy_fkt_decl(pp_mallocator_t* a, pp_cd_as_t* _o);

static int op_cd(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_section(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_type(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_alias(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_struct(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_enum(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_string_tmpl(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_int_tmpl(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_choice_tmpl(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_vector_tmpl(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_vector_tmpl_spec(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_vector_tmpl_inst(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_enumerator(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_property(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_value_expr(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);
static int op_fkt_decl(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx);

static pp_cd_as_t* resolve_sym_n_sec_recursive(const pp_cd_as_cd_t* cd,
					       pp_cd_qualified_const_id_t id,
					       pp_strstream_t* k,
					       const pp_cd_resolve_kind_t rt,
					       pp_cd_as_id_t* root_section);
static inline pp_cd_as_t* lookup_symbol(const pp_cd_as_cd_t* cd,
					pp_cd_qualified_const_id_t id);
static pp_cd_as_property_t* type_prop_get(pp_cd_as_type_t* type,
					  const char* prop);


void
pp_cd_as_destroy(pp_mallocator_t* a, pp_cd_as_t* o) {
    if (o != NULL) {
	assert(o->destroy);
	o->destroy(a, o);
    }
}

pp_cd_as_cd_t*
pp_cd_as_create_cd(pp_mallocator_t* a, int pos) {
    pp_cd_as_cd_t* o = malloc0(a, sizeof(pp_cd_as_cd_t));
    init_as(&o->base, AS_CD, pos, op_cd, destroy_cd);
    o->alloc = a;
    o->types = pp_cd_as_create_section(a, 0, AS_SECTION_TYPES, NULL);
    o->configs = pp_cd_as_create_section(a, 0, AS_SECTION_CONFIGS, NULL);
    o->values = pp_cd_as_create_section(a, 0, AS_SECTION_VALUES, NULL);
    return o;
}

void
pp_cd_destroy(void* cd) {
    pp_cd_as_destroy(NULL, (pp_cd_as_t*)cd);
}

static void
destroy_cd(pp_mallocator_t* a UNUSED, pp_cd_as_t* _o) {
    pp_cd_as_cd_t* o = (pp_cd_as_cd_t*)_o;
    pp_cd_as_destroy(o->alloc, (pp_cd_as_t*)o->types);
    pp_cd_as_destroy(o->alloc, (pp_cd_as_t*)o->configs);
    pp_cd_as_destroy(o->alloc, (pp_cd_as_t*)o->values);
    pp_hash_delete(o->symbols);
    pp_free(o->alloc, o);
}

pp_cd_as_cd_t*
pp_cd_as_merge_cd_section(pp_cd_as_cd_t* cd, pp_cd_as_section_t* section) {
    switch (section->base.type & AS_SECTION) {
      case AS_SECTION_TYPES:
	  vector_addvec(cd->types->decls, section->decls, NULL);
	  break;
      case AS_SECTION_CONFIGS:
	  vector_addvec(cd->configs->decls, section->decls, NULL);
	  break;
      case AS_SECTION_VALUES:
	  vector_addvec(cd->values->decls, section->decls, NULL);
	  break;
      default: assert(section->base.type & AS_SECTION); assert(0);
    }
    return cd;
}

pp_cd_as_section_t*
pp_cd_as_create_section(pp_mallocator_t* a, int pos, pp_cd_as_id_t type,
			vector_t* decls) {
    pp_cd_as_section_t* o = malloc0(a, sizeof(pp_cd_as_section_t));
    assert(type & AS_SECTION);
    init_as(&o->base, type, pos, op_section, destroy_section);
    if (decls != NULL) {
	if (type != AS_SECTION_VALUES) { // no need to tag value section
	    pp_cd_op_list(decls, &pp_cd_as_op_tagsec, (void*)type);
	}
	o->decls = decls;
    } else {
	o->decls = vector_new_with_alloc(NULL, 100,
				  (vector_elem_del_func)pp_cd_as_destroy, a);
    }
    return o;
}

static void
destroy_section(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_section_t* o = (pp_cd_as_section_t*)_o;
    vector_delete(o->decls);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_type(pp_mallocator_t* a, int pos, char* name,
		     pp_cd_as_id_t type) {
    pp_cd_as_type_t* o = malloc0(a, sizeof(pp_cd_as_type_t));
    assert(type == AS_TYPE_BOOL || type == AS_TYPE_FKT);
    init_type(o, type, pos, op_type, destroy_type, name);
    return o;
}

static void
destroy_type(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_t* o = (pp_cd_as_type_t*)_o;
    clean_type(a, o);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_alias(pp_mallocator_t* a, int pos, char* name,
		      pp_cd_qualified_id_t alias_type_name) {
    pp_cd_as_type_alias_t* o = malloc0(a, sizeof(pp_cd_as_type_alias_t));
    assert(alias_type_name);
    init_type(&o->base, AS_TYPE_ALIAS, pos, op_alias, destroy_alias, name);
    o->ref_type_name = alias_type_name;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_alias(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_alias_t* o = (pp_cd_as_type_alias_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o->ref_type_name);
    pp_free(a, _o);
}

pp_cd_as_type_t*
pp_cd_as_create_struct(pp_mallocator_t* a, int pos, char* name,
		       pp_cd_type_list_t* elements) {
    pp_cd_as_type_struct_t* o = malloc0(a, sizeof(pp_cd_as_type_struct_t));
    assert(elements);
    init_type(&o->base, AS_TYPE_STRUCT, pos, op_struct, destroy_struct, name);
    o->elements = elements;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_struct(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_struct_t* o = (pp_cd_as_type_struct_t*)_o;
    clean_type(a, &o->base);
    vector_delete(o->elements);
    pp_free(a, o);
}
    
pp_cd_as_type_t*
pp_cd_as_create_enum(pp_mallocator_t* a, int pos, char* name,
		     pp_cd_enumerator_list_t* elements) {
    pp_cd_as_type_enum_t* o = malloc0(a, sizeof(pp_cd_as_type_enum_t));
    assert(elements);
    init_type(&o->base, AS_TYPE_ENUM, pos, op_enum, destroy_enum, name);
    o->elements = elements;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_enum(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_enum_t* o = (pp_cd_as_type_enum_t*)_o;
    clean_type(a, &o->base);
    vector_delete(o->elements);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_string_tmpl(pp_mallocator_t* a, int pos, char* name,
			    char* regexp, int min, int max) {
    pp_cd_as_type_string_tmpl_t* o
	= malloc0(a, sizeof(pp_cd_as_type_string_tmpl_t));
    init_type(&o->base, AS_TYPE_STRING_TMPL, pos, op_string_tmpl,
	      destroy_string_tmpl, name);
    o->regexp = regexp;
    o->min = min;
    o->max = max;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_string_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_string_tmpl_t* o = (pp_cd_as_type_string_tmpl_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o->regexp);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_int_tmpl(pp_mallocator_t* a, int pos, char* name, int min,
			 int max) {
    pp_cd_as_type_int_tmpl_t* o	= malloc0(a, sizeof(pp_cd_as_type_int_tmpl_t));
    init_type(&o->base, AS_TYPE_INT_TMPL, pos, op_int_tmpl, destroy_int_tmpl,
	      name);
    o->min = min;
    o->max = max;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_int_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_int_tmpl_t* o = (pp_cd_as_type_int_tmpl_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_choice_tmpl(pp_mallocator_t* a, int pos, char* name,
			    pp_cd_qualified_id_t tmpl_name,
			    vector_t* elements) {
    pp_cd_as_type_choice_tmpl_t* o
	= malloc0(a, sizeof(pp_cd_as_type_choice_tmpl_t));
    assert(elements);
    assert(tmpl_name);
    init_type(&o->base, AS_TYPE_CHOICE_TMPL, pos, op_choice_tmpl,
	      destroy_choice_tmpl, name);
    o->tmpl_type_name = tmpl_name;
    o->elements = elements;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_choice_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_choice_tmpl_t* o = (pp_cd_as_type_choice_tmpl_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o->tmpl_type_name);
    vector_delete(o->elements);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_vector_tmpl(pp_mallocator_t* a, int pos, char* name,
			    pp_cd_qualified_id_t elem_type_name,
			    pp_cd_as_id_t idx_type, int bound) {
    pp_cd_as_type_vector_tmpl_t* o
	= malloc0(a, sizeof(pp_cd_as_type_vector_tmpl_t));
    assert(elem_type_name);
    init_type(&o->base, AS_TYPE_VECTOR_TMPL, pos, op_vector_tmpl,
	      destroy_vector_tmpl, name);
    o->elem_type_name = elem_type_name;
    o->idx_type = idx_type;
    o->bound = bound;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_vector_tmpl(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_vector_tmpl_t* o = (pp_cd_as_type_vector_tmpl_t*)_o;
    clean_type(a, &o->base);
    pp_cd_as_destroy(a, (pp_cd_as_t*)o->size_type);
    pp_free(a, o->elem_type_name);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_vector_tmpl_spec(pp_mallocator_t* a, int pos, char* name,
				 char* place_holder,
				 pp_cd_as_type_t* vector_type) {
    pp_cd_as_type_vector_tmpl_spec_t* o
	= malloc0(a, sizeof(pp_cd_as_type_vector_tmpl_spec_t));
    assert(PP_CD_IS_TYPE(&vector_type->base, AS_TYPE_VECTOR_TMPL));
    init_type(&o->base, AS_TYPE_VECTOR_TMPL_SPEC, pos, op_vector_tmpl_spec,
	      destroy_vector_tmpl_spec, name);
    o->place_holder = place_holder;
    o->vector_type = (pp_cd_as_type_vector_tmpl_t*)vector_type;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_vector_tmpl_spec(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_vector_tmpl_spec_t* o =(pp_cd_as_type_vector_tmpl_spec_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o->place_holder);
    pp_cd_as_destroy(a, (pp_cd_as_t*)o->vector_type);
    pp_free(a, o);
}

pp_cd_as_type_t*
pp_cd_as_create_vector_tmpl_inst(pp_mallocator_t* a, int pos, char* name,
				 pp_cd_qualified_id_t elem_type,
				 pp_cd_qualified_id_t ref_type) {
    pp_cd_as_type_vector_tmpl_inst_t* o
	= malloc0(a, sizeof(pp_cd_as_type_vector_tmpl_inst_t));
    assert(elem_type);
    assert(ref_type);
    init_type(&o->base, AS_TYPE_VECTOR_TMPL_INST, pos, op_vector_tmpl_inst,
	      destroy_vector_tmpl_inst, name);
    o->elem_type_name = elem_type;
    o->ref_type_name = ref_type;
    return (pp_cd_as_type_t*)o;
}

static void
destroy_vector_tmpl_inst(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_type_vector_tmpl_inst_t* o =(pp_cd_as_type_vector_tmpl_inst_t*)_o;
    clean_type(a, &o->base);
    pp_free(a, o->elem_type_name);
    pp_free(a, o->ref_type_name);
    pp_cd_as_destroy(a, (pp_cd_as_t*)o->ref_type);
    pp_free(a, o);
}

pp_cd_as_enumerator_t*
pp_cd_as_create_enumerator(pp_mallocator_t* a, int pos, char* value) {
    pp_cd_as_enumerator_t* o = malloc0(a, sizeof(pp_cd_as_enumerator_t));
    assert(value);
    init_as(&o->base, AS_ENUMERATOR, pos, op_enumerator, destroy_enumerator);
    o->value = value;
    return o;
}

static void
destroy_enumerator(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_enumerator_t* o =(pp_cd_as_enumerator_t*)_o;
    pp_free(a, o->value);
    vector_delete(o->properties);
    pp_free(a, o);
}

pp_cd_as_property_t*
pp_cd_as_create_property(pp_mallocator_t* a, int pos, char* key,
			 pp_cd_as_id_t type,
			 pp_cd_property_value_t value) {
    pp_cd_as_property_t* o = malloc0(a, sizeof(pp_cd_as_property_t));
    assert(key);
    assert(type == AS_PROPERTY_TAG || type == AS_PROPERTY_STRING);
    init_as(&o->base, type, pos, op_property, destroy_property);
    o->key = key;
    o->value = value;
    return o;
}

static void
destroy_property(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_property_t* o =(pp_cd_as_property_t*)_o;
    switch (o->base.type & AS_MISC) {
      case AS_PROPERTY_TAG: vector_delete(o->value.tags); break;
      case AS_PROPERTY_STRING: pp_free(a, o->value.text); break;
      default: assert(0);
    };
    pp_free(a, o->key);
    pp_free(a, o);
}

pp_cd_as_value_expr_t*
pp_cd_as_create_value_expr(pp_mallocator_t* a, int pos,
			   pp_cd_quali_cfg_key_t* qkey,
			   pp_cd_as_id_t type,
			   pp_cd_literal_value_t value) {
    pp_cd_as_value_expr_t* o = malloc0(a, sizeof(pp_cd_as_value_expr_t));
    assert(type & AS_EXPR);
    init_as(&o->base, type, pos, op_value_expr, destroy_value_expr);
    o->qkey = qkey;
    o->value = value;
    return o;
}

static void
destroy_value_expr(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_value_expr_t* o = (pp_cd_as_value_expr_t*)_o;
    switch (o->base.type & AS_EXPR) {
      case AS_EXPR_BOOL_LITERAL:
      case AS_EXPR_INT_LITERAL:
	  break;
      case AS_EXPR_ENUM_LITERAL:
      case AS_EXPR_STRING_LITERAL:
	  pp_free(a, o->value.stringval);
	  break;
      case AS_EXPR_FKT_DECL:
	  pp_cd_as_destroy(a, (pp_cd_as_t*)o->value.fktval);
	  break;
      case AS_EXPR_VALUE_LIST:
	  vector_delete(o->value.listval);
	  break;
      case AS_EXPR_CHOICE_LITERAL:
	  pp_free(a, o->value.choiceval.choice);
	  vector_delete(o->value.choiceval.choicelist);
	  break;
      default: assert(0);
    };
    vector_delete(o->qkey);
    pp_free(a, o);
}

pp_cd_as_fkt_decl_t*
pp_cd_as_create_fkt_decl(pp_mallocator_t* a, int pos, char* name,
			 pp_cd_type_list_t* params,
			 pp_cd_proplist_t* properties) {
    pp_cd_as_fkt_decl_t* o = malloc0(a, sizeof(pp_cd_as_fkt_decl_t));
    assert(name);
    assert(params);
    init_type(&o->base, AS_EXPR_FKT_DECL, pos, op_fkt_decl, destroy_fkt_decl, name);
    o->params = params;
    o->base.properties = properties;
    return o;
}

static void
destroy_fkt_decl(pp_mallocator_t* a, pp_cd_as_t* _o) {
    pp_cd_as_fkt_decl_t* o =(pp_cd_as_fkt_decl_t*)_o;
    clean_type(a, &o->base);
    vector_delete(o->params);
    pp_free(a, o);
}

/*
 * allocates memory and blanks it with zeros
 */
static void* malloc0(pp_mallocator_t* a, size_t size) {
    void* m = pp_malloc(a, size);
    memset(m, 0, size);
    return m;
}

/*
 * some utils to minimize code
 */
static void clean_type(pp_mallocator_t* a, pp_cd_as_type_t* t) {
    pp_free(a, t->name);
    vector_delete(t->properties);
}

static void init_as(pp_cd_as_t* as, pp_cd_as_id_t type, int pos,
		    int(*e)(struct pp_cd_as_s*, pp_cd_as_op_t*, void*),
		    void(*d)(pp_mallocator_t* a, struct pp_cd_as_s*)) {
    as->type = type;
    as->pos = pos;
    as->execute = e;
    as->destroy = d;
}

static void init_type(pp_cd_as_type_t* t, pp_cd_as_id_t type, int pos,
		      int(*e)(struct pp_cd_as_s*, pp_cd_as_op_t*, void*),
		      void(*d)(pp_mallocator_t* a, struct pp_cd_as_s*),
		      char* name) {
    t->base.type = type;
    t->base.pos = pos;
    t->base.execute = e;
    t->base.destroy = d;
    t->name = name;
}


/*
 * operations need to call the method acording to their type
 */

#define OP_DISP(name, fkt, type)			             \
    static int name(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx) { \
	assert(op->fkt != NULL);                                     \
	if (op->fkt != NULL)                                         \
	    return op->fkt((type*)asn, op, ctx);		     \
	else                                                         \
	    return 0;						     \
    }

OP_DISP(op_cd, exec_for_cd, pp_cd_as_cd_t)
OP_DISP(op_section, exec_for_section, pp_cd_as_section_t)
OP_DISP(op_type, exec_for_type, pp_cd_as_type_t)
OP_DISP(op_alias, exec_for_type_alias, pp_cd_as_type_alias_t)
OP_DISP(op_struct, exec_for_type_struct, pp_cd_as_type_struct_t)
OP_DISP(op_enum, exec_for_type_enum, pp_cd_as_type_enum_t)
OP_DISP(op_string_tmpl, exec_for_type_string_tmpl, pp_cd_as_type_string_tmpl_t)
OP_DISP(op_int_tmpl, exec_for_type_int_tmpl, pp_cd_as_type_int_tmpl_t)
OP_DISP(op_choice_tmpl, exec_for_type_choice_tmpl, pp_cd_as_type_choice_tmpl_t)
OP_DISP(op_vector_tmpl, exec_for_type_vector_tmpl, pp_cd_as_type_vector_tmpl_t)
OP_DISP(op_vector_tmpl_spec, exec_for_type_vector_tmpl_spec,
	pp_cd_as_type_vector_tmpl_spec_t)
OP_DISP(op_vector_tmpl_inst, exec_for_type_vector_tmpl_inst,
	pp_cd_as_type_vector_tmpl_inst_t)
OP_DISP(op_enumerator, exec_for_enumerator, pp_cd_as_enumerator_t)
OP_DISP(op_property, exec_for_property, pp_cd_as_property_t)
OP_DISP(op_value_expr, exec_for_value_expr, pp_cd_as_value_expr_t)
OP_DISP(op_fkt_decl, exec_for_fkt_decl, pp_cd_as_fkt_decl_t)

/*
 * this is a set of default operations,
 * an operation implementation may use to loop over all
 * of the particular abstract syntax node's child nodes
 */
int
pp_cd_execute_op(pp_cd_as_t* asn, pp_cd_as_op_t* op, void* ctx) {
    return asn->execute(asn, op, ctx);
}

int
pp_cd_op_list(vector_t* v, pp_cd_as_op_t* op, void* ctx) {
    unsigned int i, ret = 0;
    for (i = 0; i < vector_size(v); ++i) {
		pp_cd_as_t* o = (pp_cd_as_t*)vector_get(v, i);
		ret += o->execute(o, op, ctx);
    }
    return ret;
}

int
pp_cd_op_cd(pp_cd_as_cd_t*cd, pp_cd_as_op_t* op, void* ctx) {
    int ret = 0;
    ret += cd->types->base.execute((pp_cd_as_t*)cd->types, op, ctx);
    ret += cd->configs->base.execute((pp_cd_as_t*)cd->configs, op, ctx);
    if(ret == 0 && cd->values)
        ret += cd->values->base.execute((pp_cd_as_t*)cd->values, op, ctx);
    return ret;
}

/* 
 * utils and non AS type functions
 */
pp_cd_cfg_key_t* 
pp_cd_cfg_key_create(pp_mallocator_t* a, pp_cd_as_id_t type,
		     char* key, pp_cd_cfg_key_idx_t idx) {
    pp_cd_cfg_key_t* ck = malloc0(a, sizeof(pp_cd_cfg_key_t));
    ck->type = type;
    ck->key = key;
    ck->idx = idx;
    return ck;
}

void 
pp_cd_cfg_key_destroy(pp_mallocator_t* a, pp_cd_cfg_key_t* ck) {
    if (ck != NULL) {
	if (PP_CD_IS_FUND_TYPE(&ck->idx, AS_FUND_TYPE_STRING)) {
	    pp_free(a, ck->idx.value.stridx);
	}
	pp_free(a, ck->key);
	pp_free(a, ck);
    }
}

pp_cd_qualified_id_t
pp_cd_qualified_id_create(pp_mallocator_t* a, pp_cd_qualified_const_id_t scope,
			  const char* id) {
    pp_cd_qualified_id_t s;
    int slen;
    int ilen;
    if (id == NULL) return strdup(scope ? scope : "");
    	slen = scope ? strlen(scope) : 0;
    ilen = slen + strlen(id);
    if (slen > 0) {
		s = pp_malloc(a, ilen + 2);
		strcpy(s, scope);
		strcat(s, PP_CD_COMP_DELI_STR);
    } else {
		s = pp_malloc(a, ilen + 1);
		s[0] = '\0';
    }
    strcat(s, id);
    return s;
}

pp_cd_qualified_id_t
pp_cd_qualified_id_add(pp_mallocator_t* a, pp_cd_qualified_id_t scope,
		       const char*id) {
    pp_cd_qualified_id_t s;
    int slen;
    int ilen;
    /* don't add a thing, if not valid or empty */
    if (id == NULL || id[0] == '\0') return scope;
    	slen = strlen(scope);
    	ilen = slen + strlen(id);
    if (slen > 0) {
		s = pp_realloc(a, scope, ilen + 2);
		strcat(s, PP_CD_COMP_DELI_STR);
    } else {
		s = pp_realloc(a, scope, ilen + 1);
    }
    strcat(s, id);
    return s;
}

unsigned int
pp_cd_qualified_id_count_comps(const char*id) {
    unsigned n;
    const char* p = id;
    if (strlen(id) == 0) return 0;
    n = 1;
    while(NULL != (p = strchr(p, PP_CD_COMP_DELI))) {
		++n; ++p;
    }
    return n;
}

/* copy scope and add cfg key */
pp_cd_qualified_id_t
pp_cd_qualified_id_create_from_cfg_key(pp_mallocator_t* a, 
                                       pp_cd_qualified_id_t scope,
                                       const pp_cd_cfg_key_t *cfg_key) {
    return pp_cd_qualified_id_add_from_cfg_key(a, scope ? strdup(scope) : NULL,
                                               cfg_key);
}

pp_cd_qualified_id_t
pp_cd_qualified_id_add_from_cfg_key(pp_mallocator_t* a, 
                                    pp_cd_qualified_id_t scope,
                                    const pp_cd_cfg_key_t *cfg_key) {
    pp_strstream_t buf;
    pp_cd_qualified_id_t qid;
    
    assert(scope);
    
    pp_strstream_init(&buf);
    if(*scope) {
        /* scope is not empty, we start with delimiter */
        pp_strappendchr(&buf, PP_CD_COMP_DELI);
    }
    if(cfg_key->key) {
        /* regular key */
        pp_strappend(&buf, cfg_key->key);
    } else if(PP_CD_IS_CFG_KEY(cfg_key, AS_CFG_KEY_VECTOR)) {
        /* we got a plain index key */
        pp_strappend(&buf, PP_CD_VECT_ELEMS_COMP_STR);
    }
    
    qid = pp_realloc(a, scope, strlen(scope) + pp_strstream_pos(&buf) + 1);
    strcat(qid, buf.buf);
    pp_strstream_free(&buf);
    
    return qid;
}

/* ignores all the indexes */
char* pp_cd_qualikey_to_qualiid(pp_cd_quali_cfg_key_t* qkey) {
  int i;
  int l = vector_size(qkey);
  char* qid ;

  if (l == 0) return NULL;
  qid = pp_cd_qualified_id_create(pp_mallocator_heap(), NULL, 
			      ((pp_cd_cfg_key_t*)vector_get(qkey, 0))->key);
  for (i = 1; i < l; ++i) {
    qid = pp_cd_qualified_id_add(pp_mallocator_heap(), qid,
				 ((pp_cd_cfg_key_t*)vector_get(qkey, i))->key);
  }
  return qid;
}

char* pp_cd_qualikey_to_string(const char* scope,
			       pp_cd_quali_cfg_key_t* qkey) {
    int i;
    int l = vector_size(qkey);
    pp_strstream_t os;
    pp_cd_cfg_key_t* k;

    if (l == 0) return NULL;

    pp_strstream_init(&os);
  
    if (scope != NULL)
	pp_strappend(&os, scope);
  
    for (i = 0; i < l; ++i) {
	k = (pp_cd_cfg_key_t*)vector_get(qkey, i);
	if (k->key) {
	    if (i > 0 || (scope != NULL && strlen(scope) > 0))
		pp_strappendchr(&os, PP_CD_COMP_DELI);
	    pp_strappend(&os, k->key);
	}
	if (PP_CD_IS_CFG_KEY(k, AS_CFG_KEY_VECTOR)) {
	    pp_strappendchr(&os, '[');
	    switch (k->idx.type & AS_FUND_TYPE) {
	      case AS_FUND_TYPE_INT:
		  pp_strappendf(&os, "%d", k->idx.value.intidx);
		  break;
	      case AS_FUND_TYPE_STRING:
		  pp_strappend(&os, k->idx.value.stridx);
		  break;
	      default:
		  assert(0);
	    }
	    pp_strappendchr(&os, ']');
	}
    }
    return pp_strstream_buf(&os);
}
					
void
pp_cd_qid_push_scope(const char* id, char** scope) {
    *scope = pp_cd_qualified_id_add(pp_mallocator_heap(), *scope, id);
}

void
pp_cd_qid_pop_scope(char* scope) {
    char* p;
    assert(*scope != '\0');
    if(NULL == (p = strrchr(scope, PP_CD_COMP_DELI)))
	p = scope;
    *p = '\0';
}

void 
pp_cd_qid_pop_scope_n(char* scope, int n) {
    int i;
    for (i = 0; i < n; ++i) {
        pp_cd_qid_pop_scope(scope);
    }
}

const char* 
pp_cd_fund_type_name(pp_cd_as_id_t id) {
    const char* n = 0;
    switch(id) {
      case AS_FUND_TYPE_BOOL:   n = "boolean"; break;
      case AS_FUND_TYPE_INT:    n = "int"; break;
      case AS_FUND_TYPE_STRING: n = "string"; break;
      default: assert(0);
    }
    return n;
}

/*
 * symbol lookup functions
 */

pp_cd_as_t*
pp_cd_resolve_sym_n_sec(const pp_cd_as_cd_t* cd,
			pp_cd_qualified_const_id_t symbol,
			const pp_cd_resolve_kind_t rt,
			pp_cd_as_id_t* root_section) {
    pp_strstream_t k = PP_STRSTREAM_INITIALIZER;
    pp_cd_as_t* s;
    
    s = resolve_sym_n_sec_recursive(cd, symbol, &k, rt, root_section);
    
    pp_strstream_free(&k);
    return s;
}

pp_cd_as_t*
pp_cd_resolve_symbol(const pp_cd_as_cd_t* cd,
		     pp_cd_qualified_const_id_t symbol,
		     const pp_cd_resolve_kind_t rt) {
    return pp_cd_resolve_sym_n_sec(cd, symbol, rt, NULL);
}

pp_cd_as_type_t*
pp_cd_resolve_type_sym_n_sec(const pp_cd_as_cd_t* cd,
			     pp_cd_qualified_const_id_t id,
			     const pp_cd_resolve_kind_t t,
			     pp_cd_as_id_t* root_section) {
    pp_cd_as_t* s;
    
    assert(cd);
    
    s = pp_cd_resolve_sym_n_sec(cd, id, t, root_section);
    if (s != NULL && !(s->type & AS_TYPE)) {
	s = NULL;
    }
    return (pp_cd_as_type_t*)s;
}

pp_cd_as_type_t*
pp_cd_resolve_type_symbol(const pp_cd_as_cd_t* cd,
			  pp_cd_qualified_const_id_t id,
			  const pp_cd_resolve_kind_t t) {
    return pp_cd_resolve_type_sym_n_sec(cd, id, t, NULL);
}

pp_cd_as_t*
pp_cd_resolve_value_symbol( const pp_cd_as_cd_t* cd,
		            pp_cd_qualified_const_id_t symbol,
			    const char *scope) {
    int len = strlen(symbol) + strlen(scope) + 8;
    char *id = (char*)malloc(len);
    pp_cd_as_t *s;

    strcpy(id, "_val_.");
    strcat(id, scope);
    strcat(id, PP_CD_COMP_DELI_STR);
    strcat(id, symbol);

    s = (pp_cd_as_t*)pp_hash_get_entry(cd->symbols, id);

    free(id);
    
    return s;
}

pp_cd_as_t*
pp_cd_lookup_symbol(const pp_cd_as_cd_t* cd,
		    pp_cd_qualified_const_id_t symbol,
		    const char *scope) {
    int len;
    char *id;
    pp_cd_as_t *s;

    if((scope == NULL) || (strlen(scope) == 0))
        return lookup_symbol(cd, symbol);
        
    len = strlen(symbol) + strlen(scope) + 2;
    id = (char*)malloc(len);
    strcpy(id, scope);
    strcat(id, PP_CD_COMP_DELI_STR);
    strcat(id, symbol);

    s = lookup_symbol(cd, id);

    free(id);
    
    return s;
}

static pp_cd_as_t* resolve_sym_n_sec_recursive(const pp_cd_as_cd_t* cd,
					       pp_cd_qualified_const_id_t id,
					       pp_strstream_t* k,
					       const pp_cd_resolve_kind_t rt,
					       pp_cd_as_id_t* root_section) {
    const char *p1 = id, *p2;
    pp_cd_as_t* s;

    if (root_section != NULL) *root_section = AS_SECTION;
    pp_strstream_reset(k);
    while (1) {
	if (NULL == (p2 = strchr(p1, PP_CD_COMP_DELI))) {
	    pp_strappend(k, p1);
	} else {
	    pp_strappendn(k, p1, p2 - p1);
	}
	
	if (NULL == (s = lookup_symbol(cd, pp_strstream_buf(k)))) {
	    goto bailout;
        }
	
	if (root_section != NULL && *root_section == AS_SECTION)
	    *root_section = s->type & AS_SECTION;
	
	if (PP_CD_IS_TYPE(s, AS_TYPE_ALIAS) &&
	    (p2 != NULL || rt & PP_CD_RESOLVE_ALIAS)) {
	    if (NULL == (s = resolve_sym_n_sec_recursive(cd,
			         ((pp_cd_as_type_alias_t*)s)->ref_type_name,
                                 k, rt, NULL))) {
		goto bailout;
            }
        } else if (PP_CD_IS_TYPE(s, AS_TYPE_VECTOR_TMPL) && p2 && *p2) {

            if(!strncmp(p2 + 1, PP_CD_VECT_ELEMS_COMP_STR,
                        sizeof(PP_CD_VECT_ELEMS_COMP_STR) - 1)) {
                /* we got a vector qid followed by the element type key */
    
                /* skip PP_CD_VECT_ELEMS_COMP_STR */
                p1 = p2 + sizeof(PP_CD_VECT_ELEMS_COMP_STR);
                p2 = strchr(p1, PP_CD_COMP_DELI);
                
                if(p2 != NULL || rt & PP_CD_RESOLVE_VECTOR) {
                    /* we want to resolve the element type */
                    if (NULL == (s = resolve_sym_n_sec_recursive(cd, 
                                         ((pp_cd_as_type_vector_tmpl_t*)s)->
                                             elem_type_name,
                                         k, rt, NULL))) {
                        goto bailout;
                    }
                }
            } else if(strncmp(p2 + 1, PP_CD_VECT_SIZE_COMP_STR,
                              sizeof(PP_CD_VECT_SIZE_COMP_STR) - 1) &&
                      strncmp(p2 + 1, PP_CD_VECT_DEF_COMP_STR,
                              sizeof(PP_CD_VECT_DEF_COMP_STR) - 1)) {
                /* we got a vector but no element type specified (defaults) */
                if (NULL == (s = resolve_sym_n_sec_recursive(cd, 
                                     ((pp_cd_as_type_vector_tmpl_t*)s)->
                                         elem_type_name,
                                     k, rt, NULL))) {
                    goto bailout;
                }
            }
	}
	
	if (p2 == NULL) {
	    break;
	} else {
	    pp_strappendchr(k, PP_CD_COMP_DELI);
	    p1 = p2 + 1;
	}
    }
 bailout:
    return s;
}

static pp_cd_as_t* lookup_symbol(const pp_cd_as_cd_t* cd,
				 pp_cd_qualified_const_id_t id) {
    return (pp_cd_as_t*)pp_hash_get_entry(cd->symbols, id);
}

static int preserve_for_value_expr_list( vector_t *p, vector_t *preserve ) {
    unsigned long vect_size = vector_size( p );
    unsigned long tmp_vect_size = vect_size;
    unsigned long idx = 0;
    pp_cd_as_value_expr_t* ve;

    while( idx < vect_size ) {
        ve = (pp_cd_as_value_expr_t*)vector_get(p, idx);

	switch (ve->base.type & AS_EXPR) {
            case AS_EXPR_VALUE_LIST:
		/* call recursive */
		preserve_for_value_expr_list( ve->value.listval, preserve );
		break;
	    case AS_EXPR_FKT_DECL:
		/* store in preserve and delete from parent */
		vector_add( preserve, ve );
		vector_remove_dont_delete( p, idx );
		break;
	}
	
	/*** if p[ idx ] was deleted from vector, we now look again at 
	     p[ idx ], otherwise index is increased ***/
	vect_size = vector_size( p );
	if( vect_size == tmp_vect_size )
	    idx++;
	else
	    tmp_vect_size = vect_size;
    }
    return 1;
}

void pp_cd_destroy_values(void *v) {
    pp_cd_as_cd_t *cd = (pp_cd_as_cd_t*)v;
    vector_t *preserved_fkt_dekls = vector_new_with_alloc(NULL, 100,
				      (vector_elem_del_func)pp_cd_as_destroy,
				       pp_mallocator_heap());

    preserve_for_value_expr_list( cd->values->decls, preserved_fkt_dekls );
    vector_delete( cd->values->decls );
    cd->values->decls = preserved_fkt_dekls;
}

/*
 * TODO: consider implementing properties in a map
 * would save us the linear search for a property
 */
int pp_cd_type_prop_has_flag(pp_cd_as_type_t* type, const char* prop,
			     const char* flag) {
    pp_cd_as_property_t* p;
    unsigned int i, s;
    const char* f;
    
    if (NULL == (p = type_prop_get(type, prop)) ||
	!PP_CD_IS_MISC(&p->base, AS_PROPERTY_TAG))
	return 0;

    s = vector_size(p->value.tags);
    for (i = 0; i < s; ++i) {
	f = (char*)vector_get(p->value.tags, i);
	if (!strcmp(f, flag)) return 1;
    }
    return 0;
}

char* pp_cd_type_prop_get(pp_cd_as_type_t* type, const char* prop) {
    pp_cd_as_property_t* p;
    if (NULL == (p = type_prop_get(type, prop)) ||
	!PP_CD_IS_MISC(&p->base, AS_PROPERTY_STRING))
	return NULL;
    return p->value.text;
}

static pp_cd_as_property_t* type_prop_get(pp_cd_as_type_t* type,
					  const char* prop) {
    unsigned int i;
    unsigned int s;
    pp_cd_as_property_t* p;

    if (type->properties != NULL) {
	s = vector_size(type->properties);
	for (i = 0; i < s; ++i) {
	    p = (pp_cd_as_property_t*)vector_get(type->properties, i);
	    if (!strcmp(p->key, prop))
		return p;
	}
    }
    return NULL;
}

static void lookup_prop_intern(const pp_cd_as_cd_t* cd, 
			       pp_cd_qualified_const_id_t qid,
                               const char* prop,
                               vector_t* prop_stack) {
    /* super duper get prop */
    const char *p1 = qid, *p2;
    pp_cd_as_type_t *type;
    pp_cd_as_property_t* p;
    pp_strstream_t k = PP_STRSTREAM_INITIALIZER;
    
    /* at first, lookup qid for prop */
    pp_strstream_reset(&k);
    while(1) {
	if (NULL == (p2 = strchr(p1, PP_CD_COMP_DELI))) {
	    pp_strappend(&k, p1);
	} else {
	    pp_strappendn(&k, p1, p2 - p1);
	}
        
	if (NULL == (type = pp_cd_resolve_type_symbol(cd,
                                         pp_strstream_buf(&k),
					 PP_CD_RESOLVE_SIMPLE)))
	    goto bailout;

        while(1) {
            if(NULL != (p = type_prop_get(type, prop))) {
                vector_add(prop_stack, p);
                break;
            }
            if(PP_CD_IS_BASE_TYPE(type, AS_TYPE_ALIAS)) {
                /* resolve aliases until the requested prop is found */
                type = ((pp_cd_as_type_alias_t*)type)->ref_type;
            } else if(PP_CD_IS_BASE_TYPE(type, AS_TYPE_VECTOR_TMPL)) {
                /* dive into vectors */
                type = ((pp_cd_as_type_vector_tmpl_t*)type)->elem_type;
            } else
                break;        
        }
    
	if (p2 == NULL) {
	    break;
	} else {
	    pp_strappendchr(&k, PP_CD_COMP_DELI);
	    p1 = p2 + 1;
	}
    }

 bailout:
    pp_strstream_free(&k);
    
    return;
}

char* pp_cd_lookup_prop(const pp_cd_as_cd_t* cd, 
			pp_cd_qualified_const_id_t qid,
                        const char* prop) {
    int last_elem;
    char *ret_prop = NULL;
    pp_cd_as_property_t* p;

    vector_t *prop_stack = vector_new_with_alloc(NULL, 5,NULL,
				                 pp_mallocator_heap());

    lookup_prop_intern(cd, qid, prop, prop_stack);
    
    last_elem = vector_size(prop_stack);
    
    if(last_elem < 1)
        goto bailout;
    
    p = (pp_cd_as_property_t*)vector_get(prop_stack, --last_elem);
    ret_prop = p->value.text;
    
 bailout:
    vector_delete(prop_stack);
    
    return ret_prop;
}

int pp_cd_prop_has_flag(const pp_cd_as_cd_t* cd, 
                        pp_cd_qualified_const_id_t qid,
                        const char* prop, const char* flag) {
    int last_elem, ret = 0;
    pp_cd_as_property_t* p;
    unsigned int s, i;
    const char* f;
 
    vector_t *prop_stack = vector_new_with_alloc(NULL, 5,NULL,
				                 pp_mallocator_heap());

    lookup_prop_intern(cd, qid, prop, prop_stack);
    
    last_elem = vector_size(prop_stack);
    
    if(last_elem < 1)
        goto bailout;
   
    p = (pp_cd_as_property_t*)vector_get(prop_stack, --last_elem);
    s = vector_size(p->value.tags);
    for (i = 0; i < s; ++i) {
	f = (char*)vector_get(p->value.tags, i);
	if (!strcmp(f, flag)) {
            ret = 1;
            goto bailout;
        }
    }
    
 bailout:
    vector_delete(prop_stack);
    
    return ret;
}

static int as_type_match_expr[11] = {
    /* AS_TYPE_BOOL */                  AS_EXPR_BOOL_LITERAL,
    /* AS_TYPE_FKT */                   AS_EXPR_FKT_DECL,
    /* AS_TYPE_ENUM */                  AS_EXPR_STRING_LITERAL,
    /* AS_TYPE_STRUCT */                0,
    /* AS_TYPE_ALIAS */                 0,
    /* AS_TYPE_STRING_TMPL */           AS_EXPR_STRING_LITERAL,
    /* AS_TYPE_INT_TMPL */              AS_EXPR_INT_LITERAL,
    /* AS_TYPE_CHOICE_TMPL */           AS_EXPR_CHOICE_LITERAL,
    /* AS_TYPE_VECTOR_TMPL */           AS_EXPR_VALUE_LIST,
    /* AS_TYPE_VECTOR_TMPL_SPEC */      0,
    /* AS_TYPE_VECTOR_TMPL_INST */      0,
};

int cd_as_match_expr_type(pp_cd_as_type_t *as_type) {
    int iter;

    assert(as_type->base.type && AS_TYPE);
    iter = ((as_type->base.type & AS_TYPE) >> 8) - 1;
    assert(iter >= 0 && iter < 11);
    return as_type_match_expr[iter];
}

#ifdef PP_BOARD_PEMX

static pp_cd_cfg_key_desc_t*
pp_cd_cfg_key_desc_create(pp_cd_as_cd_t *cd) {
    pp_cd_cfg_key_desc_t *ckd;
    ckd = malloc0(pp_mallocator_heap(), sizeof(pp_cd_cfg_key_desc_t));
    ckd->cd = cd;
    ckd->qids = vector_new(ckd->qids, 25, free);
    return ckd;
}

void pp_cd_cfg_key_desc_destroy(void *v) {
    pp_cd_cfg_key_desc_t *ckd = (pp_cd_cfg_key_desc_t*)v;
    vector_delete(ckd->qids);
    free(ckd);
}

/**
 * Generates a vector of pp_cd_cfg_key_desc_t (ckd_0, ckd_1, ..., common)
 * corresponding to a given vector of cds (src_cd_0, src_cd_1, ...).
 * The common-cd holds qids that are present AND semantically identic in each
 * of the source cds. The remaining qids are represented by ckd_0, ckd_1, ...
 **/
vector_t* pp_cd_union_cds(vector_t *src_cds) {
    u_int src_cds_sz = vector_size(src_cds), u;
    vector_t *ret;
    pp_cd_cfg_key_desc_t **ckds, *common;
    pp_cd_as_cd_t *cd;
    char *qid, *qid_tmp;
    pp_cd_as_t *as, *as2;
    pp_hash_t *cd_hash[src_cds_sz];
    void *v;
    int is_common;

    ckds = malloc(sizeof(pp_cd_cfg_key_desc_t*) * (src_cds_sz + 1));

    /* the magic union algo... */

    /* create config key descriptions and hashes for each cd */
    for(u = 0; u < src_cds_sz; ++u) {
        cd = vector_get(src_cds, u);
        ckds[u] = pp_cd_cfg_key_desc_create(cd);
        cd_hash[u] = pp_hash_create(64);
        as = (pp_cd_as_t*)cd;
        as->execute(as, &pp_cd_as_op_get_qid, (void*)cd_hash[u]);
    }
    /* create config key descriptions for common with last cd */
    ckds[src_cds_sz] = pp_cd_cfg_key_desc_create(cd);
    common = ckds[src_cds_sz];

    /* run over hash-table 0 and get all qids present AND semantically identic
     * in each hash_table, write that to common, delete from hash-tables */
    for(pp_hash_get_first_key_and_content(cd_hash[0], &qid, &v);
        qid != NULL; pp_hash_get_next_key_and_content(cd_hash[0], &qid, &v)) {
        /* as_op_type_compare (all except params) */
        as = (pp_cd_as_t*)v;
        for(u = 1, is_common = 1; is_common && u < src_cds_sz; ++u) {
            if(NULL == (as2 = pp_cd_resolve_symbol(ckds[u]->cd, qid,
                                                   PP_CD_RESOLVE_SIMPLE)) ||
               as->type != as2->type ||
               PP_SUC != as->execute(as, &pp_cd_as_op_type_compare, (void*)as2))
                is_common = 0;
        }
        if(is_common) {
            /* qid_tmp added to qid vector will be freed by vector delete */ 
            qid_tmp = strdup(qid);
            vector_add(common->qids, qid_tmp);
            for(u = 0; u < src_cds_sz; ++u)
                pp_hash_delete_entry(cd_hash[u], qid_tmp);
        }
    }
        
    /* remaining qids for each source cd to config key description */
    for(u = 0; u < src_cds_sz; ++u) {
        for(qid = (char*)pp_hash_get_first_key(cd_hash[u]);
            qid != NULL; qid = (char*)pp_hash_get_next_key(cd_hash[u]))
            vector_add(ckds[u]->qids, strdup(qid));
        pp_hash_delete(cd_hash[u]);
    }
    
    /* end of the magic union algo... */
        
    /* push config key descriptions to return vector and return */
    ret = vector_new(NULL, src_cds_sz + 1, pp_cd_cfg_key_desc_destroy);
    for(u = 0; u <= src_cds_sz; ++u)
        vector_add(ret, ckds[u]);
    
    free(ckds);
    
    return ret;
}

#endif // PP_BOARD_PEMX

